"""
Tests for Channel class.
"""

import pytest
from predictiondata.channel import Channel, DataType


def test_channel_with_symbols():
    """Test creating a channel with symbols."""
    channel = Channel(name="trades", symbols=["market-1/YES", "market-2/NO"])
    
    assert channel.get_data_type() == DataType.TRADES
    assert channel.get_identifiers() == ["market-1/YES", "market-2/NO"]
    assert not channel.uses_token_ids()


def test_channel_with_token_ids():
    """Test creating a channel with token IDs."""
    channel = Channel(name="onchain_fills", token_ids=["0x123", "0x456"])
    
    assert channel.get_data_type() == DataType.ONCHAIN_FILLS
    assert channel.get_identifiers() == ["0x123", "0x456"]
    assert channel.uses_token_ids()


def test_channel_books():
    """Test creating a books channel."""
    channel = Channel(name="books", symbols=["market-1/YES"])
    
    assert channel.get_data_type() == DataType.BOOKS


def test_channel_invalid_type():
    """Test that invalid data type raises error."""
    with pytest.raises(ValueError):
        Channel(name="invalid_type", symbols=["market-1/YES"])


def test_channel_no_identifiers():
    """Test that channel without symbols or token_ids raises error."""
    with pytest.raises(ValueError):
        Channel(name="trades")


def test_channel_priority_token_ids():
    """Test that token_ids takes priority over symbols."""
    channel = Channel(
        name="trades",
        symbols=["market-1/YES"],
        token_ids=["0x123"]
    )
    
    assert channel.get_identifiers() == ["0x123"]
    assert channel.uses_token_ids()


def test_channel_onchain_fills_underscore():
    """Test that onchain_fills with underscore works."""
    channel = Channel(name="onchain_fills", symbols=["market-1/YES"])
    assert channel.get_data_type() == DataType.ONCHAIN_FILLS


def test_channel_onchain_fills_slash():
    """Test that onchain/fills with slash works."""
    channel = Channel(name="onchain/fills", symbols=["market-1/YES"])
    assert channel.get_data_type() == DataType.ONCHAIN_FILLS

