"""
Tests for PredictionDataClient class.
"""

import pytest
from datetime import datetime
from predictiondata.client import PredictionDataClient
from predictiondata.channel import Channel, DataType


def test_client_initialization():
    """Test client initialization."""
    client = PredictionDataClient(api_key="test_key")
    
    assert client.api_key == "test_key"
    assert client.base_url == "http://datasets.predictiondata.dev"


def test_client_custom_base_url():
    """Test client with custom base URL."""
    client = PredictionDataClient(
        api_key="test_key",
        base_url="https://custom.example.com"
    )
    
    assert client.base_url == "https://custom.example.com"


def test_build_url_with_slug():
    """Test URL building with slug."""
    client = PredictionDataClient(api_key="test_key")
    
    url = client._build_url(
        exchange="polymarket",
        data_type=DataType.TRADES,
        identifier="market-1/YES",
        date=datetime(2024, 11, 15),
        use_slug=True
    )
    
    expected = "http://datasets.predictiondata.dev/polymarket/trades/market-1/YES/2024-11-15.csv.gz?slug=true&apikey=test_key"
    assert url == expected


def test_build_url_with_token_id():
    """Test URL building with token ID."""
    client = PredictionDataClient(api_key="test_key")
    
    url = client._build_url(
        exchange="polymarket",
        data_type=DataType.ONCHAIN_FILLS,
        identifier="0x123abc",
        date=datetime(2024, 11, 15),
        use_slug=False
    )
    
    expected = "http://datasets.predictiondata.dev/polymarket/onchain/fills/0x123abc/2024-11-15.csv.gz?slug=false&apikey=test_key"
    assert url == expected


def test_parse_timestamp_trades():
    """Test timestamp parsing for trades."""
    client = PredictionDataClient(api_key="test_key")
    
    row = {
        'exchange_timestamp': '1700000000000',
        'local_timestamp': '1700000001000'
    }
    
    local_ts, exchange_ts = client._parse_timestamp(row, DataType.TRADES)
    
    assert local_ts == 1700000001000
    assert exchange_ts == 1700000000000


def test_parse_timestamp_onchain():
    """Test timestamp parsing for onchain fills."""
    client = PredictionDataClient(api_key="test_key")
    
    row = {
        'block_timestamp': '1700000000000'
    }
    
    local_ts, exchange_ts = client._parse_timestamp(row, DataType.ONCHAIN_FILLS)
    
    assert local_ts == 1700000000000
    assert exchange_ts == 1700000000000


def test_enrich_message_trades():
    """Test message enrichment for trades."""
    client = PredictionDataClient(api_key="test_key")
    
    row = {
        'exchange_timestamp': '1700000000000',
        'local_timestamp': '1700000001000',
        'side': 'BUY',
        'size': '100.5',
        'price': '0.55'
    }
    
    message = client._enrich_message(
        row,
        DataType.TRADES,
        "market-1/YES",
        datetime(2024, 11, 15)
    )
    
    assert message['_symbol'] == "market-1/YES"
    assert message['_data_type'] == "trades"
    assert message['_date'] == "2024-11-15"
    assert message['size'] == 100.5
    assert message['price'] == 0.55
    assert message['side'] == 'BUY'


def test_enrich_message_onchain():
    """Test message enrichment for onchain fills."""
    client = PredictionDataClient(api_key="test_key")
    
    row = {
        'block_number': '12345678',
        'block_timestamp': '1700000000000',
        'side': 'SELL',
        'size': '50.25',
        'price': '0.45',
        'maker': '0xabc',
        'taker': '0xdef'
    }
    
    message = client._enrich_message(
        row,
        DataType.ONCHAIN_FILLS,
        "0x123",
        datetime(2024, 11, 15)
    )
    
    assert message['_symbol'] == "0x123"
    assert message['_data_type'] == "onchain/fills"
    assert message['block_number'] == 12345678
    assert message['size'] == 50.25
    assert message['price'] == 0.45


@pytest.mark.asyncio
async def test_client_context_manager():
    """Test client as async context manager."""
    async with PredictionDataClient(api_key="test_key") as client:
        assert client.api_key == "test_key"
    
    # Session should be closed after exiting context
    assert client._session is None or client._session.closed

