from pathlib import Path
from typing import Any, Optional

from yanga.cmake.artifacts_locator import BuildArtifact, CMakeArtifactsLocator
from yanga.cmake.cmake_backend import CMakeCommand, CMakeComment, CMakeCustomTarget, CMakeElement, CMakePath
from yanga.cmake.generator import CMakeGenerator
from yanga.docs.sphinx import SphinxConfig
from yanga.domain.component_analyzer import ComponentAnalyzer
from yanga.domain.execution_context import ExecutionContext, UserRequest, UserRequestScope, UserRequestTarget
from yanga.domain.reports import ReportRelevantFiles, ReportRelevantFileType


class ReportCMakeGenerator(CMakeGenerator):
    def __init__(
        self,
        execution_context: ExecutionContext,
        output_dir: Path,
        config: Optional[dict[str, Any]] = None,
    ) -> None:
        super().__init__(execution_context, output_dir, config)
        self.artifacts_locator = CMakeArtifactsLocator(output_dir, execution_context.create_artifacts_locator())

    def generate(self) -> list[CMakeElement]:
        elements: list[CMakeElement] = []
        elements.append(CMakeComment(f"Generated by {self.__class__.__name__}"))
        elements.extend(self.create_components_cmake_elements())
        elements.extend(self.create_variant_cmake_elements())
        return elements

    def create_variant_cmake_elements(self) -> list[CMakeElement]:
        elements: list[CMakeElement] = []
        variant_report_dir = self.artifacts_locator.cmake_variant_reports_dir

        # Create variant results target to collect all component and variant results relevant for the report
        results_target_depends: list[str | CMakePath] = [
            self.artifacts_locator.get_build_artifact(BuildArtifact.REPORT_CONFIG),
            *[
                UserRequest(UserRequestScope.COMPONENT, target=UserRequestTarget.RESULTS, component_name=component.name).target_name
                for component in self.execution_context.components
            ],
        ]

        # For all the other variant report relevant files, collect all their targets, make them uniques
        # and add them as dependencies to the results target
        extra_report_relevant_targets = [
            entry.target.target_name
            for entry in self.execution_context.data_registry.find_data(ReportRelevantFiles)
            if entry.target.scope == UserRequestScope.VARIANT
            # Avoid results which are created outside the build system (e.g., previous code generation)
            and entry.target.target
            and entry.target.target != UserRequestTarget.NONE
        ]
        results_target_depends.extend(list(dict.fromkeys(extra_report_relevant_targets)))

        results_target = CMakeCustomTarget(
            name=UserRequest(
                UserRequestScope.VARIANT,
                target=UserRequestTarget.RESULTS,
            ).target_name,
            description=f"Run all targets for all component results for {self.execution_context.variant_name}",
            commands=[],
            depends=results_target_depends,
        )
        elements.append(results_target)
        elements.append(
            CMakeCustomTarget(
                name=UserRequest(
                    UserRequestScope.VARIANT,
                    target=UserRequestTarget.REPORT,
                ).target_name,
                description=f"Run sphinx build for variant {self.execution_context.variant_name}",
                commands=[
                    CMakeCommand(
                        "${CMAKE_COMMAND}",
                        [
                            "-E",
                            "make_directory",
                            variant_report_dir,
                        ],
                    ),
                    CMakeCommand(
                        "${CMAKE_COMMAND}",
                        [
                            "-E",
                            "env",
                            f"{SphinxConfig.REPORT_CONFIGURATION_FILE_ENV_NAME}={self.artifacts_locator.get_build_artifact(BuildArtifact.REPORT_CONFIG)}",
                            "--",
                            "sphinx-build",
                            "-E",
                            "-b",
                            "html",
                            self.artifacts_locator.cmake_project_dir,
                            variant_report_dir,
                        ],
                    ),
                    CMakeCommand(
                        "yanga_cmd",
                        [
                            "fix_html_links",
                            "--report-dir",
                            variant_report_dir,
                        ],
                    ),
                ],
                depends=[
                    self.artifacts_locator.get_build_artifact(BuildArtifact.REPORT_CONFIG),
                    results_target.name,
                ],
            )
        )
        return elements

    def create_components_cmake_elements(self) -> list[CMakeElement]:
        elements: list[CMakeElement] = []
        for component in self.execution_context.components:
            component_analyzer = ComponentAnalyzer([component], self.execution_context.create_artifacts_locator())
            component_build_dir = self.artifacts_locator.get_component_build_dir(component.name)
            report_config_output_file = self.artifacts_locator.get_component_build_artifact(component.name, BuildArtifact.REPORT_CONFIG)
            source_files: list[CMakePath] = []
            if not (component.docs and component.docs.exclude_productive_code):
                source_files.extend([CMakePath(source) for source in component_analyzer.collect_sources()])

            # Check if there are any test results registered for the component
            test_results = any(
                entry
                for entry in self.execution_context.data_registry.find_data(ReportRelevantFiles)
                # TODO: when test results are registered, use the TEST_RESULT type.
                #       Currently, we assume that tests were executed if there are coverage results
                if entry.file_type == ReportRelevantFileType.COVERAGE_RESULT and entry.target.component_name == component.name
            )
            if test_results:
                source_files.extend([CMakePath(source) for source in component_analyzer.collect_test_sources()])
            source_files_output_md = [component_build_dir.joinpath(f"{source_file.to_path().name}.md") for source_file in source_files]
            component_docs_target = UserRequest(
                UserRequestScope.COMPONENT,
                target=UserRequestTarget.DOCS,
                component_name=component.name,
            )

            if source_files_output_md:
                elements.append(
                    CMakeCustomTarget(
                        name=component_docs_target.target_name,
                        description=f"Generate sources docs for component {component.name}",
                        commands=[
                            CMakeCommand(
                                "clanguru",
                                [
                                    "docs",
                                    "--source-file",
                                    source_file,
                                    "--output-file",
                                    output_md,
                                    "--compilation-database",
                                    self.artifacts_locator.get_build_artifact(BuildArtifact.COMPILE_COMMANDS),
                                    "--format",
                                    "myst",
                                ],
                            )
                            for source_file, output_md in zip(source_files, source_files_output_md)
                        ],
                        depends=[self.artifacts_locator.get_build_artifact(BuildArtifact.COMPILE_COMMANDS)],
                        byproducts=source_files_output_md,
                    )
                )
                # Register the component sources md files as relevant for the component report
                self.execution_context.data_registry.insert(
                    ReportRelevantFiles(
                        target=component_docs_target,
                        files_to_be_included=[md_output.to_path() for md_output in source_files_output_md],
                        file_type=ReportRelevantFileType.SOURCES,
                    ),
                    component_docs_target.target_name,
                )

            docs_source_files = [CMakePath(source) for source in component_analyzer.collect_docs_sources()]
            if docs_source_files:
                # Register the component documentation files as relevant for the component report
                self.execution_context.data_registry.insert(
                    ReportRelevantFiles(
                        target=component_docs_target,
                        files_to_be_included=[source.to_path() for source in docs_source_files],
                        file_type=ReportRelevantFileType.DOCS,
                    ),
                    component_docs_target.target_name,
                )

            component_results_target = UserRequest(
                UserRequestScope.COMPONENT,
                target=UserRequestTarget.RESULTS,
                component_name=component.name,
            )

            # Collect all component targets that registered files relevant for the report
            result_targets = [entry.target for entry in self.execution_context.data_registry.find_data(ReportRelevantFiles) if entry.target.component_name == component.name]
            # Make list unique and keep order
            result_targets = list(dict.fromkeys(result_targets))

            elements.append(
                CMakeCustomTarget(
                    name=component_results_target.target_name,
                    description=f"Execute targets to get all results for component {component.name}",
                    commands=[],
                    depends=[
                        self.artifacts_locator.get_build_artifact(BuildArtifact.REPORT_CONFIG),
                        *[target.target_name for target in result_targets],
                    ],
                )
            )
            component_report_target = UserRequest(
                UserRequestScope.COMPONENT,
                target=UserRequestTarget.REPORT,
                component_name=component.name,
            )

            component_report_dir = self.artifacts_locator.get_component_reports_dir(component.name)
            elements.append(
                CMakeCustomTarget(
                    name=component_report_target.target_name,
                    description=f"Generate report for component {component.name}",
                    commands=[
                        CMakeCommand(
                            "${CMAKE_COMMAND}",
                            [
                                "-E",
                                "make_directory",
                                component_report_dir,
                            ],
                        ),
                        CMakeCommand(
                            "yanga_cmd",
                            [
                                "report_config",
                                "--component-name",
                                component.name,
                                "--output-file",
                                report_config_output_file,
                                "--variant-report-config",
                                self.artifacts_locator.get_build_artifact(BuildArtifact.REPORT_CONFIG),
                            ],
                        ),
                        CMakeCommand(
                            "${CMAKE_COMMAND}",
                            [
                                "-E",
                                "env",
                                f"REPORT_CONFIGURATION_FILE={report_config_output_file}",
                                "--",
                                "sphinx-build",
                                "-E",
                                "-b",
                                "html",
                                self.artifacts_locator.cmake_project_dir,
                                component_report_dir,
                            ],
                        ),
                        CMakeCommand(
                            "yanga_cmd",
                            [
                                "fix_html_links",
                                "--report-dir",
                                component_report_dir,
                            ],
                        ),
                    ],
                    depends=[
                        component_results_target.target_name,
                    ],
                )
            )

        return elements
