from pathlib import Path
from typing import Any, Optional

from clanguru.doc_generator import DocStructure, MarkdownFormatter, Section, TextContent
from kspl.generate import FileWriter, HeaderWriter, JsonWriter
from kspl.kconfig import ConfigElementType, ConfigurationData, KConfig, TriState
from py_app_dev.core.logging import logger
from pypeline.domain.pipeline import PipelineStep

from yanga.domain.execution_context import ExecutionContext, IncludeDirectoriesProvider, UserRequest, UserRequestScope, UserRequestTarget
from yanga.domain.reports import FeaturesReportRelevantFile, ReportRelevantFiles, ReportRelevantFileType


class FeaturesDocumentationWriter(FileWriter):
    """Writes the ConfigurationData as Markdown documentation file."""

    def generate_content(self, configuration_data: ConfigurationData) -> str:
        def format_value(elem_type: ConfigElementType, value: Any) -> str:
            if elem_type in (ConfigElementType.BOOL, ConfigElementType.TRISTATE):
                if value == TriState.Y:
                    return "Y"
                if value == TriState.M:
                    return "M"
                return "N"
            if elem_type is ConfigElementType.STRING:
                return f"`{value}`" if value is not None else ""
            if elem_type is ConfigElementType.HEX:
                try:
                    return f"0x{int(value):X}"
                except Exception:  # pragma: no cover
                    return str(value)
            return str(value)

        def type_label(elem_type: ConfigElementType) -> str:
            labels = {
                ConfigElementType.BOOL: "BOOL",
                ConfigElementType.TRISTATE: "TRISTATE",
                ConfigElementType.STRING: "STRING",
                ConfigElementType.INT: "INT",
                ConfigElementType.HEX: "HEX",
            }
            return labels.get(elem_type, "UNKNOWN")

        elements_sorted = sorted(configuration_data.elements, key=lambda e: e.name)
        total = len(elements_sorted)
        bool_like = [e for e in elements_sorted if e.type in (ConfigElementType.BOOL, ConfigElementType.TRISTATE)]
        enabled = [e for e in bool_like if e.value in (TriState.Y, TriState.M)]

        doc = DocStructure("Variant Feature Configuration")

        # Summary section
        summary_section = Section("Summary")
        summary_section.add_content(
            TextContent(
                "\n".join(
                    [
                        "This document lists the resolved KConfig feature symbols for the current variant.",
                        "",
                        f"Total symbols: {total}",
                        "",
                        f"* Boolean / tristate symbols: {len(bool_like)} (enabled: {len(enabled)}, disabled: {len(bool_like) - len(enabled)})",
                        f"* Other typed symbols (string/int/hex): {total - len(bool_like)}",
                    ]
                )
            )
        )
        doc.add_section(summary_section)

        # Symbols table section
        symbols_section = Section("Symbols")
        table_lines = ["| Name | Type | Value | Defined Macro |", "|------|------|-------|---------------|"]
        for elem in elements_sorted:
            macro = ""
            if elem.type in (ConfigElementType.BOOL, ConfigElementType.TRISTATE):
                if elem.value == TriState.Y:
                    macro = f"CONFIG_{elem.name}"
                elif elem.value == TriState.M:
                    macro = f"CONFIG_{elem.name}_MODULE"
            elif elem.type in (ConfigElementType.STRING, ConfigElementType.INT, ConfigElementType.HEX):
                macro = f"CONFIG_{elem.name}"
            table_lines.append(f"| {elem.name} | {type_label(elem.type)} | {format_value(elem.type, elem.value)} | {macro} |")
        symbols_section.add_content(TextContent("\n".join(table_lines)))
        doc.add_section(symbols_section)

        footer_section = Section("Generated By")
        footer_section.add_content(TextContent("KConfigGen"))
        doc.add_section(footer_section)

        return MarkdownFormatter().format(doc)


class KConfigIncludeDirectoriesProvider(IncludeDirectoriesProvider):
    def __init__(self, output_dir: Path) -> None:
        self.output_dir = output_dir

    def get_include_directories(self) -> list[Path]:
        return [self.output_dir]


class KConfigGen(PipelineStep[ExecutionContext]):
    def __init__(self, execution_context: ExecutionContext, group_name: Optional[str] = None, config: Optional[dict[str, Any]] = None) -> None:
        super().__init__(execution_context, group_name, config)
        self.logger = logger.bind()
        self.input_files: list[Path] = []

    @property
    def output_dir(self) -> Path:
        return self.execution_context.create_artifacts_locator().variant_build_dir / "kconfig"

    def get_name(self) -> str:
        return self.__class__.__name__

    @property
    def header_file(self) -> Path:
        return self.output_dir.joinpath("autoconf.h")

    @property
    def json_config_file(self) -> Path:
        return self.output_dir.joinpath("autoconf.json")

    @property
    def features_doc_file(self) -> Path:
        return self.output_dir.joinpath("autoconf.md")

    def run(self) -> int:
        self.logger.debug(f"Run {self.get_name()} stage. Output dir: {self.output_dir}")
        kconfig_model_file = self.project_root_dir.joinpath("KConfig")
        if not kconfig_model_file.is_file():
            self.logger.info("No KConfig file found. Skip this stage.")
            return 0
        kconfig = KConfig(
            kconfig_model_file,
            self.execution_context.features_selection_file,
        )
        self.input_files = kconfig.get_parsed_files()
        config = kconfig.collect_config_data()
        HeaderWriter(self.header_file).write(config)
        JsonWriter(self.json_config_file).write(config)
        # Register the JSON config file as relevant for configuring the report generation
        self.execution_context.data_registry.insert(
            FeaturesReportRelevantFile(provider=self.get_name(), json_config_file=self.json_config_file),
            self.get_name(),
        )
        FeaturesDocumentationWriter(self.features_doc_file).write(config)
        # Register the documentation file as relevant for the variant report
        self.execution_context.data_registry.insert(
            ReportRelevantFiles(
                target=UserRequest(
                    UserRequestScope.VARIANT,
                    target=UserRequestTarget.NONE,
                ),
                files_to_be_included=[self.features_doc_file],
                file_type=ReportRelevantFileType.OTHER,
            ),
            self.get_name(),
        )
        return 0

    def get_inputs(self) -> list[Path]:
        # TODO: Use as input only the user config file where variant configuration is defined.
        # Now all the user config files are used as inputs, which will trigger the generation
        # if any of the file has changed.
        return self.execution_context.user_config_files + self.input_files

    def get_outputs(self) -> list[Path]:
        return [self.header_file, self.features_doc_file, self.json_config_file]

    def update_execution_context(self) -> None:
        # Update the include directories for the subsequent steps
        self.execution_context.add_include_dirs_provider(KConfigIncludeDirectoriesProvider(self.output_dir))
