"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccountProvider = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const is_complete_handler_function_1 = require("./is-complete-handler-function");
const on_event_handler_function_1 = require("./on-event-handler-function");
/**
 * Creates a custom resource provider to asynchronously create Accounts in AWS organization. <strong>Account deletion is currently not supported!</strong>
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#provider-framework
 */
class AccountProvider extends aws_cdk_lib_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.onEventHandler = new on_event_handler_function_1.OnEventHandlerFunction(this, "OnEventHandlerFunction", {
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            initialPolicy: [
                new aws_iam_1.PolicyStatement({
                    actions: ["organizations:CreateAccount", "organizations:ListAccounts"],
                    resources: ["*"],
                }),
            ],
        });
        this.isCompleteHandler = new is_complete_handler_function_1.IsCompleteHandlerFunction(this, "IsCompleteHandlerFunction", {
            timeout: aws_cdk_lib_1.Duration.minutes(1),
            initialPolicy: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        "organizations:DescribeCreateAccountStatus",
                        "organizations:ListAccounts",
                        "organizations:DescribeAccount",
                        "organizations:ListParents",
                        "organizations:ListRoots",
                        "organizations:MoveAccount",
                    ],
                    resources: ["*"],
                }),
            ],
        });
        this.provider = new custom_resources_1.Provider(this, "Provider", {
            onEventHandler: this.onEventHandler,
            isCompleteHandler: this.isCompleteHandler,
            queryInterval: aws_cdk_lib_1.Duration.seconds(5),
        });
    }
    /**
     * Retrieve AccountProvider as stack singleton resource.
     *
     * @see https://github.com/aws/aws-cdk/issues/5023
     */
    static getOrCreate(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = "cdk-organizations.AccountProvider";
        const existing = stack.node.tryFindChild(id);
        return existing || new AccountProvider(stack, id, {});
    }
}
exports.AccountProvider = AccountProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWNjb3VudC1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9hY2NvdW50LXByb3ZpZGVyL2FjY291bnQtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkNBQTZFO0FBQzdFLGlEQUFzRDtBQUV0RCxtRUFBd0Q7QUFFeEQsaUZBQTJFO0FBQzNFLDJFQUFxRTtBQUdyRTs7OztHQUlHO0FBQ0gsTUFBYSxlQUFnQixTQUFRLHlCQUFXO0lBK0I5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTJCO1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXhCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxrREFBc0IsQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7WUFDL0UsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixhQUFhLEVBQUU7Z0JBQ2IsSUFBSSx5QkFBZSxDQUFDO29CQUNsQixPQUFPLEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSw0QkFBNEIsQ0FBQztvQkFDdEUsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNqQixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSx3REFBeUIsQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLEVBQUU7WUFDeEYsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUM1QixhQUFhLEVBQUU7Z0JBQ2IsSUFBSSx5QkFBZSxDQUFDO29CQUNsQixPQUFPLEVBQUU7d0JBQ1AsMkNBQTJDO3dCQUMzQyw0QkFBNEI7d0JBQzVCLCtCQUErQjt3QkFDL0IsMkJBQTJCO3dCQUMzQix5QkFBeUI7d0JBQ3pCLDJCQUEyQjtxQkFDNUI7b0JBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNqQixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksMkJBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzdDLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztZQUNuQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCO1lBQ3pDLGFBQWEsRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDbkMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQWpFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFnQjtRQUN4QyxNQUFNLEtBQUssR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM5QixNQUFNLEVBQUUsR0FBRyxtQ0FBbUMsQ0FBQztRQUMvQyxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUM3QyxPQUFRLFFBQTRCLElBQUksSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUM3RSxDQUFDO0NBd0RGO0FBbkVELDBDQW1FQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IER1cmF0aW9uLCBOZXN0ZWRTdGFjaywgTmVzdGVkU3RhY2tQcm9wcywgU3RhY2sgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtaWFtXCI7XG5pbXBvcnQgeyBGdW5jdGlvbiB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtbGFtYmRhXCI7XG5pbXBvcnQgeyBQcm92aWRlciB9IGZyb20gXCJhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgSXNDb21wbGV0ZUhhbmRsZXJGdW5jdGlvbiB9IGZyb20gXCIuL2lzLWNvbXBsZXRlLWhhbmRsZXItZnVuY3Rpb25cIjtcbmltcG9ydCB7IE9uRXZlbnRIYW5kbGVyRnVuY3Rpb24gfSBmcm9tIFwiLi9vbi1ldmVudC1oYW5kbGVyLWZ1bmN0aW9uXCI7XG5leHBvcnQgaW50ZXJmYWNlIEFjY291bnRQcm92aWRlclByb3BzIGV4dGVuZHMgTmVzdGVkU3RhY2tQcm9wcyB7fVxuXG4vKipcbiAqIENyZWF0ZXMgYSBjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXIgdG8gYXN5bmNocm9ub3VzbHkgY3JlYXRlIEFjY291bnRzIGluIEFXUyBvcmdhbml6YXRpb24uIDxzdHJvbmc+QWNjb3VudCBkZWxldGlvbiBpcyBjdXJyZW50bHkgbm90IHN1cHBvcnRlZCE8L3N0cm9uZz5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jZGsvYXBpL3YxL2RvY3MvY3VzdG9tLXJlc291cmNlcy1yZWFkbWUuaHRtbCNwcm92aWRlci1mcmFtZXdvcmtcbiAqL1xuZXhwb3J0IGNsYXNzIEFjY291bnRQcm92aWRlciBleHRlbmRzIE5lc3RlZFN0YWNrIHtcbiAgLyoqXG4gICAqIFJldHJpZXZlIEFjY291bnRQcm92aWRlciBhcyBzdGFjayBzaW5nbGV0b24gcmVzb3VyY2UuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy81MDIzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGdldE9yQ3JlYXRlKHNjb3BlOiBDb25zdHJ1Y3QpOiBBY2NvdW50UHJvdmlkZXIge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGlkID0gXCJjZGstb3JnYW5pemF0aW9ucy5BY2NvdW50UHJvdmlkZXJcIjtcbiAgICBjb25zdCBleGlzdGluZyA9IHN0YWNrLm5vZGUudHJ5RmluZENoaWxkKGlkKTtcbiAgICByZXR1cm4gKGV4aXN0aW5nIGFzIEFjY291bnRQcm92aWRlcikgfHwgbmV3IEFjY291bnRQcm92aWRlcihzdGFjaywgaWQsIHt9KTtcbiAgfVxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBBY2NvdW50IGFuZCByZXR1cm5zIHRoZSBDcmVhdGVBY2NvdW50U3RhdHVzIElEIG9uIENyZWF0ZS4gUGFzc2VzIHRoZSBQaHlzaWNhbFJlc291cmNlSWQgb24gVXBkYXRlIHRocm91Z2guIEZhaWxzIG9uIERlbGV0ZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTSmF2YVNjcmlwdFNESy9sYXRlc3QvQVdTL09yZ2FuaXphdGlvbnMuaHRtbCNjcmVhdGVBY2NvdW50LXByb3BlcnR5XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgb25FdmVudEhhbmRsZXI6IEZ1bmN0aW9uO1xuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBDcmVhdGVBY2NvdW50U3RhdHVzIGFuZCByZXR1cm5zIHRoZSBjb21wbGV0aW9ucyBzdGF0dXMuIEZhaWxzIG9uIERlbGV0ZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTSmF2YVNjcmlwdFNESy9sYXRlc3QvQVdTL09yZ2FuaXphdGlvbnMuaHRtbCNkZXNjcmliZUNyZWF0ZUFjY291bnRTdGF0dXMtcHJvcGVydHlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpc0NvbXBsZXRlSGFuZGxlcjogRnVuY3Rpb247XG4gIC8qKlxuICAgKiBUaGUgYXN5bmNocm9udW9zIHByb3ZpZGVyIHRvIGNyZWF0ZSBvciB1cGRhdGUgYW4gQWNjb3VudC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS92MS9kb2NzL2N1c3RvbS1yZXNvdXJjZXMtcmVhZG1lLmh0bWwjYXN5bmNocm9ub3VzLXByb3ZpZGVycy1pc2NvbXBsZXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvdmlkZXI6IFByb3ZpZGVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBY2NvdW50UHJvdmlkZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy5vbkV2ZW50SGFuZGxlciA9IG5ldyBPbkV2ZW50SGFuZGxlckZ1bmN0aW9uKHRoaXMsIFwiT25FdmVudEhhbmRsZXJGdW5jdGlvblwiLCB7XG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDEwKSxcbiAgICAgIGluaXRpYWxQb2xpY3k6IFtcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogW1wib3JnYW5pemF0aW9uczpDcmVhdGVBY2NvdW50XCIsIFwib3JnYW5pemF0aW9uczpMaXN0QWNjb3VudHNcIl0sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbXCIqXCJdLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmlzQ29tcGxldGVIYW5kbGVyID0gbmV3IElzQ29tcGxldGVIYW5kbGVyRnVuY3Rpb24odGhpcywgXCJJc0NvbXBsZXRlSGFuZGxlckZ1bmN0aW9uXCIsIHtcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICBpbml0aWFsUG9saWN5OiBbXG4gICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgIFwib3JnYW5pemF0aW9uczpEZXNjcmliZUNyZWF0ZUFjY291bnRTdGF0dXNcIixcbiAgICAgICAgICAgIFwib3JnYW5pemF0aW9uczpMaXN0QWNjb3VudHNcIixcbiAgICAgICAgICAgIFwib3JnYW5pemF0aW9uczpEZXNjcmliZUFjY291bnRcIixcbiAgICAgICAgICAgIFwib3JnYW5pemF0aW9uczpMaXN0UGFyZW50c1wiLFxuICAgICAgICAgICAgXCJvcmdhbml6YXRpb25zOkxpc3RSb290c1wiLFxuICAgICAgICAgICAgXCJvcmdhbml6YXRpb25zOk1vdmVBY2NvdW50XCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtcIipcIl0sXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHRoaXMucHJvdmlkZXIgPSBuZXcgUHJvdmlkZXIodGhpcywgXCJQcm92aWRlclwiLCB7XG4gICAgICBvbkV2ZW50SGFuZGxlcjogdGhpcy5vbkV2ZW50SGFuZGxlcixcbiAgICAgIGlzQ29tcGxldGVIYW5kbGVyOiB0aGlzLmlzQ29tcGxldGVIYW5kbGVyLFxuICAgICAgcXVlcnlJbnRlcnZhbDogRHVyYXRpb24uc2Vjb25kcyg1KSxcbiAgICB9KTtcbiAgfVxufVxuIl19