"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_sdk_1 = require("aws-sdk");
let organizationsClient;
/**
 * The onEvent handler is invoked whenever a resource lifecycle event for an organization occurs
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#handling-lifecycle-events-onevent
 */
async function handler(event) {
    console.log(`Request of type ${event.RequestType} received`);
    if (!organizationsClient) {
        organizationsClient = new aws_sdk_1.Organizations({ region: "us-east-1" });
    }
    console.log("Payload: %j", event);
    if (event.RequestType == "Create") {
        // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#createOrganization-property
        try {
            const response = await organizationsClient
                .createOrganization({
                FeatureSet: event.ResourceProperties.FeatureSet,
            })
                .promise();
            console.log("Creating organization: %j", response);
            return {
                PhysicalResourceId: response.Organization?.Id,
                Data: {
                    ...response.Organization,
                },
            };
        }
        catch (e) {
            const error = e;
            if (error.code == "AlreadyInOrganizationException") {
                // https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html#API_CreateOrganization_Errors
                console.log("Organization already created.");
            }
            else {
                throw error;
            }
        }
    }
    // https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/Organizations.html#describeOrganization-property
    const response = await organizationsClient
        .describeOrganization()
        .promise();
    // TODO: Try to delete organization (RemovalPolicy)
    return {
        PhysicalResourceId: response.Organization?.Id,
        Data: {
            ...response.Organization,
        },
    };
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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