"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationProvider = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const on_event_handler_function_1 = require("./on-event-handler-function");
/**
 * Creates a custom resource provider to create the organization in AWS organization.
 *
 * <strong>If the organization already exists, it will be just returned.</strong>
 * <strong>Organization deletion is currently not supported!</strong>
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#provider-framework
 */
class OrganizationProvider extends aws_cdk_lib_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.onEventHandler = new on_event_handler_function_1.OnEventHandlerFunction(this, "OnEventHandlerFunction", {
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            initialPolicy: [
                new aws_iam_1.PolicyStatement({
                    actions: ["organizations:CreateOrganization", "organizations:DescribeOrganization"],
                    resources: ["*"],
                }),
                // permit the creation of service-linked role https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_create.html#create-org
                new aws_iam_1.PolicyStatement({
                    actions: ["iam:CreateServiceLinkedRole"],
                    resources: ["arn:aws:iam::*:role/*"],
                }),
            ],
        });
        this.provider = new custom_resources_1.Provider(this, "Provider", {
            onEventHandler: this.onEventHandler,
        });
    }
    /**
     * Retrieve OrganizationProvider as stack singleton resource.
     *
     * @see https://github.com/aws/aws-cdk/issues/5023
     */
    static getOrCreate(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = "cdk-organizations.OrganizationProvider";
        const existing = stack.node.tryFindChild(id);
        return existing || new OrganizationProvider(stack, id, {});
    }
}
exports.OrganizationProvider = OrganizationProvider;
//# sourceMappingURL=data:application/json;base64,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