import { TagManager } from "aws-cdk-lib";
import * as aws_iam from "aws-cdk-lib/aws-iam";
import * as custom_resources from "aws-cdk-lib/custom-resources";
import { Construct, IConstruct } from "constructs";
import { IParent } from "./parent";
import { IPolicy, PolicyType } from "./policy";
import { IPolicyAttachmentTarget } from "./policy-attachment";
import { ITaggableResource } from "./tag-resource";
/**
 * Specifies the feature set supported by the new organization. Each feature set supports different levels of functionality.
 *
 * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#feature-set
 */
export declare enum FeatureSet {
    /**
     * All member accounts have their bills consolidated to and paid by the management account. For more information, see [Consolidated billing](https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#feature-set-cb-only) in the AWS Organizations User Guide. The consolidated billing feature subset isn’t available for organizations in the AWS GovCloud (US) Region.
     */
    CONSOLIDATED_BILLING = "CONSOLIDATED_BILLING",
    /**
     * In addition to all the features supported by the consolidated billing feature set, the management account can also apply any policy type to any member account in the organization. For more information, see [All features](https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#feature-set-all) in the AWS Organizations User Guide.
     */
    ALL = "ALL"
}
export interface OrganizationProps {
    /**
     * Enabling features in your organization.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_support-all-features.html
     *
     * @default ALL
     */
    readonly featureSet?: FeatureSet;
}
/**
 * Creates an organization to consolidate your AWS accounts so that you can administer them as a single unit. An organization has one management account along with zero or more member accounts. You can organize the accounts in a hierarchical, tree-like structure with a root at the top and organizational units nested under the root. Each account can be directly in the root, or placed in one of the OUs in the hierarchy. An organization has the functionality that is determined by the feature set that you enable.
 *
 * <strong>The account whose user is calling the CreateOrganization operation automatically becomes the management account of the new organization.</strong>
 *
 * <strong>For deletion of an organization you must previously remove all the member accounts, OUs, and policies from the organization!</strong>
 *
 * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_create.html#create-org
 */
export interface IOrganization extends IConstruct {
    /**
     * The unique identifier (ID) of an organization. The regex pattern for an organization ID string requires "o-" followed by from 10 to 32 lowercase letters or digits.
     */
    readonly organizationId: string;
    /**
     * The Amazon Resource Name (ARN) of an organization.
     */
    readonly organizationArn: string;
    /**
     * Specifies the functionality that currently is available to the organization. If set to "ALL", then all features are enabled and policies can be applied to accounts in the organization. If set to "CONSOLIDATED_BILLING", then only consolidated billing functionality is available.
     */
    readonly featureSet: FeatureSet;
    /**
     * The Amazon Resource Name (ARN) of the account that is designated as the management account for the organization.
     */
    readonly managementAccountArn: string;
    /**
     * The unique identifier (ID) of the management account of an organization.
     */
    readonly managementAccountId: string;
    /**
     * The email address that is associated with the AWS account that is designated as the management account for the organization.
     */
    readonly managementAccountEmail: string;
    /**
     * The principal that represents this AWS Organization
     */
    readonly principal: aws_iam.IPrincipal;
}
export declare class Organization extends Construct implements IOrganization {
    /**
     * Describe the organization that the current account belongs to.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    static of(scope: Construct, id: string): IOrganization;
    readonly organizationId: string;
    readonly organizationArn: string;
    readonly featureSet: FeatureSet;
    readonly managementAccountArn: string;
    readonly managementAccountId: string;
    readonly managementAccountEmail: string;
    readonly principal: aws_iam.IPrincipal;
    /**
     * The root of the current organization, which is automatically created.
     */
    readonly root: Root;
    private readonly resource;
    constructor(scope: Construct, id: string, props?: OrganizationProps);
    /**
     * Enables trusted access for a supported AWS service (trusted service), which performs tasks in your organization and its accounts on your behalf.
     * @param servicePrincipal The supported AWS service that you specify
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services_list.html
     */
    enableAwsServiceAccess(servicePrincipal: string): void;
    /**
     * Enables policy types in the following two broad categories: Authorization policies and Management policies.
     * @param policyType: the type of the policy that you specify
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html#orgs-policy-types
     */
    enablePolicyType(policyType: PolicyType): void;
    /**
     * Attach a policy. Before you can attach the policy, you must enable that policy type for use. You can use policies when you have all features enabled.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html
     */
    attachPolicy(policy: IPolicy): void;
}
/**
 * The parent container for all the accounts for your organization. If you apply a policy to the root, it applies to all organizational units (OUs) and accounts in the organization.
 * <strong>Currently, you can have only one root. AWS Organizations automatically creates it for you when you create an organization.</strong>
 * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html
 */
export declare class Root extends Construct implements IParent, IPolicyAttachmentTarget, ITaggableResource {
    /**
     * The unique identifier (ID) for the root. The regex pattern for a root ID string requires "r-" followed by from 4 to 32 lowercase letters or digits.
     */
    readonly rootId: string;
    protected readonly resource: custom_resources.AwsCustomResource;
    private readonly scope;
    readonly tags: TagManager;
    constructor(scope: Construct, id: string);
    identifier(): string;
    /**
     * Attach a policy. Before you can attach the policy, you must enable that policy type for use. You can use policies when you have all features enabled.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html
     */
    attachPolicy(policy: IPolicy): void;
    /**
     * Enables and disables Enables a policy type. After you enable a policy type in a root, you can attach policies of that type to the root, any organizational unit (OU), or account in that root.
     *
     * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_enable-disable.html
     */
    enablePolicyType(policyType: PolicyType): void;
}
