"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Policy = exports.PolicyType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const tag_resource_1 = require("./tag-resource");
const validators_1 = require("./validators");
/**
 * Organizations offers policy types in the following two broad categories:
 * <ol>
 *     <li>Authorization policies help you to centrally manage the security of the AWS accounts in your organization.</li>
 *     <li>Management policies enable you to centrally configure and manage AWS services and their features.</li>
 * </ol>
 *
 * @see https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies.html#orgs-policy-types
 */
var PolicyType;
(function (PolicyType) {
    /**
     * Service control policies (SCPs) offer central control over the maximum available permissions for all of the accounts in your organization.
     */
    PolicyType["SERVICE_CONTROL_POLICY"] = "SERVICE_CONTROL_POLICY";
    /**
     * Tag policies help you standardize the tags attached to the AWS resources in your organization's accounts.
     */
    PolicyType["TAG_POLICY"] = "TAG_POLICY";
    /**
     * Backup policies help you centrally manage and apply backup plans to the AWS resources across your organization's accounts.
     */
    PolicyType["BACKUP_POLICY"] = "BACKUP_POLICY";
    /**
     * Artificial Intelligence (AI) services opt-out policies enable you to control data collection for AWS AI services for all of your organization's accounts.
     */
    PolicyType["AISERVICES_OPT_OUT_POLICY"] = "AISERVICES_OPT_OUT_POLICY";
})(PolicyType = exports.PolicyType || (exports.PolicyType = {}));
class Policy extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.tags = new aws_cdk_lib_1.TagManager(aws_cdk_lib_1.TagType.KEY_VALUE, "Custom::Organizations_Policy");
        const { content, description, policyName, policyType } = props;
        if (!validators_1.Validators.of().policyContent(content)) {
            aws_cdk_lib_1.Annotations.of(this).addError("The text content of the policy must be valid and between 1 and 1,000,000 characters long.");
        }
        const policy = new custom_resources_1.AwsCustomResource(this, "PolicyCustomResource", {
            resourceType: "Custom::Organizations_Policy",
            onCreate: {
                service: "Organizations",
                action: "createPolicy",
                region: "us-east-1",
                parameters: {
                    Content: content,
                    Description: description,
                    Name: policyName,
                    Type: policyType,
                },
                outputPaths: ["Policy.PolicySummary.Id"],
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse("Policy.PolicySummary.Id"),
            },
            onUpdate: {
                service: "Organizations",
                action: "updatePolicy",
                region: "us-east-1",
                parameters: {
                    Content: content,
                    Description: description,
                    Name: policyName,
                    PolicyId: new custom_resources_1.PhysicalResourceIdReference(),
                },
                outputPaths: ["Policy.PolicySummary.Id"],
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse("Policy.PolicySummary.Id"),
            },
            onDelete: {
                service: "Organizations",
                action: "deletePolicy",
                region: "us-east-1",
                parameters: {
                    PolicyId: new custom_resources_1.PhysicalResourceIdReference(),
                },
            },
            installLatestAwsSdk: false,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        this.policyId = policy.getResponseField("Policy.PolicySummary.Id");
        const tagResource = new tag_resource_1.TagResource(this, "Tags", { resourceId: this.policyId, tags: this.tags.renderedTags });
        tagResource.node.addDependency(policy);
    }
    identifier() {
        return this.policyId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@pepperize/cdk-organizations.Policy", version: "0.7.173" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3BvbGljeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUErRDtBQUMvRCxtRUFLc0M7QUFDdEMsMkNBQW1EO0FBQ25ELGlEQUFnRTtBQUNoRSw2Q0FBMEM7QUFFMUM7Ozs7Ozs7O0dBUUc7QUFDSCxJQUFZLFVBaUJYO0FBakJELFdBQVksVUFBVTtJQUNwQjs7T0FFRztJQUNILCtEQUFpRCxDQUFBO0lBQ2pEOztPQUVHO0lBQ0gsdUNBQXlCLENBQUE7SUFDekI7O09BRUc7SUFDSCw2Q0FBK0IsQ0FBQTtJQUMvQjs7T0FFRztJQUNILHFFQUF1RCxDQUFBO0FBQ3pELENBQUMsRUFqQlcsVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFpQnJCO0FBb0NELE1BQWEsTUFBTyxTQUFRLHNCQUFTO0lBS25DLFlBQW1CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtCO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFIVixTQUFJLEdBQUcsSUFBSSx3QkFBVSxDQUFDLHFCQUFPLENBQUMsU0FBUyxFQUFFLDhCQUE4QixDQUFDLENBQUM7UUFLaEYsTUFBTSxFQUFFLE9BQU8sRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRSxHQUFHLEtBQUssQ0FBQztRQUUvRCxJQUFJLENBQUMsdUJBQVUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDM0MseUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQiwyRkFBMkYsQ0FDNUYsQ0FBQztTQUNIO1FBRUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxvQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7WUFDakUsWUFBWSxFQUFFLDhCQUE4QjtZQUM1QyxRQUFRLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLE1BQU0sRUFBRSxjQUFjO2dCQUN0QixNQUFNLEVBQUUsV0FBVztnQkFDbkIsVUFBVSxFQUFFO29CQUNWLE9BQU8sRUFBRSxPQUFPO29CQUNoQixXQUFXLEVBQUUsV0FBVztvQkFDeEIsSUFBSSxFQUFFLFVBQVU7b0JBQ2hCLElBQUksRUFBRSxVQUFVO2lCQUNqQjtnQkFDRCxXQUFXLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQztnQkFDeEMsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsWUFBWSxDQUFDLHlCQUF5QixDQUFDO2FBQy9FO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxlQUFlO2dCQUN4QixNQUFNLEVBQUUsY0FBYztnQkFDdEIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLFVBQVUsRUFBRTtvQkFDVixPQUFPLEVBQUUsT0FBTztvQkFDaEIsV0FBVyxFQUFFLFdBQVc7b0JBQ3hCLElBQUksRUFBRSxVQUFVO29CQUNoQixRQUFRLEVBQUUsSUFBSSw4Q0FBMkIsRUFBRTtpQkFDNUM7Z0JBQ0QsV0FBVyxFQUFFLENBQUMseUJBQXlCLENBQUM7Z0JBQ3hDLGtCQUFrQixFQUFFLHFDQUFrQixDQUFDLFlBQVksQ0FBQyx5QkFBeUIsQ0FBQzthQUMvRTtZQUNELFFBQVEsRUFBRTtnQkFDUixPQUFPLEVBQUUsZUFBZTtnQkFDeEIsTUFBTSxFQUFFLGNBQWM7Z0JBQ3RCLE1BQU0sRUFBRSxXQUFXO2dCQUNuQixVQUFVLEVBQUU7b0JBQ1YsUUFBUSxFQUFFLElBQUksOENBQTJCLEVBQUU7aUJBQzVDO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRSxLQUFLO1lBQzFCLE1BQU0sRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZLENBQUMsRUFBRSxTQUFTLEVBQUUsMENBQXVCLENBQUMsWUFBWSxFQUFFLENBQUM7U0FDbEcsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsR0FBRyxNQUFNLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUVuRSxNQUFNLFdBQVcsR0FBRyxJQUFJLDBCQUFXLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxFQUFFLFVBQVUsRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUM7UUFDL0csV0FBVyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVELFVBQVU7UUFDUixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDdkIsQ0FBQzs7QUEvREgsd0JBZ0VDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQW5ub3RhdGlvbnMsIFRhZ01hbmFnZXIsIFRhZ1R5cGUgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7XG4gIEF3c0N1c3RvbVJlc291cmNlLFxuICBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSxcbiAgUGh5c2ljYWxSZXNvdXJjZUlkLFxuICBQaHlzaWNhbFJlc291cmNlSWRSZWZlcmVuY2UsXG59IGZyb20gXCJhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgSVRhZ2dhYmxlUmVzb3VyY2UsIFRhZ1Jlc291cmNlIH0gZnJvbSBcIi4vdGFnLXJlc291cmNlXCI7XG5pbXBvcnQgeyBWYWxpZGF0b3JzIH0gZnJvbSBcIi4vdmFsaWRhdG9yc1wiO1xuXG4vKipcbiAqIE9yZ2FuaXphdGlvbnMgb2ZmZXJzIHBvbGljeSB0eXBlcyBpbiB0aGUgZm9sbG93aW5nIHR3byBicm9hZCBjYXRlZ29yaWVzOlxuICogPG9sPlxuICogICAgIDxsaT5BdXRob3JpemF0aW9uIHBvbGljaWVzIGhlbHAgeW91IHRvIGNlbnRyYWxseSBtYW5hZ2UgdGhlIHNlY3VyaXR5IG9mIHRoZSBBV1MgYWNjb3VudHMgaW4geW91ciBvcmdhbml6YXRpb24uPC9saT5cbiAqICAgICA8bGk+TWFuYWdlbWVudCBwb2xpY2llcyBlbmFibGUgeW91IHRvIGNlbnRyYWxseSBjb25maWd1cmUgYW5kIG1hbmFnZSBBV1Mgc2VydmljZXMgYW5kIHRoZWlyIGZlYXR1cmVzLjwvbGk+XG4gKiA8L29sPlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29yZ2FuaXphdGlvbnMvbGF0ZXN0L3VzZXJndWlkZS9vcmdzX21hbmFnZV9wb2xpY2llcy5odG1sI29yZ3MtcG9saWN5LXR5cGVzXG4gKi9cbmV4cG9ydCBlbnVtIFBvbGljeVR5cGUge1xuICAvKipcbiAgICogU2VydmljZSBjb250cm9sIHBvbGljaWVzIChTQ1BzKSBvZmZlciBjZW50cmFsIGNvbnRyb2wgb3ZlciB0aGUgbWF4aW11bSBhdmFpbGFibGUgcGVybWlzc2lvbnMgZm9yIGFsbCBvZiB0aGUgYWNjb3VudHMgaW4geW91ciBvcmdhbml6YXRpb24uXG4gICAqL1xuICBTRVJWSUNFX0NPTlRST0xfUE9MSUNZID0gXCJTRVJWSUNFX0NPTlRST0xfUE9MSUNZXCIsXG4gIC8qKlxuICAgKiBUYWcgcG9saWNpZXMgaGVscCB5b3Ugc3RhbmRhcmRpemUgdGhlIHRhZ3MgYXR0YWNoZWQgdG8gdGhlIEFXUyByZXNvdXJjZXMgaW4geW91ciBvcmdhbml6YXRpb24ncyBhY2NvdW50cy5cbiAgICovXG4gIFRBR19QT0xJQ1kgPSBcIlRBR19QT0xJQ1lcIixcbiAgLyoqXG4gICAqIEJhY2t1cCBwb2xpY2llcyBoZWxwIHlvdSBjZW50cmFsbHkgbWFuYWdlIGFuZCBhcHBseSBiYWNrdXAgcGxhbnMgdG8gdGhlIEFXUyByZXNvdXJjZXMgYWNyb3NzIHlvdXIgb3JnYW5pemF0aW9uJ3MgYWNjb3VudHMuXG4gICAqL1xuICBCQUNLVVBfUE9MSUNZID0gXCJCQUNLVVBfUE9MSUNZXCIsXG4gIC8qKlxuICAgKiBBcnRpZmljaWFsIEludGVsbGlnZW5jZSAoQUkpIHNlcnZpY2VzIG9wdC1vdXQgcG9saWNpZXMgZW5hYmxlIHlvdSB0byBjb250cm9sIGRhdGEgY29sbGVjdGlvbiBmb3IgQVdTIEFJIHNlcnZpY2VzIGZvciBhbGwgb2YgeW91ciBvcmdhbml6YXRpb24ncyBhY2NvdW50cy5cbiAgICovXG4gIEFJU0VSVklDRVNfT1BUX09VVF9QT0xJQ1kgPSBcIkFJU0VSVklDRVNfT1BUX09VVF9QT0xJQ1lcIixcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQb2xpY3lQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgcG9saWN5IHRleHQgY29udGVudCB0byBhZGQgdG8gdGhlIG5ldyBwb2xpY3kuIFRoZSB0ZXh0IHRoYXQgeW91IHN1cHBseSBtdXN0IGFkaGVyZSB0byB0aGUgcnVsZXMgb2YgdGhlIHBvbGljeSB0eXBlIHlvdSBzcGVjaWZ5IGluIHRoZSBUeXBlIHBhcmFtZXRlci5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnRlbnQ6IHN0cmluZztcbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIGRlc2NyaXB0aW9uIHRvIGFzc2lnbiB0byB0aGUgcG9saWN5LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZnJpZW5kbHkgbmFtZSB0byBhc3NpZ24gdG8gdGhlIHBvbGljeS5cbiAgICovXG4gIHJlYWRvbmx5IHBvbGljeU5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIHBvbGljeSB0byBjcmVhdGUuIFlvdSBjYW4gc3BlY2lmeSBvbmUgb2YgdGhlIGZvbGxvd2luZyB2YWx1ZXM6XG4gICAqL1xuICByZWFkb25seSBwb2xpY3lUeXBlOiBQb2xpY3lUeXBlO1xufVxuXG4vKipcbiAqIFBvbGljaWVzIGluIEFXUyBPcmdhbml6YXRpb25zIGVuYWJsZSB5b3UgdG8gYXBwbHkgYWRkaXRpb25hbCB0eXBlcyBvZiBtYW5hZ2VtZW50IHRvIHRoZSBBV1MgYWNjb3VudHMgaW4geW91ciBvcmdhbml6YXRpb24uIDxzdHJvbmc+WW91IGNhbiB1c2UgcG9saWNpZXMgd2hlbiBhbGwgZmVhdHVyZXMgYXJlIGVuYWJsZWQgaW4geW91ciBvcmdhbml6YXRpb24uPC9zdHJvbmc+XG4gKlxuICogPHN0cm9uZz5CZWZvcmUgeW91IGNhbiBjcmVhdGUgYW5kIGF0dGFjaCBhIHBvbGljeSB0byB5b3VyIG9yZ2FuaXphdGlvbiwgeW91IG11c3QgZW5hYmxlIHRoYXQgcG9saWN5IHR5cGUgZm9yIHVzZS48L3N0cm9uZz5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcmdhbml6YXRpb25zL2xhdGVzdC91c2VyZ3VpZGUvb3Jnc19tYW5hZ2VfcG9saWNpZXMuaHRtbFxuICogQHNlZSBGZWF0dXJlU2V0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVBvbGljeSBleHRlbmRzIElDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogVGhlIHVuaXF1ZSBpZGVudGlmaWVyIChJRCkgb2YgdGhlIHBvbGljeS4gVGhlIHJlZ2V4IHBhdHRlcm4gZm9yIGEgcG9saWN5IElEIHN0cmluZyByZXF1aXJlcyBcInAtXCIgZm9sbG93ZWQgYnkgZnJvbSA4IHRvIDEyOCBsb3dlcmNhc2Ugb3IgdXBwZXJjYXNlIGxldHRlcnMsIGRpZ2l0cywgb3IgdGhlIHVuZGVyc2NvcmUgY2hhcmFjdGVyIChfKS5cbiAgICovXG4gIHJlYWRvbmx5IHBvbGljeUlkOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBjbGFzcyBQb2xpY3kgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJUG9saWN5LCBJVGFnZ2FibGVSZXNvdXJjZSB7XG4gIHB1YmxpYyByZWFkb25seSBwb2xpY3lJZDogc3RyaW5nO1xuXG4gIHJlYWRvbmx5IHRhZ3MgPSBuZXcgVGFnTWFuYWdlcihUYWdUeXBlLktFWV9WQUxVRSwgXCJDdXN0b206Ok9yZ2FuaXphdGlvbnNfUG9saWN5XCIpO1xuXG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUG9saWN5UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgeyBjb250ZW50LCBkZXNjcmlwdGlvbiwgcG9saWN5TmFtZSwgcG9saWN5VHlwZSB9ID0gcHJvcHM7XG5cbiAgICBpZiAoIVZhbGlkYXRvcnMub2YoKS5wb2xpY3lDb250ZW50KGNvbnRlbnQpKSB7XG4gICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRFcnJvcihcbiAgICAgICAgXCJUaGUgdGV4dCBjb250ZW50IG9mIHRoZSBwb2xpY3kgbXVzdCBiZSB2YWxpZCBhbmQgYmV0d2VlbiAxIGFuZCAxLDAwMCwwMDAgY2hhcmFjdGVycyBsb25nLlwiXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IHBvbGljeSA9IG5ldyBBd3NDdXN0b21SZXNvdXJjZSh0aGlzLCBcIlBvbGljeUN1c3RvbVJlc291cmNlXCIsIHtcbiAgICAgIHJlc291cmNlVHlwZTogXCJDdXN0b206Ok9yZ2FuaXphdGlvbnNfUG9saWN5XCIsXG4gICAgICBvbkNyZWF0ZToge1xuICAgICAgICBzZXJ2aWNlOiBcIk9yZ2FuaXphdGlvbnNcIixcbiAgICAgICAgYWN0aW9uOiBcImNyZWF0ZVBvbGljeVwiLCAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTSmF2YVNjcmlwdFNESy9sYXRlc3QvQVdTL09yZ2FuaXphdGlvbnMuaHRtbCNjcmVhdGVQb2xpY3ktcHJvcGVydHlcbiAgICAgICAgcmVnaW9uOiBcInVzLWVhc3QtMVwiLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgQ29udGVudDogY29udGVudCxcbiAgICAgICAgICBEZXNjcmlwdGlvbjogZGVzY3JpcHRpb24sXG4gICAgICAgICAgTmFtZTogcG9saWN5TmFtZSxcbiAgICAgICAgICBUeXBlOiBwb2xpY3lUeXBlLFxuICAgICAgICB9LFxuICAgICAgICBvdXRwdXRQYXRoczogW1wiUG9saWN5LlBvbGljeVN1bW1hcnkuSWRcIl0sXG4gICAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogUGh5c2ljYWxSZXNvdXJjZUlkLmZyb21SZXNwb25zZShcIlBvbGljeS5Qb2xpY3lTdW1tYXJ5LklkXCIpLFxuICAgICAgfSxcbiAgICAgIG9uVXBkYXRlOiB7XG4gICAgICAgIHNlcnZpY2U6IFwiT3JnYW5pemF0aW9uc1wiLFxuICAgICAgICBhY3Rpb246IFwidXBkYXRlUG9saWN5XCIsIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9BV1MvT3JnYW5pemF0aW9ucy5odG1sI3VwZGF0ZVBvbGljeS1wcm9wZXJ0eVxuICAgICAgICByZWdpb246IFwidXMtZWFzdC0xXCIsXG4gICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBDb250ZW50OiBjb250ZW50LFxuICAgICAgICAgIERlc2NyaXB0aW9uOiBkZXNjcmlwdGlvbixcbiAgICAgICAgICBOYW1lOiBwb2xpY3lOYW1lLFxuICAgICAgICAgIFBvbGljeUlkOiBuZXcgUGh5c2ljYWxSZXNvdXJjZUlkUmVmZXJlbmNlKCksXG4gICAgICAgIH0sXG4gICAgICAgIG91dHB1dFBhdGhzOiBbXCJQb2xpY3kuUG9saWN5U3VtbWFyeS5JZFwiXSxcbiAgICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQuZnJvbVJlc3BvbnNlKFwiUG9saWN5LlBvbGljeVN1bW1hcnkuSWRcIiksXG4gICAgICB9LFxuICAgICAgb25EZWxldGU6IHtcbiAgICAgICAgc2VydmljZTogXCJPcmdhbml6YXRpb25zXCIsXG4gICAgICAgIGFjdGlvbjogXCJkZWxldGVQb2xpY3lcIiwgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0phdmFTY3JpcHRTREsvbGF0ZXN0L0FXUy9Pcmdhbml6YXRpb25zLmh0bWwjZGVsZXRlUG9saWN5LXByb3BlcnR5XG4gICAgICAgIHJlZ2lvbjogXCJ1cy1lYXN0LTFcIixcbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgIFBvbGljeUlkOiBuZXcgUGh5c2ljYWxSZXNvdXJjZUlkUmVmZXJlbmNlKCksXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgaW5zdGFsbExhdGVzdEF3c1NkazogZmFsc2UsXG4gICAgICBwb2xpY3k6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LmZyb21TZGtDYWxscyh7IHJlc291cmNlczogQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3kuQU5ZX1JFU09VUkNFIH0pLFxuICAgIH0pO1xuICAgIHRoaXMucG9saWN5SWQgPSBwb2xpY3kuZ2V0UmVzcG9uc2VGaWVsZChcIlBvbGljeS5Qb2xpY3lTdW1tYXJ5LklkXCIpO1xuXG4gICAgY29uc3QgdGFnUmVzb3VyY2UgPSBuZXcgVGFnUmVzb3VyY2UodGhpcywgXCJUYWdzXCIsIHsgcmVzb3VyY2VJZDogdGhpcy5wb2xpY3lJZCwgdGFnczogdGhpcy50YWdzLnJlbmRlcmVkVGFncyB9KTtcbiAgICB0YWdSZXNvdXJjZS5ub2RlLmFkZERlcGVuZGVuY3kocG9saWN5KTtcbiAgfVxuXG4gIGlkZW50aWZpZXIoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5wb2xpY3lJZDtcbiAgfVxufVxuIl19