"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_sdk_1 = require("aws-sdk");
let organizationsClient;
/**
 * The onEvent handler is invoked whenever a resource lifecycle event for a TagResource occurs
 *
 * @see https://docs.aws.amazon.com/cdk/api/v1/docs/custom-resources-readme.html#handling-lifecycle-events-onevent
 */
async function handler(event) {
    console.log(`Request of type ${event.RequestType} received`);
    if (!organizationsClient) {
        organizationsClient = new aws_sdk_1.Organizations({ region: "us-east-1" });
    }
    console.log("Payload: %j", event);
    // Get all AWS organizations service tags
    const listTagsForResourceResponse = await organizationsClient
        .listTagsForResource({
        ResourceId: event.ResourceProperties.ResourceId,
    })
        .promise();
    const oldTags = listTagsForResourceResponse.Tags ?? [];
    const oldTagKeys = oldTags.map((tag) => tag.Key);
    const newTags = event.ResourceProperties.Tags ?? [];
    const newTagKeys = newTags.map((tag) => tag.Key);
    // Remove AWS organizations service tags
    const tagKeysToRemove = oldTagKeys.filter((tagKey) => !newTagKeys.includes(tagKey));
    if (tagKeysToRemove.length) {
        await organizationsClient
            .untagResource({
            ResourceId: event.ResourceProperties.ResourceId,
            TagKeys: tagKeysToRemove,
        })
            .promise();
    }
    if (event.RequestType == "Delete") {
        return { PhysicalResourceId: event.PhysicalResourceId };
    }
    if (newTags.length) {
        // Update AWS organizations service tags
        await organizationsClient
            .tagResource({
            ResourceId: event.ResourceProperties.ResourceId,
            Tags: newTags,
        })
            .promise();
    }
    return { PhysicalResourceId: event.ResourceProperties.ResourceId, ResourceProperties: event.ResourceProperties };
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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