//! Types specific for modeling-app projects.

use anyhow::Result;
use indexmap::IndexMap;
use kittycad_modeling_cmds::units::UnitLength;
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use validator::Validate;

use crate::settings::types::{
    AppColor, DefaultTrue, OnboardingStatus, ProjectCommandBarSettings, ProjectTextEditorSettings, is_default,
};

/// Project specific settings for the app.
/// These live in `project.toml` in the base of the project directory.
/// Updating the settings for the project in the app will update this file automatically.
/// Do not edit this file manually, as it may be overwritten by the app.
/// Manual edits can cause corruption of the settings file.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct ProjectConfiguration {
    /// The settings for the project.
    #[serde(default)]
    #[validate(nested)]
    pub settings: PerProjectSettings,
}

impl ProjectConfiguration {
    // TODO: remove this when we remove backwards compatibility with the old settings file.
    pub fn parse_and_validate(toml_str: &str) -> Result<Self> {
        let settings = toml::from_str::<Self>(toml_str)?;

        settings.validate()?;

        Ok(settings)
    }
}

/// High level project settings.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct PerProjectSettings {
    /// Information about the project itself.
    /// Choices about how settings are merged have prevent me (lee) from easily
    /// moving this out of the settings structure.
    #[serde(default)]
    #[validate(nested)]
    pub meta: ProjectMetaSettings,

    /// The settings for the Design Studio.
    #[serde(default)]
    #[validate(nested)]
    pub app: ProjectAppSettings,
    /// Settings that affect the behavior while modeling.
    #[serde(default)]
    #[validate(nested)]
    pub modeling: ProjectModelingSettings,
    /// Settings that affect the behavior of the KCL text editor.
    #[serde(default)]
    #[validate(nested)]
    pub text_editor: ProjectTextEditorSettings,
    /// Settings that affect the behavior of the command bar.
    #[serde(default)]
    #[validate(nested)]
    pub command_bar: ProjectCommandBarSettings,
}

/// Information about the project.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct ProjectMetaSettings {
    #[serde(default, skip_serializing_if = "is_default")]
    pub id: uuid::Uuid,
}

/// Project specific application settings.
// TODO: When we remove backwards compatibility with the old settings file, we can remove the
// aliases to camelCase (and projects plural) from everywhere.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct ProjectAppSettings {
    /// The settings for the appearance of the app.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub appearance: ProjectAppearanceSettings,
    /// The onboarding status of the app.
    #[serde(default, skip_serializing_if = "is_default")]
    pub onboarding_status: OnboardingStatus,
    /// Permanently dismiss the banner warning to download the desktop app.
    /// This setting only applies to the web app. And is temporary until we have Linux support.
    #[serde(default, skip_serializing_if = "is_default")]
    pub dismiss_web_banner: bool,
    /// When the user is idle, and this is true, the stream will be torn down.
    #[serde(default, skip_serializing_if = "is_default")]
    pub stream_idle_mode: bool,
    /// When the user is idle, and this is true, the stream will be torn down.
    #[serde(default, skip_serializing_if = "is_default")]
    pub allow_orbit_in_sketch_mode: bool,
    /// Whether to show the debug panel, which lets you see various states
    /// of the app to aid in development.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub show_debug_panel: Option<bool>,
    /// Settings that affect the behavior of the command bar.
    #[serde(default, skip_serializing_if = "IndexMap::is_empty")]
    pub named_views: IndexMap<uuid::Uuid, NamedView>,
}

/// Project specific appearance settings.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct ProjectAppearanceSettings {
    /// The hue of the primary theme color for the app.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub color: AppColor,
}

/// Project specific settings that affect the behavior while modeling.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[serde(rename_all = "snake_case")]
#[ts(export)]
pub struct ProjectModelingSettings {
    /// The default unit to use in modeling dimensions.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub base_unit: Option<UnitLength>,
    /// Highlight edges of 3D objects?
    #[serde(default, skip_serializing_if = "is_default")]
    pub highlight_edges: DefaultTrue,
    /// Whether or not Screen Space Ambient Occlusion (SSAO) is enabled.
    #[serde(default, skip_serializing_if = "is_default")]
    pub enable_ssao: DefaultTrue,
    /// When enabled, the grid will use a fixed size based on your selected units rather than automatically scaling with zoom level.
    /// If true, the grid cells will be fixed-size, where the width is your default length unit.
    /// If false, the grid will get larger as you zoom out, and smaller as you zoom in.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub fixed_size_grid: Option<bool>,
    /// When enabled, tools like line, rectangle, etc. will snap to the grid.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub snap_to_grid: Option<bool>,
    /// The space between major grid lines, specified in the current unit.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub major_grid_spacing: Option<f64>,
    /// The number of minor grid lines per major grid line.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub minor_grids_per_major: Option<f64>,
    /// The number of snaps between minor grid lines. 1 means snapping to each minor grid line.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub snaps_per_minor: Option<f64>,
}

fn named_view_point_version_one() -> f64 {
    1.0
}

#[derive(Debug, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq)]
#[serde(rename_all = "snake_case")]
#[ts(export)]
pub struct NamedView {
    /// User defined name to identify the named view. A label.
    #[serde(default)]
    pub name: String,
    /// Engine camera eye off set
    #[serde(default)]
    pub eye_offset: f64,
    /// Engine camera vertical FOV
    #[serde(default)]
    pub fov_y: f64,
    // Engine camera is orthographic or perspective projection
    #[serde(default)]
    pub is_ortho: bool,
    /// Engine camera is orthographic camera scaling enabled
    #[serde(default)]
    pub ortho_scale_enabled: bool,
    /// Engine camera orthographic scaling factor
    #[serde(default)]
    pub ortho_scale_factor: f64,
    /// Engine camera position that the camera pivots around
    #[serde(default)]
    pub pivot_position: [f64; 3],
    /// Engine camera orientation in relation to the pivot position
    #[serde(default)]
    pub pivot_rotation: [f64; 4],
    /// Engine camera world coordinate system orientation
    #[serde(default)]
    pub world_coord_system: String,
    /// Version number of the view point if the engine camera API changes
    #[serde(default = "named_view_point_version_one")]
    pub version: f64,
}

#[cfg(test)]
mod tests {
    use indexmap::IndexMap;
    use pretty_assertions::assert_eq;
    use serde_json::Value;

    use super::{
        NamedView, PerProjectSettings, ProjectAppSettings, ProjectAppearanceSettings, ProjectCommandBarSettings,
        ProjectConfiguration, ProjectMetaSettings, ProjectModelingSettings, ProjectTextEditorSettings,
    };
    use crate::settings::types::UnitLength;

    #[test]
    fn test_project_settings_empty_file_parses() {
        let empty_settings_file = r#""#;

        let parsed = toml::from_str::<ProjectConfiguration>(empty_settings_file).unwrap();
        assert_eq!(parsed, ProjectConfiguration::default());

        // Write the file back out.
        let serialized = toml::to_string(&parsed).unwrap();
        assert_eq!(
            serialized,
            r#"[settings.meta]

[settings.app]

[settings.modeling]

[settings.text_editor]

[settings.command_bar]
"#
        );

        let parsed = ProjectConfiguration::parse_and_validate(empty_settings_file).unwrap();
        assert_eq!(parsed, ProjectConfiguration::default());
    }

    #[test]
    fn test_project_settings_color_validation_error() {
        let settings_file = r#"[settings.app.appearance]
color = 1567.4"#;

        let result = ProjectConfiguration::parse_and_validate(settings_file);
        if let Ok(r) = result {
            panic!("Expected an error, but got success: {r:?}");
        }
        assert!(result.is_err());

        assert!(
            result
                .unwrap_err()
                .to_string()
                .contains("color: Validation error: color")
        );
    }

    #[test]
    fn named_view_serde_json() {
        let json = r#"
        [
          {
            "name":"dog",
            "pivot_rotation":[0.53809947,0.0,0.0,0.8428814],
            "pivot_position":[0.5,0,0.5],
            "eye_offset":231.52048,
            "fov_y":45,
            "ortho_scale_factor":1.574129,
            "is_ortho":true,
            "ortho_scale_enabled":true,
            "world_coord_system":"RightHandedUpZ"
          }
    ]
    "#;
        // serde_json to a NamedView will produce default values
        let named_views: Vec<NamedView> = serde_json::from_str(json).unwrap();
        let version = named_views[0].version;
        assert_eq!(version, 1.0);
    }

    #[test]
    fn named_view_serde_json_string() {
        let json = r#"
        [
          {
            "name":"dog",
            "pivot_rotation":[0.53809947,0.0,0.0,0.8428814],
            "pivot_position":[0.5,0,0.5],
            "eye_offset":231.52048,
            "fov_y":45,
            "ortho_scale_factor":1.574129,
            "is_ortho":true,
            "ortho_scale_enabled":true,
            "world_coord_system":"RightHandedUpZ"
          }
    ]
    "#;

        // serde_json to string does not produce default values
        let named_views: Value = match serde_json::from_str(json) {
            Ok(x) => x,
            Err(_) => return,
        };
        println!("{}", named_views);
    }

    #[test]
    fn test_project_settings_named_views() {
        let conf = ProjectConfiguration {
            settings: PerProjectSettings {
                meta: ProjectMetaSettings { id: uuid::Uuid::nil() },
                app: ProjectAppSettings {
                    appearance: ProjectAppearanceSettings { color: 138.0.into() },
                    onboarding_status: Default::default(),
                    dismiss_web_banner: false,
                    stream_idle_mode: false,
                    allow_orbit_in_sketch_mode: false,
                    show_debug_panel: Some(true),
                    named_views: IndexMap::from([
                        (
                            uuid::uuid!("323611ea-66e3-43c9-9d0d-1091ba92948c"),
                            NamedView {
                                name: String::from("Hello"),
                                eye_offset: 1236.4015,
                                fov_y: 45.0,
                                is_ortho: false,
                                ortho_scale_enabled: false,
                                ortho_scale_factor: 45.0,
                                pivot_position: [-100.0, 100.0, 100.0],
                                pivot_rotation: [-0.16391756, 0.9862819, -0.01956843, 0.0032552152],
                                world_coord_system: String::from("RightHandedUpZ"),
                                version: 1.0,
                            },
                        ),
                        (
                            uuid::uuid!("423611ea-66e3-43c9-9d0d-1091ba92948c"),
                            NamedView {
                                name: String::from("Goodbye"),
                                eye_offset: 1236.4015,
                                fov_y: 45.0,
                                is_ortho: false,
                                ortho_scale_enabled: false,
                                ortho_scale_factor: 45.0,
                                pivot_position: [-100.0, 100.0, 100.0],
                                pivot_rotation: [-0.16391756, 0.9862819, -0.01956843, 0.0032552152],
                                world_coord_system: String::from("RightHandedUpZ"),
                                version: 1.0,
                            },
                        ),
                    ]),
                },
                modeling: ProjectModelingSettings {
                    base_unit: Some(UnitLength::Yards),
                    highlight_edges: Default::default(),
                    enable_ssao: true.into(),
                    snap_to_grid: None,
                    major_grid_spacing: None,
                    minor_grids_per_major: None,
                    snaps_per_minor: None,
                    fixed_size_grid: None,
                },
                text_editor: ProjectTextEditorSettings {
                    text_wrapping: Some(false),
                    blinking_cursor: Some(false),
                },
                command_bar: ProjectCommandBarSettings {
                    include_settings: Some(false),
                },
            },
        };
        let serialized = toml::to_string(&conf).unwrap();
        let old_project_file = r#"[settings.meta]

[settings.app]
show_debug_panel = true

[settings.app.appearance]
color = 138.0

[settings.app.named_views.323611ea-66e3-43c9-9d0d-1091ba92948c]
name = "Hello"
eye_offset = 1236.4015
fov_y = 45.0
is_ortho = false
ortho_scale_enabled = false
ortho_scale_factor = 45.0
pivot_position = [-100.0, 100.0, 100.0]
pivot_rotation = [-0.16391756, 0.9862819, -0.01956843, 0.0032552152]
world_coord_system = "RightHandedUpZ"
version = 1.0

[settings.app.named_views.423611ea-66e3-43c9-9d0d-1091ba92948c]
name = "Goodbye"
eye_offset = 1236.4015
fov_y = 45.0
is_ortho = false
ortho_scale_enabled = false
ortho_scale_factor = 45.0
pivot_position = [-100.0, 100.0, 100.0]
pivot_rotation = [-0.16391756, 0.9862819, -0.01956843, 0.0032552152]
world_coord_system = "RightHandedUpZ"
version = 1.0

[settings.modeling]
base_unit = "yd"

[settings.text_editor]
text_wrapping = false
blinking_cursor = false

[settings.command_bar]
include_settings = false
"#;

        assert_eq!(serialized, old_project_file)
    }
}
