import logging
from typing import Any

import neo4j
import scaleway
from scaleway.iam.v1alpha1 import IamV1Alpha1API
from scaleway.iam.v1alpha1 import User

from cartography.client.core.tx import load
from cartography.graph.job import GraphJob
from cartography.intel.scaleway.utils import scaleway_obj_to_dict
from cartography.models.scaleway.iam.user import ScalewayUserSchema
from cartography.util import timeit

logger = logging.getLogger(__name__)


@timeit
def sync(
    neo4j_session: neo4j.Session,
    client: scaleway.Client,
    common_job_parameters: dict[str, Any],
    org_id: str,
    update_tag: int,
) -> None:
    users = get(client, org_id)
    formatted_users = transform_users(users)
    load_users(neo4j_session, formatted_users, org_id, update_tag)
    cleanup(neo4j_session, common_job_parameters)


@timeit
def get(
    client: scaleway.Client,
    org_id: str,
) -> list[User]:
    api = IamV1Alpha1API(client)
    return api.list_users_all(organization_id=org_id)


def transform_users(users: list[User]) -> list[dict[str, Any]]:
    formatted_users = []
    for user in users:
        formatted_users.append(scaleway_obj_to_dict(user))
    return formatted_users


@timeit
def load_users(
    neo4j_session: neo4j.Session,
    data: list[dict[str, Any]],
    org_id: str,
    update_tag: int,
) -> None:
    logger.info("Loading %d Scaleway Users into Neo4j.", len(data))
    load(
        neo4j_session,
        ScalewayUserSchema(),
        data,
        lastupdated=update_tag,
        ORG_ID=org_id,
    )


@timeit
def cleanup(
    neo4j_session: neo4j.Session, common_job_parameters: dict[str, Any]
) -> None:
    GraphJob.from_node_schema(ScalewayUserSchema(), common_job_parameters).run(
        neo4j_session
    )
