# -*- coding: utf-8 -*-

import numpy as np
from logging import getLogger, StreamHandler, Formatter, DEBUG

logger = getLogger(__name__)
handler = StreamHandler()
handler.setFormatter(Formatter('[%(asctime)s %(levelname)s %(module)s] %(message)s'))
logger.addHandler(handler)
np.set_printoptions(precision=3, suppress=True, linewidth=np.inf)


class OnsetTransform():
    """
    Transform module for onsets detection.

    Parameters
    ----

    keyword arguments
    ````

    th_power : int
       Threshold of power diff between neighbored frames to regart as onset

    th_time_diff : int
       Threshold of time diff between neighbored onsets to regart as same onset
    """

    def __init__(self, bufsize, framerate, channels, verbose=False, *args, **kwargs):
        self.bufsize = bufsize
        self.framerate = framerate
        self.channels = channels
        if verbose:
            logger.setLevel(DEBUG)
            handler.setLevel(DEBUG)

        self.th_power = kwargs.pop('th_power', None)
        if self.th_power is None:
            raise ValueError('"--th_power" option is required')

        self.th_power = float(self.th_power)
        self.th_time_diff = int(kwargs.pop('th_time_diff', 100))

        self.ms_per_buf = None
        self.old_data = None
        self.total_bufsize = 0
        self.last_onset = 0

    def transform(self, stfted_matrix):
        """
        Transform from STFTed formed 2-D matrix to array of onset detected times.

        Parameters
        ---
        stfted_matrix : 2-D matrix
            Target data to transform generated by STFT

        Returns
        ---
        Array of onset times in milliseconds
        """

        # transform from STFTed matrix to power spectrogram
        p_spec = np.abs(stfted_matrix) ** 2

        if self.old_data is None:
            merged_matrix = stfted_matrix
            self.ms_per_buf = (self.bufsize / self.framerate) * 1000
        else:
            merged_matrix = np.append(self.old_data, p_spec, axis=0)

        # power diff by subtracting unshifted frames from current frames
        pdiff = np.abs(merged_matrix[1:] - merged_matrix[:-1])
        pdiff_means = np.mean(pdiff, axis=1)

        logger.info('means of power is: {}'
                    .format(np.mean(p_spec, axis=1)))
        logger.info('means of power difference with neighbor frames is: {}'
                    .format(pdiff_means))

        # filter onsets and convert bools by threshold of power
        is_onsets = pdiff_means > self.th_power

        if is_onsets.any():
            logger.debug('try to filter frames which are over th_power by th_time_diff...')
            onset_times = (np.where(is_onsets)[0] * (self.ms_per_buf / stfted_matrix.shape[0]) +
                           self.total_bufsize / self.framerate * 1000).astype(int)
            tdiff = np.diff(np.append(self.last_onset, onset_times))
            onsets = onset_times[tdiff > self.th_time_diff]
            self.last_onset = onset_times[-1]

        else:
            onsets = []

        self.total_bufsize += self.bufsize
        self.old_data = p_spec[-1:]

        if len(onsets) > 0:
            logger.info('onset detected at: {}'.format(onsets))

        return onsets
