# -*- coding: utf-8 -*-

import os.path
import wave
import numpy as np
from numpy import hamming
from nmftools import nmf, ssnmf
from logging import getLogger, StreamHandler, Formatter, DEBUG

from ..utils import stft
from ..utils import string_to_int

logger = getLogger(__name__)
handler = StreamHandler()
handler.setFormatter(Formatter('[%(asctime)s %(levelname)s %(module)s] %(message)s'))
logger.addHandler(handler)


class SsnmfTransform():
    """
    Transform module for SSNMF.

    Parameters
    ----

    keyword arguments
    ````

    supervise_filepath : string (required)
          Path of WAV file to use supervise.

    n_fft : int or None
        FFT window size to apply supervise data. Default is 1024.

    n_hop : int or None
        STFT hop length to apply supervise data. Default is 256

    n_t_comp : int or None
        Number of components to decompose without supervise data by SSNMF.
        Default is 3.

    n_s_comp : int or None
        Number of components to decompose only supervise data by SSNMF. Default is 3.

    n_t_iter : int or None
        Number of iteration to iterate SSNMF. Default is 50.

    n_s_iter : int or None
        Number of iteration to iterate NMF to supervise data. Default is 50.
    """

    def __init__(self, bufsize, framerate, channels, verbose=False, *args, **kwargs):
        self.bufsize = bufsize
        self.framerate = framerate
        self.channels = channels
        if verbose:
            logger.setLevel(DEBUG)
            handler.setLevel(DEBUG)

        sup_filepath = kwargs.get('supervise_filepath', None)
        self.n_fft = int(kwargs.get('n_fft', 1024))
        self.n_hop = int(kwargs.get('n_hop', 256))
        self.n_t_comp = int(kwargs.get('n_t_comp', 3))
        self.n_s_comp = int(kwargs.get('n_s_comp', 3))
        self.n_t_iter = int(kwargs.get('n_t_iter', 50))
        self.n_s_iter = int(kwargs.get('n_s_iter', 50))

        # generate base component of supervise data
        if sup_filepath is None:
            raise ValueError('"--supervise_filepath" option is required')
        if not os.path.isfile(sup_filepath):
            raise IOError('file not found in "{}"'.format(sup_filepath))

        # transform supervise data: read => int => STFT => Spectrogram => NMF
        wf = wave.open(sup_filepath, 'rb')
        sup_y = string_to_int(wf.readframes(wf.getnframes()),
                              wf.getnchannels())

        stfted = stft(sup_y, self.n_fft, self.n_hop, hamming)

        # gen spectrogram from STFTed matrix
        sup_spec = np.abs(stfted.T)

        # preserve base component in module-global
        logger.debug('try to decompose supervise data with NMF...')
        self.sup_base, _, _ = nmf(sup_spec, R=self.n_s_comp, n_iter=self.n_s_iter)
        logger.info('base components were generated from supervise data')

    def transform(self, stfted_matrix):
        """
        Transform stream data which formed stfted matrix to decomposed components.

        Parameters
        ----
        stfted_matrix : 2-D matrix
            Target data to transform generated by STFT

        Returns
        ----
        Remixed STFTef 2-D matrix, which remixed by only base components generated from
        supervise data by NMF.
        """

        # gen spectrogram from STFTed matrix
        spec = np.abs(stfted_matrix).T

        logger.debug(('try to decompose spectrogram {}-matrix by SSNMF ' +
                      'with supervised base components {}-matrix...')
                     .format(spec.shape, self.sup_base.shape))

        decomposed = ssnmf(spec, F=self.sup_base, R=self.n_t_comp, n_iter=self.n_t_iter)

        logger.info(('decomposed STFTed matrix by SSNMF to base components and activation ' +
                     'by SSNMF. These shapes is F:{}, G:{}, H:{}, U:{}, final cost is {}')
                    .format(decomposed[0].shape, decomposed[1].shape,
                            decomposed[2].shape, decomposed[3].shape, decomposed[4][-1]))

        angle = np.angle(stfted_matrix.T)
        remixed = np.dot(decomposed[0], decomposed[1]) * (np.cos(angle) + 1j * np.sin(angle))
        logger.info('remixed STFTed formed {}-matrix from decomposed matrices'.format(remixed.shape))  # NOQA

        return remixed.T
