from pathlib import Path
from typing import Union

import xarray as xr

# Import the modules used
from amocarray import logger, utilities
from amocarray.logger import log_error, log_info, log_warning
from amocarray.utilities import apply_defaults

log = logger.log  # Use the global logger
#Denmark Strait Overflow
# Default source and file list DSO

DSO_DEFAULT_SOURCE = "https://icdc.cen.uni-hamburg.de/thredds/fileServer/ftpthredds/dso_transport/"
DSO_DEFAULT_FILES = ["DSO_transport_hourly_1996_2021.nc"]
DSO_TRANSPORT_FILES = ["DSO_transport_hourly_1996_2021.nc"]


DSO_METADATA = {"project": "Overflow time-series through Denmark Strait",
                "weblink": "https://www.cen.uni-hamburg.de/en/icdc/data/ocean/denmark-strait-overflow.html",
                "comment": "Dataset accessed and processed via http://github.com/AMOCcommunity/amocarray",
                "acknowledgement": "The timeseries was generated by Institution of Oceanography Hamburg and Hafrannsóknastofnun / Marine and Freshwater Research Institute (Reykjavik, Iceland). They were supported through funding from the NACLIM, EU-FP7, grant agr. n.308299, until 2016, and from RACE II (Förderkennzeichen 03F0729B, until 2018), RACE-Synthese (Förderkennzeichen 03F0825B, until 2020) German Federal Ministry for Education and Research (BMBF). Nordic WOCE, VEINS, MOEN (contract no. EVK2-CT-2002-00141), ASOF-W (contract no. EVK2-CT-2002-00149), NAClim (grant agr. nr. 308299) THOR (grant agr. nr. 212643), AtlantOS, Blue Action. Collaborative Research Centre TRR 181 Energy Transfers in Atmosphere and Ocean funded by the Deutsche Forschungsgemeinschaft (DFG, German Research Foundation) - Project number 274762653. Thanks to ICDC, CEN, University of Hamburg for data support.",
                "doi": "doi:10.1002/2017JC012803",
                "paper": "Jochumsen, K., M. Moritz, N. Nunes, D. Quadfasel, K. M. Larsen, B. Hansen, H. Valdimarsson and S. Jonsson (2017): Revised transport estimates of the Denmark Strait Overflow, Journal of Geophysical Research J. Geophys. Res., 122, 3434-3450, doi:10.1002/2017JC012803.",
}

DSO_FILE_METADATA = {
        "DSO_transport_hourly_1996_2021.nc": {
            "data_product": "Overflow time-series through Denmark Strait",
    },
        
}

@apply_defaults(DSO_DEFAULT_SOURCE, DSO_DEFAULT_FILES)
def read_dso(
    source: str,
##    source: Union[str, Path, None],
    file_list: Union[str, list[str]],
    transport_only: bool = True,
    data_dir: Union[str, Path, None] = None,
    redownload: bool = False,
) -> list[xr.Dataset]:
    
    """Load the Denmark Strait Overflow (DSO) datasets from a URL or local file path into xarray Datasets.
    Parameters
    ----------
    source : str, optional
        Local path to the data directory (remote source is handled per-file).
    file_list : str or list of str, optional
        Filename or list of filenames to process.
        Defaults to DSO_DEFAULT_FILES.
    transport_only : bool, optional
        If True, restrict to transport files only.
    data_dir : str, Path or None, optional
        Optional local data directory.
    redownload : bool, optional
        If True, force redownload of the data.

    Returns
    -------                                                         list of xr.Dataset
        List of loaded xarray datasets with basic inline and file-specific metadata.

    Raises
    ------
    ValueError
    If no source is provided for a file and no default URL mapping is found.
    FileNotFoundError                                                   If the file cannot be downloaded or does not exist locally.
    """
    log.info("Starting to read DSO dataset") # Ensure file_list has a default
    if file_list is None:
        file_list = DSO_DEFAULT_FILES
    if transport_only:
        file_list = DSO_TRANSPORT_FILES
    if isinstance(file_list, str):
        file_list = [file_list]
    # Determine the local storage path
    local_data_dir = Path(data_dir) if data_dir else utilities.get_default_data_dir()
    local_data_dir.mkdir(parents=True, exist_ok=True)

    datasets = []

    for file in file_list:
        if not file.lower().endswith(".nc"):
            log_warning("Skipping non-NetCDF file: %s", file)
            continue

        download_url = (
            f"{source.rstrip('/')}/{file}" if utilities._is_valid_url(source) else None
        )

        file_path = utilities.resolve_file_path(
            file_name=file,
            source=source,
            download_url=download_url,
            local_data_dir=local_data_dir,
            redownload=redownload,
        )

        try:
            log_info("Opening DSO dataset: %s", file_path)
            ds = xr.open_dataset(file_path)
        except Exception as e:
            log_error("Failed to open NetCDF file: %s: %s", file_path, e)
            raise FileNotFoundError(f"Failed to open NetCDF file: {file_path}: {e}")

        file_metadata = DSO_FILE_METADATA.get(file, {})
        log_info("Attaching metadata to DSO dataset from file: %s", file)

        utilities.safe_update_attrs(
            ds,
            {
                "source_file": file,
                "source_path": str(file_path),
                **DSO_METADATA,
                **file_metadata,
            },
        )

        datasets.append(ds)

    if not datasets:
        log_error("No valid DSO NetCDF files found in %s", file_list)
        raise FileNotFoundError(f"No valid DSO NetCDF files found in {file_list}")
    log_info("Successfully loaded %d DSO dataset(s)", len(datasets))
    return datasets
