from fastapi import UploadFile
from pydantic import BaseModel, Field, PrivateAttr
from typing import Annotated, Generic, Self, TypeVar
from maleo.types.integer import OptInt
from maleo.types.string import SeqOfStrs


class Document(BaseModel):
    _raw: UploadFile | None = PrivateAttr(None)
    content: Annotated[bytes, Field(..., description="Content")]
    content_type: Annotated[str, Field(..., description="Content Type")]
    filename: Annotated[str, Field(..., description="Filename")]
    size: Annotated[int, Field(..., description="Size", gt=0)]

    @classmethod
    async def from_file(
        cls,
        file: UploadFile,
        *,
        max_size: OptInt = None,
        valid_content_types: SeqOfStrs | str | None = None,
        valid_extesions: SeqOfStrs | str | None = None,
    ) -> Self:
        content = await file.read()
        if not content:
            raise ValueError("Content can not be empty")

        size = file.size
        if size is None or size <= 0:
            raise ValueError("Size can not be None and must be larger than 0")
        if max_size is not None:
            if size > max_size:
                raise ValueError("Size can not exceed ")

        content_type = file.content_type
        if content_type is None:
            raise ValueError("Content type can not be None")
        if valid_content_types is not None:
            if isinstance(valid_content_types, str):
                if content_type != valid_content_types:
                    raise ValueError(
                        f"Invalid content type of '{content_type}'. Must be '{valid_content_types}'"
                    )
            else:
                if content_type not in valid_content_types:
                    raise ValueError(
                        f"Invalid content type of '{content_type}'. Must be one of {valid_content_types}"
                    )

        filename = file.filename
        if filename is None:
            raise ValueError("Filename can not be None")
        if valid_extesions is not None:
            if isinstance(valid_extesions, str):
                if not filename.endswith(valid_extesions):
                    raise ValueError(f"Invalid extension. Must be '{valid_extesions}'")
            else:
                if not any(filename.endswith(ext) for ext in valid_extesions):
                    raise ValueError(
                        f"Invalid extension. Must be one of {valid_extesions}"
                    )

        filename = filename.replace(" ", "_")

        document = cls(
            content=content, content_type=content_type, filename=filename, size=size
        )
        document._raw = file

        return document


OptDocument = Document | None
OptDocumentT = TypeVar("OptDocumentT", bound=OptDocument)
ListOfDocuments = list[Document]
OptListOfDocuments = ListOfDocuments | None
OptListOfDocumentsT = TypeVar("OptListOfDocumentsT", bound=OptListOfDocuments)


class DocumentMixin(BaseModel, Generic[OptDocumentT]):
    document: Annotated[OptDocumentT, Field(..., description="Document")]


class DocumentsMixin(BaseModel, Generic[OptListOfDocumentsT]):
    documents: Annotated[
        OptListOfDocumentsT, Field(..., description="Documents", min_length=1)
    ]
