# coding: utf-8

"""
righteous.api


Implements the RightScale API for EC2 instance management.
http://support.rightscale.com/15-References/RightScale_API_Reference_Guide/02-Management/02-Servers
"""

from . import config
import sys, base64
import requests
from urllib import urlencode
import omnijson as json

from logging import getLogger
log = getLogger(__name__)

ACCOUNT_URL = 'https://my.rightscale.com/api/acct/'

def _debug(message):
    log.debug(message)
    if config.settings.debug:
        config.settings.debug.write('%s\n' % message)

def _build_headers(headers=None):
    request_headers = {'X-API-VERSION': '1.0'}
    if headers:
        request_headers.update(headers)
    if config.settings.cookies:
        request_headers['Cookie'] = config.settings.cookies
    return request_headers

def _request(path, method='GET', body=None, headers={}, prepend_api_base=True):
    if prepend_api_base:
        path = ACCOUNT_URL + config.settings.account_id + path
    headers = _build_headers(headers=headers)
    log.debug('%s to %s with data=%s, headers=%s', method, path, body, headers)
    return requests.request(method, path, data=body, headers=headers, config=config.settings.requests_config or {})

def init(username, password, account_id, **kwargs):
    """
    Initialises righteous configuration

    :param username: String of a Rightscale username
    :param password: String of the user's password
    :param account_id: String of the Rightscale account_id
    :params kwargs: Key word arguments for additional configuration
    """

    if not username or not password or not account_id:
        raise Exception('Username, password and account_id are required parameters')

    config.settings.username = username
    config.settings.password = password
    config.settings.account_id = account_id
    config.settings.api_base = ACCOUNT_URL + account_id

    config.settings.default_deployment_id = kwargs.get('default_deployment_id', None)
    config.settings.debug = kwargs.get('debug', False)

    config.settings.create_server_parameters = {}
    for key, value in kwargs.items():
        config.settings.create_server_parameters[key] = value
        if key=='default_deployment_id':
            config.settings.create_server_parameters['deployment_href'] = '%s%s/deployments/%s' % (ACCOUNT_URL, account_id, config.settings.default_deployment_id)

    if config.settings.debug:
        config.settings.requests_config = {'verbose': sys.stderr}
        config.settings.debug = sys.stderr

## http://support.rightscale.com/12-Guides/03-RightScale_API/OAuth
#def oauth_token():
    #response = _request('https://my.rightscale.com/api/oauth2', 
                        #method='POST', 
                        #body=urlencode({'grant_type':'refresh_token','refresh_token':config.username}), 
                        #headers={"X-API-Version":"1.5"}, prepend_api_base=False)
    #if response.status_code == 200:
        #data = json.loads(response.content)
        #print data
        #if not 'access_token' in data:
            #raise Exception('No token')
        #else:
            #access_token = data['access_token']

     #-H "X-API-Version:1.5" \
     #-H "Authorization: Bearer $access_token" \

def login(username=None, password=None, account_id=None):
    """
    Logins to RightScale and stores the auth cookie for future requests

    :param username: (optional) String representing the username to login with
    :param password: (optional) String representing the password to login with
    :param account_id: (optional) String of the Rightscale account_id
    """
    if not username or not password or not account_id:
        username = config.settings.username
        password = config.settings.password
        account_id = config.settings.account_id

    if not username or not password or not account_id:
        raise Exception('Username, password or account_id not specified in configuration or as an API parameter')

    response = _request('/login', headers={'Authorization':"Basic %s" % base64.encodestring('%s:%s' % (username, password))[:-1]})

    if response.status_code == 204:
        config.settings.cookies = response.headers['set-cookie']
        config.settings.username = username
        config.settings.password = password
        return True
    return False

def list_servers(deployment_id=None):
    """
    Lists servers in a deployment

    Returns JSON

    :param deployment_id: (optional) String representing Deployment to list servers from
    """
    if not deployment_id:
        deployment_id = config.settings.default_deployment_id

    if not deployment_id:
        raise Exception('Deployment id not specified in configuration or as an API parameter')

    response = _request('/deployments/%s.js' % deployment_id)
    return json.loads(response.content)

def find_server(nickname):
    """
    Finds a server based on nickname

    :param nickname: (optional) String representing the nickname of the server to lookup
    """
    response = _request('/servers.js?filter=nickname=%s' % nickname)

    servers = json.loads(response.content)
    return servers[0] if len(servers) else None

def _lookup_server(server_href, nickname):
    if not nickname and not server_href:
        raise ValueError('Either nickname or server_href must be specified')

    if server_href:
        return server_href
    elif nickname:
        return find_server(nickname)['href']

def server_info(server_href, nickname=None):
    """
    Detailed server information

    :param server_href: URL representing the server to retrieve information about
    :param nickname: (optional) String representing the nickname of the server
    """
    response = _request('%s.js' % _lookup_server(server_href, nickname), prepend_api_base=False)
    return json.loads(response.content)

def server_settings(server_href, nickname=None):
    """
    Current server settings

    :param server_href: URL representing the server to query settngs from
    :param nickname: (optional) String representing the nickname of the server
    """
    response = _request('%s/settings.js' % _lookup_server(server_href, nickname), prepend_api_base=False)
    return json.loads(response.content)

def start_server(server_href, nickname=None):
    """
    Starts a server.

    :param server_href: URL representing the server to start
    :param nickname: (optional) String representing the nickname of the server
    """
    return _request('%s/start' % _lookup_server(server_href, nickname), method='POST', prepend_api_base=False)

def stop_server(server_href, nickname=None):
    """
    Stops a server.

    :param server_href: URL representing the server to stop
    :param nickname: (optional) String representing the nickname of the server
    """
    return _request('%s/stop' % _lookup_server(server_href, nickname), method='POST', prepend_api_base=False).status_code == 201

def delete_server(server_href, nickname=None):
    """
    Deletes a server from RightScale

    :param server_href: URL representing the server to delete
    :param nickname: (optional) String representing the nickname of the server
    """
    return _request(_lookup_server(server_href, nickname), method='DELETE', prepend_api_base=False).status_code == 200

def create_server(nickname, create_server_parameters=None):
    """
    Create a server.

    :param nickname: String representing the nickname of the server
    :param create_server_parameters: (optional) Dictionary of server creation parameters
    """
    if not create_server_parameters:
        create_server_parameters = config.settings.create_server_parameters
    else:
        log.debug('using supplied parameters: %s', create_server_parameters)
    _debug(create_server_parameters)
    create_data = {'server[nickname]' : nickname}
    for key, value in create_server_parameters.items():
        create_data['server[%s]' % key] = value
    response = _request('/servers', method='POST', body=urlencode(create_data))
    _debug(response)
    location = response.headers.get('location')
    _debug('Created %s: %s (%s:%s)' % (nickname, location, response.status_code, response.content))
    return location

def set_server_parameters(server_href, parameters):
    """
    Updates/sets any ServerTemplate parameters for a server

    :param server_url: URL representing the server to update
    :param parameters: Dictionary of ServerTemplate parameters to set
    """
    input_data = []
    for key,value in parameters.items():
        input_data.append('server[parameters][%s]=text:%s' % (key.upper(), value))
    update = '&'.join(input_data)
    return _request(server_href, method='PUT', body=update, headers={'Content-Type': 'application/x-www-form-urlencoded'}, prepend_api_base=False)

def create_and_start_server(nickname, create_server_parameters=None, server_template_parameters=None):
    """Creates and starts a server.
    Returns a tuple of operation status, href of the created, started server

    :param nickname: String representing the nickname of the server
    :param create_server_parameters: (optional) Dictionary of server creation parameters
    :param server_template_parameters: (optional) Dictionary of ServerTemplate parameters
    """
    log.debug('in create_and_Start_server %s', create_server_parameters)
    server_href = create_server(nickname, create_server_parameters)
    _debug(server_href)

    if server_href:
        if server_template_parameters:
            set_server_parameters(server_href, server_template_parameters)

        start_server_response = start_server(server_href)
        _debug(start_server_response)
        success = start_server_response.status_code == 201
        location = start_server_response.headers['location'] if success else None
        return success, location
    else:
        return False, None
