from writer.abstract import register_abstract_template
from writer.blocks.base_block import BlueprintBlock
from writer.ss_types import AbstractTemplate, WriterConfigurationError


class WriterAddChatMessage(BlueprintBlock):

    @classmethod
    def register(cls, type: str):
        super(WriterAddChatMessage, cls).register(type)
        register_abstract_template(type, AbstractTemplate(
            baseType="blueprints_node",
            writer={
                "name": "Add chat message",
                "description": "Adds a message to the conversation history. Use for displaying user or AI messages.",
                "category": "Writer",
                "deprecated": True,
                "fields": {
                    "conversationStateElement": {
                        "name": "Conversation state element",
                        "desc": "The variable that has your conversation object.",
                        "type": "Binding",
                    },
                    "message": {
                        "name": "Message",
                        "type": "Object",
                        "init": '{ "role": "assistant", "content": "Hello" }',
                        "desc": "A message object. Content can be text string or array of objects for multimodal (text + images) with X5 model.",
                        "validator": {
                            "type": "object",
                            "properties": {
                                "role": {"type": "string"},
                                "content": {
                                    "oneOf": [
                                        {"type": "string"},
                                        {
                                            "type": "array",
                                            "items": {
                                                "oneOf": [
                                                    {
                                                        "type": "object",
                                                        "properties": {
                                                            "type": {"type": "string", "enum": ["text"]},
                                                            "text": {"type": "string"}
                                                        },
                                                        "required": ["type", "text"],
                                                        "additionalProperties": False
                                                    },
                                                    {
                                                        "type": "object",
                                                        "properties": {
                                                            "type": {"type": "string", "enum": ["image_url"]},
                                                            "image_url": {
                                                                "type": "object",
                                                                "properties": {
                                                                    "url": {"type": "string"}
                                                                },
                                                                "required": ["url"],
                                                                "additionalProperties": False
                                                            }
                                                        },
                                                        "required": ["type", "image_url"],
                                                        "additionalProperties": False
                                                    }
                                                ]
                                            }
                                        }
                                    ]
                                },
                            },
                            "additionalProperties": False,
                        }
                    }
                },
                "outs": {
                    "success": {
                        "name": "Success",
                        "description": "The message was added to the conversation.",
                        "style": "success",
                    },
                    "error": {
                        "name": "Error",
                        "description": "There was an error adding the message.",
                        "style": "error",
                    },
                },
            }
        ))

    def run(self):
        try:
            import writer.ai

            conversation_state_element = self._get_field("conversationStateElement", required=True)
            message = self._get_field("message", as_json=True, required=True)

            conversation = self.evaluator.evaluate_expression(conversation_state_element, self.instance_path, self.execution_environment)

            if conversation is None or not isinstance(conversation, writer.ai.Conversation):
                raise WriterConfigurationError("The state element specified doesn't contain a conversation. Initialize one using the block 'Start chat conversation'.")
            
            writer.ai.Conversation.validate_message(message)
            conversation += message

            self._set_state(conversation_state_element, conversation)            
            self.result = None
            self.outcome = "success"
        except BaseException as e:
            self.outcome = "error"
            raise e

    
