"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        switch (artifact.type) {
            case cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK:
                return new cloudformation_artifact_1.CloudFormationStackArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.CDK_TREE:
                return new tree_cloud_artifact_1.TreeCloudArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.ASSET_MANIFEST:
                return new asset_manifest_artifact_1.AssetManifestArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY:
                return new nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact(assembly, id, artifact);
            default:
                return undefined;
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
}
exports.CloudArtifact = CloudArtifact;
// needs to be defined at the end to avoid a cyclic dependency
const asset_manifest_artifact_1 = require("./artifacts/asset-manifest-artifact");
const cloudformation_artifact_1 = require("./artifacts/cloudformation-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWQtYXJ0aWZhY3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbG91ZC1hcnRpZmFjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSwyREFBMkQ7QUFFM0QseUNBQTBGO0FBOEIxRjs7R0FFRztBQUNILE1BQWEsYUFBYTtJQTRDeEIsWUFBc0MsUUFBdUIsRUFBa0IsRUFBVSxFQUFFLFFBQW1DO1FBQXhGLGFBQVEsR0FBUixRQUFRLENBQWU7UUFBa0IsT0FBRSxHQUFGLEVBQUUsQ0FBUTtRQUN2RixJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUN6QixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztRQUN0QyxJQUFJLENBQUMsY0FBYyxHQUFHLFFBQVEsQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDO0lBQ3BELENBQUM7SUEvQ0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUF1QixFQUFFLEVBQVUsRUFBRSxRQUFtQztRQUNqRyxRQUFRLFFBQVEsQ0FBQyxJQUFJLEVBQUU7WUFDckIsS0FBSyxRQUFRLENBQUMsWUFBWSxDQUFDLHdCQUF3QjtnQkFDakQsT0FBTyxJQUFJLHFEQUEyQixDQUFDLFFBQVEsRUFBRSxFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDakUsS0FBSyxRQUFRLENBQUMsWUFBWSxDQUFDLFFBQVE7Z0JBQ2pDLE9BQU8sSUFBSSx1Q0FBaUIsQ0FBQyxRQUFRLEVBQUUsRUFBRSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3ZELEtBQUssUUFBUSxDQUFDLFlBQVksQ0FBQyxjQUFjO2dCQUN2QyxPQUFPLElBQUksK0NBQXFCLENBQUMsUUFBUSxFQUFFLEVBQUUsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUMzRCxLQUFLLFFBQVEsQ0FBQyxZQUFZLENBQUMscUJBQXFCO2dCQUM5QyxPQUFPLElBQUksNERBQTJCLENBQUMsUUFBUSxFQUFFLEVBQUUsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUNqRTtnQkFDRSxPQUFPLFNBQVMsQ0FBQztTQUNwQjtJQUNILENBQUM7SUE2QkQ7O09BRUc7SUFDSCxJQUFXLFlBQVk7UUFDckIsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQUUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO1NBQUU7UUFFdEMsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRTtZQUN4QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUM3QyxJQUFJLENBQUMsR0FBRyxFQUFFO2dCQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsWUFBWSxJQUFJLENBQUMsRUFBRSxxQ0FBcUMsRUFBRSxFQUFFLENBQUMsQ0FBQzthQUMvRTtZQUNELE9BQU8sR0FBRyxDQUFDO1FBQ2IsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUM7SUFDcEIsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGtCQUFrQixDQUFDLElBQVk7UUFDcEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQXVCLENBQUM7UUFDaEQsS0FBSyxNQUFNLElBQUksSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQzVELEtBQUssTUFBTSxLQUFLLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRTtnQkFDeEQsSUFBSSxLQUFLLENBQUMsSUFBSSxLQUFLLElBQUksRUFBRTtvQkFDdkIsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7aUJBQ2pDO2FBQ0Y7U0FDRjtRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFTyxjQUFjO1FBQ3BCLE1BQU0sUUFBUSxHQUFHLElBQUksS0FBSyxFQUFvQixDQUFDO1FBRS9DLEtBQUssTUFBTSxDQUFFLEVBQUUsRUFBRSxRQUFRLENBQUUsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxJQUFJLEVBQUcsQ0FBQyxFQUFFO1lBQzVFLEtBQUssTUFBTSxLQUFLLElBQUksUUFBUSxFQUFFO2dCQUM1QixJQUFJLEtBQTRCLENBQUM7Z0JBQ2pDLFFBQVEsS0FBSyxDQUFDLElBQUksRUFBRTtvQkFDbEIsS0FBSyxRQUFRLENBQUMseUJBQXlCLENBQUMsSUFBSTt3QkFDMUMsS0FBSyxHQUFHLGdDQUFxQixDQUFDLE9BQU8sQ0FBQzt3QkFDdEMsTUFBTTtvQkFDUixLQUFLLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLO3dCQUMzQyxLQUFLLEdBQUcsZ0NBQXFCLENBQUMsS0FBSyxDQUFDO3dCQUNwQyxNQUFNO29CQUNSLEtBQUssUUFBUSxDQUFDLHlCQUF5QixDQUFDLElBQUk7d0JBQzFDLEtBQUssR0FBRyxnQ0FBcUIsQ0FBQyxJQUFJLENBQUM7d0JBQ25DLE1BQU07b0JBQ1I7d0JBQ0UsU0FBUztpQkFDWjtnQkFFRCxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO2FBQ3JDO1NBQ0Y7UUFFRCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0NBQ0Y7QUE3R0Qsc0NBNkdDO0FBRUQsOERBQThEO0FBQzlELGlGQUE0RTtBQUM1RSxpRkFBa0Y7QUFDbEYsK0ZBQXlGO0FBQ3pGLHlFQUFvRSIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGN4c2NoZW1hIGZyb20gJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSc7XG5pbXBvcnQgeyBDbG91ZEFzc2VtYmx5IH0gZnJvbSAnLi9jbG91ZC1hc3NlbWJseSc7XG5pbXBvcnQgeyBNZXRhZGF0YUVudHJ5UmVzdWx0LCBTeW50aGVzaXNNZXNzYWdlLCBTeW50aGVzaXNNZXNzYWdlTGV2ZWwgfSBmcm9tICcuL21ldGFkYXRhJztcblxuLyoqXG4gKiBBcnRpZmFjdCBwcm9wZXJ0aWVzIGZvciBDbG91ZEZvcm1hdGlvbiBzdGFja3MuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXdzQ2xvdWRGb3JtYXRpb25TdGFja1Byb3BlcnRpZXMge1xuICAvKipcbiAgICogQSBmaWxlIHJlbGF0aXZlIHRvIHRoZSBhc3NlbWJseSByb290IHdoaWNoIGNvbnRhaW5zIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBmb3IgdGhpcyBzdGFjay5cbiAgICovXG4gIHJlYWRvbmx5IHRlbXBsYXRlRmlsZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWYWx1ZXMgZm9yIENsb3VkRm9ybWF0aW9uIHN0YWNrIHBhcmFtZXRlcnMgdGhhdCBzaG91bGQgYmUgcGFzc2VkIHdoZW4gdGhlIHN0YWNrIGlzIGRlcGxveWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHsgW2lkOiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogVGhlIG5hbWUgdG8gdXNlIGZvciB0aGUgQ2xvdWRGb3JtYXRpb24gc3RhY2suXG4gICAqIEBkZWZhdWx0IC0gbmFtZSBkZXJpdmVkIGZyb20gYXJ0aWZhY3QgSURcbiAgICovXG4gIHJlYWRvbmx5IHN0YWNrTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciB0byBlbmFibGUgdGVybWluYXRpb24gcHJvdGVjdGlvbiBmb3IgdGhpcyBzdGFjay5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHRlcm1pbmF0aW9uUHJvdGVjdGlvbj86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBhcnRpZmFjdCB3aXRoaW4gYSBjbG91ZCBhc3NlbWJseS5cbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkQXJ0aWZhY3Qge1xuICAvKipcbiAgICogUmV0dXJucyBhIHN1YmNsYXNzIG9mIGBDbG91ZEFydGlmYWN0YCBiYXNlZCBvbiB0aGUgYXJ0aWZhY3QgdHlwZSBkZWZpbmVkIGluIHRoZSBhcnRpZmFjdCBtYW5pZmVzdC5cbiAgICogQHBhcmFtIGFzc2VtYmx5IFRoZSBjbG91ZCBhc3NlbWJseSBmcm9tIHdoaWNoIHRvIGxvYWQgdGhlIGFydGlmYWN0XG4gICAqIEBwYXJhbSBpZCBUaGUgYXJ0aWZhY3QgSURcbiAgICogQHBhcmFtIGFydGlmYWN0IFRoZSBhcnRpZmFjdCBtYW5pZmVzdFxuICAgKiBAcmV0dXJucyB0aGUgYENsb3VkQXJ0aWZhY3RgIHRoYXQgbWF0Y2hlcyB0aGUgYXJ0aWZhY3QgdHlwZSBvciBgdW5kZWZpbmVkYCBpZiBpdCdzIGFuIGFydGlmYWN0IHR5cGUgdGhhdCBpcyB1bnJlY29nbml6ZWQgYnkgdGhpcyBtb2R1bGUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21NYW5pZmVzdChhc3NlbWJseTogQ2xvdWRBc3NlbWJseSwgaWQ6IHN0cmluZywgYXJ0aWZhY3Q6IGN4c2NoZW1hLkFydGlmYWN0TWFuaWZlc3QpOiBDbG91ZEFydGlmYWN0IHwgdW5kZWZpbmVkIHtcbiAgICBzd2l0Y2ggKGFydGlmYWN0LnR5cGUpIHtcbiAgICAgIGNhc2UgY3hzY2hlbWEuQXJ0aWZhY3RUeXBlLkFXU19DTE9VREZPUk1BVElPTl9TVEFDSzpcbiAgICAgICAgcmV0dXJuIG5ldyBDbG91ZEZvcm1hdGlvblN0YWNrQXJ0aWZhY3QoYXNzZW1ibHksIGlkLCBhcnRpZmFjdCk7XG4gICAgICBjYXNlIGN4c2NoZW1hLkFydGlmYWN0VHlwZS5DREtfVFJFRTpcbiAgICAgICAgcmV0dXJuIG5ldyBUcmVlQ2xvdWRBcnRpZmFjdChhc3NlbWJseSwgaWQsIGFydGlmYWN0KTtcbiAgICAgIGNhc2UgY3hzY2hlbWEuQXJ0aWZhY3RUeXBlLkFTU0VUX01BTklGRVNUOlxuICAgICAgICByZXR1cm4gbmV3IEFzc2V0TWFuaWZlc3RBcnRpZmFjdChhc3NlbWJseSwgaWQsIGFydGlmYWN0KTtcbiAgICAgIGNhc2UgY3hzY2hlbWEuQXJ0aWZhY3RUeXBlLk5FU1RFRF9DTE9VRF9BU1NFTUJMWTpcbiAgICAgICAgcmV0dXJuIG5ldyBOZXN0ZWRDbG91ZEFzc2VtYmx5QXJ0aWZhY3QoYXNzZW1ibHksIGlkLCBhcnRpZmFjdCk7XG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgYXJ0aWZhY3QncyBtYW5pZmVzdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG1hbmlmZXN0OiBjeHNjaGVtYS5BcnRpZmFjdE1hbmlmZXN0O1xuXG4gIC8qKlxuICAgKiBUaGUgc2V0IG9mIG1lc3NhZ2VzIGV4dHJhY3RlZCBmcm9tIHRoZSBhcnRpZmFjdCdzIG1ldGFkYXRhLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG1lc3NhZ2VzOiBTeW50aGVzaXNNZXNzYWdlW107XG5cbiAgLyoqXG4gICAqIElEcyBvZiBhbGwgZGVwZW5kZW5jaWVzLiBVc2VkIHdoZW4gdG9wb2xvZ2ljYWxseSBzb3J0aW5nIHRoZSBhcnRpZmFjdHMgd2l0aGluIHRoZSBjbG91ZCBhc3NlbWJseS5cbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgX2RlcGVuZGVuY3lJRHM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBDYWNoZSBvZiByZXNvbHZlZCBkZXBlbmRlbmNpZXMuXG4gICAqL1xuICBwcml2YXRlIF9kZXBzPzogQ2xvdWRBcnRpZmFjdFtdO1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgYXNzZW1ibHk6IENsb3VkQXNzZW1ibHksIHB1YmxpYyByZWFkb25seSBpZDogc3RyaW5nLCBtYW5pZmVzdDogY3hzY2hlbWEuQXJ0aWZhY3RNYW5pZmVzdCkge1xuICAgIHRoaXMubWFuaWZlc3QgPSBtYW5pZmVzdDtcbiAgICB0aGlzLm1lc3NhZ2VzID0gdGhpcy5yZW5kZXJNZXNzYWdlcygpO1xuICAgIHRoaXMuX2RlcGVuZGVuY3lJRHMgPSBtYW5pZmVzdC5kZXBlbmRlbmNpZXMgfHwgW107XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhbGwgdGhlIGFydGlmYWN0cyB0aGF0IHRoaXMgYXJ0aWZhY3QgZGVwZW5kcyBvbi5cbiAgICovXG4gIHB1YmxpYyBnZXQgZGVwZW5kZW5jaWVzKCk6IENsb3VkQXJ0aWZhY3RbXSB7XG4gICAgaWYgKHRoaXMuX2RlcHMpIHsgcmV0dXJuIHRoaXMuX2RlcHM7IH1cblxuICAgIHRoaXMuX2RlcHMgPSB0aGlzLl9kZXBlbmRlbmN5SURzLm1hcChpZCA9PiB7XG4gICAgICBjb25zdCBkZXAgPSB0aGlzLmFzc2VtYmx5LnRyeUdldEFydGlmYWN0KGlkKTtcbiAgICAgIGlmICghZGVwKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgQXJ0aWZhY3QgJHt0aGlzLmlkfSBkZXBlbmRzIG9uIG5vbi1leGlzdGluZyBhcnRpZmFjdCAke2lkfWApO1xuICAgICAgfVxuICAgICAgcmV0dXJuIGRlcDtcbiAgICB9KTtcblxuICAgIHJldHVybiB0aGlzLl9kZXBzO1xuICB9XG5cbiAgLyoqXG4gICAqIEByZXR1cm5zIGFsbCB0aGUgbWV0YWRhdGEgZW50cmllcyBvZiBhIHNwZWNpZmljIHR5cGUgaW4gdGhpcyBhcnRpZmFjdC5cbiAgICogQHBhcmFtIHR5cGVcbiAgICovXG4gIHB1YmxpYyBmaW5kTWV0YWRhdGFCeVR5cGUodHlwZTogc3RyaW5nKTogTWV0YWRhdGFFbnRyeVJlc3VsdFtdIHtcbiAgICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXk8TWV0YWRhdGFFbnRyeVJlc3VsdD4oKTtcbiAgICBmb3IgKGNvbnN0IHBhdGggb2YgT2JqZWN0LmtleXModGhpcy5tYW5pZmVzdC5tZXRhZGF0YSB8fCB7fSkpIHtcbiAgICAgIGZvciAoY29uc3QgZW50cnkgb2YgKHRoaXMubWFuaWZlc3QubWV0YWRhdGEgfHwge30pW3BhdGhdKSB7XG4gICAgICAgIGlmIChlbnRyeS50eXBlID09PSB0eXBlKSB7XG4gICAgICAgICAgcmVzdWx0LnB1c2goeyBwYXRoLCAuLi5lbnRyeSB9KTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJNZXNzYWdlcygpIHtcbiAgICBjb25zdCBtZXNzYWdlcyA9IG5ldyBBcnJheTxTeW50aGVzaXNNZXNzYWdlPigpO1xuXG4gICAgZm9yIChjb25zdCBbIGlkLCBtZXRhZGF0YSBdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMubWFuaWZlc3QubWV0YWRhdGEgfHwgeyB9KSkge1xuICAgICAgZm9yIChjb25zdCBlbnRyeSBvZiBtZXRhZGF0YSkge1xuICAgICAgICBsZXQgbGV2ZWw6IFN5bnRoZXNpc01lc3NhZ2VMZXZlbDtcbiAgICAgICAgc3dpdGNoIChlbnRyeS50eXBlKSB7XG4gICAgICAgICAgY2FzZSBjeHNjaGVtYS5BcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlLldBUk46XG4gICAgICAgICAgICBsZXZlbCA9IFN5bnRoZXNpc01lc3NhZ2VMZXZlbC5XQVJOSU5HO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgY2FzZSBjeHNjaGVtYS5BcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlLkVSUk9SOlxuICAgICAgICAgICAgbGV2ZWwgPSBTeW50aGVzaXNNZXNzYWdlTGV2ZWwuRVJST1I7XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgICBjYXNlIGN4c2NoZW1hLkFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuSU5GTzpcbiAgICAgICAgICAgIGxldmVsID0gU3ludGhlc2lzTWVzc2FnZUxldmVsLklORk87XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgIH1cblxuICAgICAgICBtZXNzYWdlcy5wdXNoKHsgbGV2ZWwsIGVudHJ5LCBpZCB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbWVzc2FnZXM7XG4gIH1cbn1cblxuLy8gbmVlZHMgdG8gYmUgZGVmaW5lZCBhdCB0aGUgZW5kIHRvIGF2b2lkIGEgY3ljbGljIGRlcGVuZGVuY3lcbmltcG9ydCB7IEFzc2V0TWFuaWZlc3RBcnRpZmFjdCB9IGZyb20gJy4vYXJ0aWZhY3RzL2Fzc2V0LW1hbmlmZXN0LWFydGlmYWN0JztcbmltcG9ydCB7IENsb3VkRm9ybWF0aW9uU3RhY2tBcnRpZmFjdCB9IGZyb20gJy4vYXJ0aWZhY3RzL2Nsb3VkZm9ybWF0aW9uLWFydGlmYWN0JztcbmltcG9ydCB7IE5lc3RlZENsb3VkQXNzZW1ibHlBcnRpZmFjdCB9IGZyb20gJy4vYXJ0aWZhY3RzL25lc3RlZC1jbG91ZC1hc3NlbWJseS1hcnRpZmFjdCc7XG5pbXBvcnQgeyBUcmVlQ2xvdWRBcnRpZmFjdCB9IGZyb20gJy4vYXJ0aWZhY3RzL3RyZWUtY2xvdWQtYXJ0aWZhY3QnOyJdfQ==