"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnvironmentPlaceholders = void 0;
/**
 * Placeholders which can be used manifests
 *
 * These can occur both in the Asset Manifest as well as the general
 * Cloud Assembly manifest.
 */
class EnvironmentPlaceholders {
    /**
     * Replace the environment placeholders in all strings found in a complex object.
     *
     * Duplicated between cdk-assets and aws-cdk CLI because we don't have a good single place to put it
     * (they're nominally independent tools).
     */
    static replace(object, values) {
        return this.recurse(object, value => {
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_REGION, values.region);
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_ACCOUNT, values.accountId);
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_PARTITION, values.partition);
            return value;
        });
    }
    /**
     * Like 'replace', but asynchronous
     */
    static async replaceAsync(object, provider) {
        let needRegion = false;
        let needAccountId = false;
        let needPartition = false;
        this.recurse(object, value => {
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_REGION) > 1) {
                needRegion = true;
            }
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_ACCOUNT) > 1) {
                needAccountId = true;
            }
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_PARTITION) > 1) {
                needPartition = true;
            }
            return value;
        });
        const region = needRegion ? await provider.region() : undefined;
        const accountId = needAccountId ? await provider.accountId() : undefined;
        const partition = needPartition ? await provider.partition() : undefined;
        return this.recurse(object, value => {
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_REGION, region !== null && region !== void 0 ? region : 'WONTHAPPEN');
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_ACCOUNT, accountId !== null && accountId !== void 0 ? accountId : 'WONTHAPPEN');
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_PARTITION, partition !== null && partition !== void 0 ? partition : 'WONTHAPPEN');
            return value;
        });
    }
    static recurse(value, cb) {
        if (typeof value === 'string') {
            return cb(value);
        }
        if (typeof value !== 'object' || value === null) {
            return value;
        }
        if (Array.isArray(value)) {
            return value.map(x => this.recurse(x, cb));
        }
        const ret = {};
        for (const [key, inner] of Object.entries(value)) {
            ret[key] = this.recurse(inner, cb);
        }
        return ret;
    }
}
exports.EnvironmentPlaceholders = EnvironmentPlaceholders;
/**
 * Insert this into the destination fields to be replaced with the current region
 */
EnvironmentPlaceholders.CURRENT_REGION = '${AWS::Region}';
/**
 * Insert this into the destination fields to be replaced with the current account
 */
EnvironmentPlaceholders.CURRENT_ACCOUNT = '${AWS::AccountId}';
/**
 * Insert this into the destination fields to be replaced with the current partition
 */
EnvironmentPlaceholders.CURRENT_PARTITION = '${AWS::Partition}';
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
//# sourceMappingURL=data:application/json;base64,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