"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cxapi = require("../lib");
test('cloud assembly builder', () => {
    // GIVEN
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cloud-assembly-builder-tests'));
    const session = new cxapi.CloudAssemblyBuilder(outdir);
    const templateFile = 'foo.template.json';
    // WHEN
    session.addArtifact('my-first-artifact', {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://1222344/us-east-1',
        dependencies: ['minimal-artifact'],
        metadata: {
            foo: [{ data: '123', type: 'foo', trace: [] }],
        },
        properties: {
            templateFile,
            parameters: {
                prop1: '1234',
                prop2: '555',
            },
        },
    });
    session.addArtifact('tree-artifact', {
        type: cxschema.ArtifactType.CDK_TREE,
        properties: {
            file: 'foo.tree.json',
        },
    });
    session.addMissing({
        key: 'foo',
        provider: cxschema.ContextProvider.VPC_PROVIDER,
        props: {
            account: '1234',
            region: 'us-east-1',
            filter: {
                a: 'a',
            },
        },
    });
    session.addArtifact('minimal-artifact', {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://111/helo-world',
        properties: {
            templateFile,
        },
    });
    fs.writeFileSync(path.join(session.outdir, templateFile), JSON.stringify({
        Resources: {
            MyTopic: {
                Type: 'AWS::S3::Topic',
            },
        },
    }));
    const assembly = session.buildAssembly();
    const manifest = assembly.manifest;
    // THEN
    // verify the manifest looks right
    expect(manifest).toStrictEqual({
        version: cxschema.Manifest.version(),
        missing: [
            {
                key: 'foo',
                provider: 'vpc-provider',
                props: {
                    account: '1234',
                    region: 'us-east-1',
                    filter: {
                        a: 'a',
                    },
                },
            },
        ],
        artifacts: {
            'tree-artifact': {
                type: 'cdk:tree',
                properties: {
                    file: 'foo.tree.json',
                },
            },
            'my-first-artifact': {
                type: 'aws:cloudformation:stack',
                environment: 'aws://1222344/us-east-1',
                dependencies: ['minimal-artifact'],
                metadata: { foo: [{ data: '123', type: 'foo', trace: [] }] },
                properties: {
                    templateFile: 'foo.template.json',
                    parameters: {
                        prop1: '1234',
                        prop2: '555',
                    },
                },
            },
            'minimal-artifact': {
                type: 'aws:cloudformation:stack',
                environment: 'aws://111/helo-world',
                properties: { templateFile: 'foo.template.json' },
            },
        },
    });
    // verify we have a template file
    expect(assembly.getStackByName('minimal-artifact').template).toStrictEqual({
        Resources: {
            MyTopic: {
                Type: 'AWS::S3::Topic',
            },
        },
    });
});
test('outdir must be a directory', () => {
    expect(() => new cxapi.CloudAssemblyBuilder(__filename)).toThrow('must be a directory');
});
test('duplicate missing values with the same key are only reported once', () => {
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cloud-assembly-builder-tests'));
    const session = new cxapi.CloudAssemblyBuilder(outdir);
    const props = {
        account: '1234',
        region: 'asdf',
        filter: { a: 'a' },
    };
    session.addMissing({ key: 'foo', provider: cxschema.ContextProvider.VPC_PROVIDER, props });
    session.addMissing({ key: 'foo', provider: cxschema.ContextProvider.VPC_PROVIDER, props });
    const assembly = session.buildAssembly();
    expect(assembly.manifest.missing.length).toEqual(1);
});
test('write and read nested cloud assembly artifact', () => {
    // GIVEN
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cloud-assembly-builder-tests'));
    const session = new cxapi.CloudAssemblyBuilder(outdir);
    const innerAsmDir = path.join(outdir, 'hello');
    new cxapi.CloudAssemblyBuilder(innerAsmDir).buildAssembly();
    // WHEN
    session.addArtifact('Assembly', {
        type: cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY,
        properties: {
            directoryName: 'hello',
        },
    });
    const asm = session.buildAssembly();
    // THEN
    const art = asm.tryGetArtifact('Assembly');
    expect(art).toBeInstanceOf(cxapi.NestedCloudAssemblyArtifact);
    expect(art === null || art === void 0 ? void 0 : art.fullPath).toEqual(path.join(outdir, 'hello'));
    const nested = art === null || art === void 0 ? void 0 : art.nestedAssembly;
    expect(nested === null || nested === void 0 ? void 0 : nested.artifacts.length).toEqual(0);
});
test('artifcats are written in topological order', () => {
    // GIVEN
    const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cloud-assembly-builder-tests'));
    const session = new cxapi.CloudAssemblyBuilder(outdir);
    const templateFile = 'foo.template.json';
    const innerAsmDir = path.join(outdir, 'hello');
    new cxapi.CloudAssemblyBuilder(innerAsmDir).buildAssembly();
    // WHEN
    // Create the following dependency order:
    // A ->
    //      C -> D
    // B ->
    session.addArtifact('artifact-D', {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://1222344/us-east-1',
        dependencies: ['artifact-C'],
        properties: {
            templateFile,
        },
    });
    session.addArtifact('artifact-C', {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://1222344/us-east-1',
        dependencies: ['artifact-B', 'artifact-A'],
        properties: {
            templateFile,
        },
    });
    session.addArtifact('artifact-B', {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://1222344/us-east-1',
        properties: {
            templateFile,
        },
    });
    session.addArtifact('artifact-A', {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: 'aws://1222344/us-east-1',
        properties: {
            templateFile,
        },
    });
    const asm = session.buildAssembly();
    const artifactsIds = asm.artifacts.map(a => a.id);
    // THEN
    expect(artifactsIds.indexOf('artifact-A')).toBeLessThan(artifactsIds.indexOf('artifact-C'));
    expect(artifactsIds.indexOf('artifact-B')).toBeLessThan(artifactsIds.indexOf('artifact-C'));
    expect(artifactsIds.indexOf('artifact-C')).toBeLessThan(artifactsIds.indexOf('artifact-D'));
});
//# sourceMappingURL=data:application/json;base64,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