"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const lib_1 = require("../lib");
const FIXTURES = path.join(__dirname, 'fixtures');
test('empty assembly', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'empty'));
    expect(assembly.artifacts).toEqual([]);
    expect(assembly.runtime).toEqual({ libraries: {} });
    expect(assembly.stacks).toEqual([]);
    expect(assembly.version).toEqual('0.0.0');
    expect(assembly.manifest).toMatchSnapshot();
    expect(assembly.tree()).toBeUndefined();
});
test('assembly with a single cloudformation stack and tree metadata', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'single-stack'));
    expect(assembly.artifacts).toHaveLength(2);
    expect(assembly.stacks).toHaveLength(1);
    expect(assembly.manifest.missing).toBeUndefined();
    expect(assembly.runtime).toEqual({ libraries: {} });
    const stack = assembly.stacks[0];
    expect(stack.manifest).toMatchSnapshot();
    expect(stack.assets).toHaveLength(0);
    expect(stack.dependencies).toEqual([]);
    expect(stack.environment).toEqual({ account: '37736633', region: 'us-region-1', name: 'aws://37736633/us-region-1' });
    expect(stack.template).toEqual({ Resources: { MyBucket: { Type: 'AWS::S3::Bucket' } } });
    expect(stack.messages).toEqual([]);
    expect(stack.manifest.metadata).toEqual(undefined);
    expect(stack.originalName).toEqual('MyStackName');
    expect(stack.stackName).toEqual('MyStackName');
    expect(stack.id).toEqual('MyStackName');
    const treeArtifact = assembly.tree();
    expect(treeArtifact).toBeDefined();
    expect(treeArtifact.file).toEqual('foo.tree.json');
    expect(treeArtifact.manifest).toMatchSnapshot();
});
test('assembly with invalid tree metadata', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-manifest-type-tree'));
    expect(() => assembly.tree()).toThrow(/Multiple artifacts/);
});
test('assembly with tree metadata having no file property specified', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'tree-no-file-property'))).toThrow(/Invalid assembly manifest/);
});
test('assembly with cloudformation artifact having no environment property specified', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-manifest-type-cloudformation'))).toThrow(/Invalid CloudFormation stack artifact/);
});
test('assembly with missing context', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'missing-context'));
    expect(assembly.manifest.missing).toMatchSnapshot();
});
test('assembly with multiple stacks', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'multiple-stacks'));
    expect(assembly.stacks).toHaveLength(2);
    expect(assembly.artifacts).toHaveLength(2);
});
test('fails for invalid environment format', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-env-format')))
        .toThrow('Unable to parse environment specification');
});
test('fails if stack artifact does not have properties', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'stack-without-params')))
        .toThrow('Invalid CloudFormation stack artifact. Missing \"templateFile\" property in cloud assembly manifest');
});
test('messages', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'messages'));
    expect(assembly.stacks[0].messages).toMatchSnapshot();
});
test('assets', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'assets'));
    expect(assembly.stacks[0].assets).toMatchSnapshot();
});
test('can-read-0.36.0', () => {
    // WHEN
    new lib_1.CloudAssembly(path.join(FIXTURES, 'single-stack-0.36'));
    // THEN: no exception
    expect(true).toBeTruthy();
});
test('dependencies', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'depends'));
    expect(assembly.stacks).toHaveLength(4);
    // expect stacks to be listed in topological order
    expect(assembly.stacks.map(s => s.id)).toEqual(['StackA', 'StackD', 'StackC', 'StackB']);
    expect(assembly.stacks[0].dependencies).toEqual([]);
    expect(assembly.stacks[1].dependencies).toEqual([]);
    expect(assembly.stacks[2].dependencies.map(x => x.id)).toEqual(['StackD']);
    expect(assembly.stacks[3].dependencies.map(x => x.id)).toEqual(['StackC', 'StackD']);
});
test('fails for invalid dependencies', () => {
    expect(() => new lib_1.CloudAssembly(path.join(FIXTURES, 'invalid-depends'))).toThrow('Artifact StackC depends on non-existing artifact StackX');
});
test('stack artifacts can specify an explicit stack name that is different from the artifact id', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'explicit-stack-name'));
    expect(assembly.getStackByName('TheStackName').stackName).toStrictEqual('TheStackName');
    expect(assembly.getStackByName('TheStackName').id).toStrictEqual('stackid1');
    // deprecated but still test
    expect(assembly.getStack('TheStackName').stackName).toStrictEqual('TheStackName');
    expect(assembly.getStack('TheStackName').id).toStrictEqual('stackid1');
});
test('getStackByName fails if there are multiple stacks with the same name', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'multiple-stacks-same-name'));
    expect(() => assembly.getStackByName('the-physical-name-of-the-stack')).toThrow(/There are multiple stacks with the stack name \"the-physical-name-of-the-stack\" \(stack1\,stack2\)\. Use \"getStackArtifact\(id\)\" instead/);
});
test('getStackArtifact retrieves a stack by artifact id', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'multiple-stacks-same-name'));
    expect(assembly.getStackArtifact('stack1').stackName).toEqual('the-physical-name-of-the-stack');
    expect(assembly.getStackArtifact('stack2').stackName).toEqual('the-physical-name-of-the-stack');
    expect(assembly.getStackArtifact('stack2').id).toEqual('stack2');
    expect(assembly.getStackArtifact('stack1').id).toEqual('stack1');
});
test('displayName shows both artifact ID and stack name if needed', () => {
    const a1 = new lib_1.CloudAssembly(path.join(FIXTURES, 'multiple-stacks-same-name'));
    expect(a1.getStackArtifact('stack1').displayName).toStrictEqual('stack1 (the-physical-name-of-the-stack)');
    expect(a1.getStackArtifact('stack2').displayName).toStrictEqual('stack2 (the-physical-name-of-the-stack)');
    const a2 = new lib_1.CloudAssembly(path.join(FIXTURES, 'single-stack'));
    const art1 = a2.getStackArtifact('MyStackName');
    const art2 = a2.getStackByName('MyStackName');
    expect(art1).toBe(art2);
    expect(art1.displayName).toBe('MyStackName');
    expect(art1.id).toBe('MyStackName');
    expect(art1.stackName).toBe('MyStackName');
});
test('can read assembly with asset manifest', () => {
    const assembly = new lib_1.CloudAssembly(path.join(FIXTURES, 'asset-manifest'));
    expect(assembly.stacks).toHaveLength(1);
    expect(assembly.artifacts).toHaveLength(2);
});
//# sourceMappingURL=data:application/json;base64,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