"""
Core Cython module for high-performance pathfinding operations.

This module provides optimized data structures and utility functions for
graph-based pathfinding algorithms, including binary heap implementation,
coordinate transformations, and intermediate step calculations for complex
movement patterns.

The module is designed for maximum performance with:
- nogil sections for true parallelization potential
- Efficient memory management with C++ vectors
- Optimized algorithms for spatial operations
- Direct memory access patterns for cache efficiency

Performance Notes:
    - All critical path operations are implemented without Python overhead
    - Memory views provide zero-copy access to NumPy arrays
    - C++ vectors offer dynamic sizing with minimal allocation overhead
    - Intermediate step calculations use floating-point precision for accuracy
"""

# cython: language_level=3, boundscheck=False, wraparound=False
# cython: initializedcheck=False, cdivision=True, nonecheck=False

import numpy as np
cimport numpy as np
from libc.math cimport sqrt, sqrtf, floor, ceil, abs, logf
from libcpp.vector cimport vector
from libcpp cimport bool
import psutil
import sys
from libc.stdint cimport UINT32_MAX, UINT64_MAX

# Define INFINITY for float32
cdef float INF_F32 = 1e38

# ==================== SYSTEM RESOURCE MANAGEMENT ====================

cdef SystemLimits get_system_limits() except*:
    """
    Query system resources and determine safe operating limits.
    
    This function analyzes the current system's available memory and processing
    capabilities to establish safe boundaries for pathfinding operations. It
    prevents memory exhaustion and ensures efficient resource utilization.
    
    Returns:
        SystemLimits structure containing:
        - max_memory_bytes: Total safe memory allocation limit
        - available_memory_bytes: Currently available memory
        - max_array_size: Maximum safe array size
        - max_path_length: Maximum path vertices to prevent overflow
        - max_buckets: Maximum delta-stepping buckets
        - max_iterations: Safe iteration limit
        - num_cores: Available CPU cores
    
    Safety Factors:
        - Memory usage capped at 85% of available to prevent system instability
        - Array sizes respect Python's sys.maxsize limitations
        - Iteration counts bounded to prevent infinite loops
    """
    cdef SystemLimits limits
    cdef object mem_info
    cdef uint64_t total_memory, available_memory
    cdef double safety_factor = 0.85
    cdef uint64_t MAX_SAFE_SIZE = <uint64_t>2147483647

    mem_info = psutil.virtual_memory()
    total_memory = <uint64_t>mem_info.total
    available_memory = <uint64_t>mem_info.available

    limits.max_memory_bytes = <uint64_t>(total_memory * safety_factor)
    limits.available_memory_bytes = <uint64_t>(available_memory * safety_factor)

    limits.max_array_size = min(
        limits.available_memory_bytes // 16,
        <uint64_t>sys.maxsize if sys.maxsize > 0 else MAX_SAFE_SIZE,
        UINT64_MAX // 2
    )

    limits.max_path_length = min(
        limits.max_array_size,
        <uint64_t>sys.maxsize if sys.maxsize > 0 else MAX_SAFE_SIZE
    )

    cdef uint64_t bucket_calc = limits.available_memory_bytes // (1024 * sizeof(uint64_t))
    if bucket_calc > UINT32_MAX // 2:
        limits.max_buckets = UINT32_MAX // 2
    else:
        limits.max_buckets = <uint32_t>bucket_calc

    if sys.maxsize < 2147483647:
        limits.max_iterations = sys.maxsize // 2
    else:
        limits.max_iterations = 2147483647

    if limits.max_array_size < <uint64_t>limits.max_iterations:
        limits.max_iterations = <int>min(limits.max_array_size, <uint64_t>2147483647)

    num_cores = psutil.cpu_count(logical=True)
    # Limit amount of cores to 12 as it does not increase performance further if more
    # than 12 cores are used!
    limits.num_cores = 12 if num_cores > 12 else num_cores

    return limits


cdef uint32_t calculate_initial_bucket_size(uint64_t total_cells, SystemLimits* limits) noexcept nogil:
    """
    Calculate optimal initial bucket size based on problem size and system resources.
    
    This function determines the initial memory allocation for delta-stepping
    buckets, balancing between memory efficiency and reallocation overhead.
    
    Parameters:
        total_cells: Total number of cells in the raster
        limits: System resource limits
    
    Returns:
        Initial bucket count (between 1000 and max_buckets/10)
    
    Heuristics:
        - Base size: 1% of total cells (capped at 100k)
        - Memory-based limit: available_memory / (10000 * sizeof(uint64_t))
        - Minimum: 1000 buckets for small problems
        - Maximum: 10% of system maximum to leave headroom
    """
    cdef uint32_t size
    cdef uint64_t memory_based_size

    size = <uint32_t>min(total_cells // 100, 100000)

    memory_based_size = limits.available_memory_bytes // (10000 * sizeof(uint64_t))
    if memory_based_size < size:
        size = <uint32_t>memory_based_size

    if size < 1000:
        size = 1000
    if size > limits.max_buckets // 10:
        size = limits.max_buckets // 10

    return size


cdef int calculate_thread_buffer_capacity(uint64_t total_cells, int num_threads, SystemLimits* limits) noexcept nogil:
    """
    Calculate per-thread buffer capacity for parallel edge relaxation.
    
    This function determines the optimal buffer size for each thread during
    parallel pathfinding operations, ensuring efficient memory usage without
    excessive allocation overhead.
    
    Parameters:
        total_cells: Total cells in the problem
        num_threads: Number of parallel threads
        limits: System resource limits
    
    Returns:
        Buffer capacity per thread (at least 256 elements)
    
    Algorithm:
        - Divides available memory fairly among threads
        - Considers cells-per-thread workload
        - Applies dynamic minimum based on problem size
        - Enforces hard minimum of 256 for small problems
    """
    cdef uint64_t per_thread_memory
    cdef uint64_t capacity_64
    cdef int capacity
    cdef int MAX_INT = 2147483647
    cdef int dynamic_minimum
    cdef uint64_t cells_per_thread

    if num_threads <= 0:
        num_threads = 1

    cells_per_thread = total_cells // num_threads

    dynamic_minimum = max(
        256,
        min(
            <int>(cells_per_thread // 100),
            65536
        )
    )

    per_thread_memory = limits.available_memory_bytes // (num_threads * 4)

    capacity_64 = min(
        per_thread_memory // 16,
        total_cells // (num_threads * 10),
        <uint64_t>MAX_INT // 2
    )

    if capacity_64 > limits.max_array_size // num_threads:
        capacity_64 = limits.max_array_size // num_threads

    if capacity_64 > <uint64_t>MAX_INT // 2:
        capacity = MAX_INT // 2
    else:
        capacity = <int>capacity_64

    if capacity < dynamic_minimum:
        capacity = dynamic_minimum

    if capacity < 256:
        capacity = 256

    return capacity

# ==================== INTERMEDIATE STEP CALCULATIONS ====================

cdef vector[IntermediatePoint] _calculate_intermediate_steps_cython(int dr, int dc) noexcept nogil:
    """
    Calculate intermediate steps for movement between non-adjacent cells.

    This function determines all cells that must be traversed when moving
    from one raster cell to another that is not immediately adjacent. The
    algorithm ensures that paths remain connected and accounts for diagonal
    movements that might otherwise skip over obstacle cells.

    Algorithm Details:
        - For simple moves (distance <= 1): No intermediate steps needed
        - For single-step diagonals: Add orthogonal components separately
        - For complex moves: Use linear interpolation with floor/ceil sampling

    Parameters (all nogil compatible):
        dr: Row displacement (-∞ to +∞, typically -10 to +10)
        dc: Column displacement (-∞ to +∞, typically -10 to +10)

    Returns:
        Vector of IntermediatePoint structs representing the path steps

    Performance Notes:
        - Executes entirely without GIL for maximum concurrency potential
        - Uses efficient C++ vector for dynamic result storage
        - Floating-point calculations maintain sub-pixel accuracy
    """
    cdef vector[IntermediatePoint] result
    cdef IntermediatePoint point
    cdef int abs_dr = abs(dr)
    cdef int abs_dc = abs(dc)
    cdef int sum_abs = abs_dr + abs_dc
    cdef int k, p
    cdef double dr_k, dc_k, ddr, ddc, dk, dp
    cdef int8_t floor_dr, floor_dc, ceil_dr, ceil_dc

    if sum_abs <= 1:
        # Adjacent or same cell - no intermediate steps required
        pass
    elif max(abs_dr, abs_dc) == 1:
        # Single diagonal step - decompose into orthogonal components
        # This ensures we check cells along both row and column directions
        point.dr = <int8_t>dr
        point.dc = 0
        result.push_back(point)

        point.dr = 0
        point.dc = <int8_t>dc
        result.push_back(point)
    else:
        # Complex movement requiring linear interpolation
        # Sample points along the line between source and destination
        k = max(abs_dr, abs_dc)  # Number of major steps
        ddr = <double>dr
        ddc = <double>dc
        dk = <double>k

        for p in range(1, k):
            dp = <double>p
            # Calculate fractional position along the movement vector
            dr_k = (dp * ddr) / dk
            dc_k = (dp * ddc) / dk

            # Add floor approximation (conservative path)
            floor_dr = <int8_t>floor(dr_k)
            floor_dc = <int8_t>floor(dc_k)
            point.dr = floor_dr
            point.dc = floor_dc
            result.push_back(point)

            # Add ceiling approximation if different (ensures connectivity)
            ceil_dr = <int8_t>ceil(dr_k)
            ceil_dc = <int8_t>ceil(dc_k)
            if floor_dr != ceil_dr or floor_dc != ceil_dc:
                point.dr = ceil_dr
                point.dc = ceil_dc
                result.push_back(point)
    return result


cdef inline double _get_cost_factor_cython(int dr, int dc, int intermediates_count) noexcept nogil:
    """
    Calculate movement cost factor incorporating distance and path complexity.

    The cost factor adjusts the base movement cost to account for both the
    geometric distance traveled and the complexity introduced by intermediate
    steps. This ensures that longer, more complex movements are properly
    penalized relative to simpler alternatives.

    Formula: distance / (2.0 + intermediate_steps_count)
    - Numerator: Euclidean distance between source and destination
    - Denominator: Base penalty (2.0) plus complexity penalty

    Parameters (all nogil compatible):
        dr: Row displacement for the movement
        dc: Column displacement for the movement
        intermediates_count: Number of intermediate steps required

    Returns:
        Multiplicative factor to apply to raw cell costs (always positive)

    Performance Notes:
        - Inlined for zero function call overhead
        - Single square root operation for distance calculation
        - Result cached in StepData structure to avoid recalculation
    """
    cdef double distance = sqrt(<double>(dr * dr + dc * dc))
    cdef double divisor = 2.0 + <double>intermediates_count
    return distance / divisor


cdef inline float _get_cost_factor_cython_f32(int dr, int dc, int intermediates_count) noexcept nogil:
    """
    Float32 version of cost factor calculation for memory-efficient operations.
    
    Identical to _get_cost_factor_cython but returns float32 for delta-stepping
    algorithms that use single precision for performance.
    """
    cdef float distance = sqrtf(<float>(dr * dr + dc * dc))
    cdef float divisor = <float>2.0 + <float>intermediates_count
    return distance / divisor

# ==================== CACHED STEP PRECOMPUTATION ====================

cdef vector[CachedStepData] precompute_cached_steps(np.ndarray[int8_t, ndim=2] steps_arr):
    """
    Precompute and cache intermediate steps for all movement directions.
    
    This function calculates intermediate steps once during initialization
    and caches them for reuse throughout the pathfinding process. This
    significantly improves performance by avoiding repeated calculations.
    
    Parameters:
        steps_arr: 2D array where each row contains [dr, dc] for one direction
    
    Returns:
        Vector of CachedStepData containing precomputed intermediate steps
    
    Performance Impact:
        - Reduces intermediate calculation overhead by 10-20x
        - Memory trade-off: ~100-500 bytes per movement direction
        - One-time cost amortized over thousands of edge relaxations
    """
    cdef vector[CachedStepData] cached_steps
    cdef CachedStepData step_cache
    cdef int num_steps, s, dr, dc
    cdef vector[IntermediatePoint] intermediates

    num_steps = <int>steps_arr.shape[0]
    cached_steps.reserve(<size_t>num_steps)

    for s in range(num_steps):
        dr = steps_arr[s, 0]
        dc = steps_arr[s, 1]

        intermediates = _calculate_intermediate_steps_cython(dr, dc)
        step_cache.intermediates = intermediates
        step_cache.intermediate_count = <int>intermediates.size()

        cached_steps.push_back(step_cache)

    return cached_steps


cdef int check_path_cached(const vector[IntermediatePoint]& cached_intermediates,
                          int current_row, int current_col,
                          const uint8_t[:, :] exclude_mask, const uint16_t[:, :] raster,
                          int rows, int cols, float* total_cost) except -1 nogil:
    """
    Validate movement path using cached intermediate steps.
    
    This optimized version of check_path uses precomputed intermediate steps
    to validate path traversability and calculate costs more efficiently.
    
    Parameters:
        cached_intermediates: Precomputed intermediate steps for this direction
        current_row: Starting row position
        current_col: Starting column position
        exclude_mask: 2D traversability mask
        raster: 2D cost raster
        rows: Total rows in raster
        cols: Total columns in raster
        total_cost: Output parameter for intermediate costs
    
    Returns:
        1 if path is valid, 0 if blocked
    
    Performance Notes:
        - 2-3x faster than recalculating intermediates
        - Direct vector access without allocation
        - Early termination on first invalid cell
    """
    cdef float cost = 0.0
    cdef int i, int_row, int_col, num_intermediates
    cdef IntermediatePoint point

    num_intermediates = <int>cached_intermediates.size()

    for i in range(num_intermediates):
        point = cached_intermediates[i]
        int_row = current_row + point.dr
        int_col = current_col + point.dc

        if int_row < 0 or int_row >= rows or int_col < 0 or int_col >= cols:
            return 0

        if exclude_mask[int_row, int_col] == 0:
            return 0

        cost += <float>raster[int_row, int_col]

    total_cost[0] = cost
    return 1


cdef vector[StepData] precompute_directions_optimized(np.ndarray[int8_t, ndim=2] steps_arr,
                                                     const vector[CachedStepData]& cached_steps):
    """
    Create optimized direction data using cached intermediate steps.
    
    This function combines step directions with precomputed cost factors,
    creating a complete dataset for efficient edge relaxation during
    pathfinding operations.
    
    Parameters:
        steps_arr: Raw movement directions
        cached_steps: Precomputed intermediate step data
    
    Returns:
        Vector of StepData with directions and cost factors
    
    Memory Layout:
        - Each StepData: ~16 bytes (2 ints + 1 double)
        - Typical total: ~128-512 bytes for 8-32 directions
        - Cache-friendly sequential access pattern
    """
    cdef vector[StepData] directions
    cdef StepData direction
    cdef int s, dr, dc, steps_count

    steps_count = <int>steps_arr.shape[0]
    directions.reserve(<size_t>steps_count)

    for s in range(steps_count):
        dr = steps_arr[s, 0]
        dc = steps_arr[s, 1]

        direction.dr = dr
        direction.dc = dc
        direction.cost_factor = _get_cost_factor_cython_f32(dr, dc, cached_steps[s].intermediate_count)

        directions.push_back(direction)

    return directions

# ==================== BINARY HEAP IMPLEMENTATION ====================

cdef inline int heap_init(BinaryHeap* heap) except -1 nogil:
    """
    Initialize an empty binary heap with reasonable default capacity.

    This function prepares a heap for use by clearing any existing contents
    and pre-allocating memory for expected usage patterns. The initial
    capacity of 1000 nodes is chosen based on typical pathfinding scenarios.

    Parameters:
        heap: Pointer to BinaryHeap structure to initialize

    Returns:
        0 on success (error handling may be added in future versions)

    Performance Notes:
        - Reserve operation minimizes memory allocations during heap growth
        - Clear operation is O(1) for vectors
        - No memory allocation failures in current implementation
    """
    heap.nodes.clear()
    heap.nodes.reserve(1000)
    return 0


cdef inline bool heap_empty(const BinaryHeap* heap) noexcept nogil:
    """
    Check if the binary heap contains any nodes.

    Parameters:
        heap: Pointer to BinaryHeap structure to check

    Returns:
        True if heap is empty, False if it contains nodes

    Performance Notes:
        - O(1) operation using vector's size method
        - Marked noexcept for maximum compiler optimization
    """
    return heap.nodes.size() == 0


cdef inline PQNode heap_top(const BinaryHeap* heap) noexcept nogil:
    """
    Retrieve the minimum priority node without removing it from the heap.

    In a min-heap, the root node (index 0) always contains the element
    with the smallest priority value. This operation does not modify
    the heap structure.

    Parameters:
        heap: Pointer to BinaryHeap structure (must not be empty)

    Returns:
        PQNode with the minimum priority value

    Warning:
        Calling this function on an empty heap results in undefined behavior.
        Always check heap_empty() first in production code.

    Performance Notes:
        - O(1) operation - simple array access
        - No bounds checking for maximum performance
    """
    return heap.nodes[0]


cdef inline int heap_push(BinaryHeap* heap, uint32_t idx, double priority) except -1 nogil:
    """
    Insert a new node into the binary heap maintaining heap property.

    This function adds a node to the heap and restores the min-heap property
    by bubbling the new element up the tree until it reaches its correct
    position. The heap property ensures parent nodes always have priority
    values less than or equal to their children.

    Algorithm:
        1. Add new node at the end of the heap (next available position)
        2. Compare with parent and swap if new node has lower priority
        3. Repeat until heap property is satisfied or root is reached

    Parameters:
        heap: Pointer to BinaryHeap structure to modify
        idx: Graph node index to insert
        priority: Priority value (lower values = higher priority)

    Returns:
        0 on success (error handling may be added in future versions)

    Time Complexity: O(log n) where n is the number of nodes in heap
    """
    cdef PQNode node
    node.index = idx
    node.priority = priority
    heap.nodes.push_back(node)

    # Bubble up to maintain heap property
    cdef npy_intp pos = heap.nodes.size() - 1
    cdef npy_intp parent
    cdef PQNode temp

    while pos > 0:
        parent = (pos - 1) // 2
        if heap.nodes[parent].priority <= heap.nodes[pos].priority:
            break  # Heap property satisfied

        # Swap with parent
        temp = heap.nodes[pos]
        heap.nodes[pos] = heap.nodes[parent]
        heap.nodes[parent] = temp
        pos = parent

    return 0


cdef inline int heap_pop(BinaryHeap* heap) except -1 nogil:
    """
    Remove the minimum priority node from the heap maintaining heap property.

    This function removes the root node (minimum priority) and restores the
    heap property by moving the last element to the root and sifting it down
    to its correct position. This is the standard algorithm for heap deletion.

    Algorithm:
        1. Replace root with the last element in the heap
        2. Remove the last element (now duplicated at root)
        3. Sift down the new root until heap property is satisfied

    Parameters:
        heap: Pointer to BinaryHeap structure to modify

    Returns:
        0 on success, 1 if heap was empty

    Time Complexity: O(log n) where n is the number of nodes in heap

    Performance Notes:
        - Early returns for empty or single-element heaps
        - Efficient sift-down with minimal comparisons
    """
    if heap.nodes.size() == 0:
        return 1  # Error: empty heap

    if heap.nodes.size() > 1:
        # Move last element to root position
        heap.nodes[0] = heap.nodes[heap.nodes.size() - 1]

    heap.nodes.pop_back()

    if heap.nodes.size() <= 1:
        return 0  # No sifting needed for 0 or 1 elements

    # Sift down to restore heap property
    cdef npy_intp pos = 0
    cdef npy_intp left, right, smallest
    cdef npy_intp heap_size = heap.nodes.size()
    cdef PQNode temp

    while True:
        left = 2 * pos + 1
        right = 2 * pos + 2
        smallest = pos

        # Find the smallest among current node and its children
        if (left < heap_size and
                heap.nodes[left].priority < heap.nodes[smallest].priority):
            smallest = left

        if (right < heap_size and
                heap.nodes[right].priority < heap.nodes[smallest].priority):
            smallest = right

        if smallest == pos:
            break  # Heap property satisfied

        # Swap with smallest child
        temp = heap.nodes[pos]
        heap.nodes[pos] = heap.nodes[smallest]
        heap.nodes[smallest] = temp
        pos = smallest
    return 0

# ==================== INDEX CONVERSION FUNCTIONS ====================

cdef inline uint32_t ravel_index(int row, int col, int cols) noexcept nogil:
    """
    Convert 2D raster coordinates to 1D graph node index.

    This function performs the standard row-major order conversion from
    2D array indices to a linear index. This mapping is essential for
    representing raster cells as graph nodes.

    Formula: linear_index = row * cols + col

    Parameters:
        row: Row index in the raster (0-based)
        col: Column index in the raster (0-based)
        cols: Total number of columns in the raster

    Returns:
        Linear node index suitable for graph algorithms

    Performance Notes:
        - Single multiplication and addition operation
        - Inlined for zero function call overhead
        - No bounds checking for maximum performance
    """
    return <uint32_t>(row * cols + col)


cdef inline int unravel_index(uint32_t idx, int cols, npy_intp* row, npy_intp* col) except -1 nogil:
    """
    Convert 1D graph node index back to 2D raster coordinates.

    This function performs the inverse of ravel_index, converting a linear
    graph node index back to row and column coordinates in the original raster.
    The results are written to the provided pointer locations.

    Formula:
        row = linear_index // cols
        col = linear_index % cols

    Parameters:
        idx: Linear node index from graph algorithms
        cols: Total number of columns in the raster
        row: Pointer to store the calculated row index
        col: Pointer to store the calculated column index

    Returns:
        0 on success (error handling may be added in future versions)

    Performance Notes:
        - Single division and modulo operation
        - Uses pointer outputs to avoid return value copying
        - Inlined for zero function call overhead
    """
    row[0] = idx // cols
    col[0] = idx % cols
    return 0

# ==================== PATH VALIDATION ====================

cdef int check_path(int dr, int dc, int current_row, int current_col,
                    const uint8_t[:, :] exclude_mask, const uint16_t[:, :] raster,
                    int rows, int cols, double* total_cost) except -1 nogil:
    """
    Validate a movement path and calculate intermediate step costs.

    This function checks if a movement from the current position is valid by
    examining all intermediate cells that would be traversed. If any
    intermediate cell is out of bounds or blocked, the movement is invalid.
    If valid, the total cost of traversing intermediate cells is calculated.

    Parameters:
        dr: Row displacement for the movement
        dc: Column displacement for the movement
        current_row: Starting row position
        current_col: Starting column position
        exclude_mask: 2D mask indicating traversable cells (1=ok, 0=blocked)
        raster: 2D cost raster for calculating traversal costs
        rows: Total number of rows in the raster
        cols: Total number of columns in the raster
        total_cost: Pointer to store the calculated intermediate costs

    Returns:
        1 if path is valid, 0 if path is blocked or out of bounds

    Performance Notes:
        - Executes without GIL for maximum concurrency potential
        - Early termination on first invalid cell encountered
        - Direct memory access patterns for cache efficiency
    """
    cdef double cost = 0.0
    cdef int i, int_row, int_col

    # Get intermediate steps for this movement
    cdef vector[IntermediatePoint] intermediates = (
        _calculate_intermediate_steps_cython(dr, dc))
    cdef IntermediatePoint point

    # Check each intermediate point along the path
    for i in range(intermediates.size()):
        point = intermediates[i]
        int_row = current_row + point.dr
        int_col = current_col + point.dc

        # Validate bounds and traversability
        if (int_row < 0 or int_row >= rows or
                int_col < 0 or int_col >= cols or
                exclude_mask[int_row, int_col] == 0):
            return 0  # Invalid path

        # Accumulate cost of traversing this intermediate cell
        cost += raster[int_row, int_col]

    # Path is valid - return total intermediate cost
    total_cost[0] = cost
    return 1


cdef vector[StepData] precompute_directions(np.ndarray[int8_t, ndim=2] steps_arr):
    """
    Precompute movement data for all possible directions in the neighborhood.

    This function calculates cost factors and intermediate step counts for
    each movement direction during initialization. Precomputing this data
    avoids expensive recalculation during pathfinding and significantly
    improves runtime performance.

    Parameters:
        steps_arr: 2D array where each row contains [dr, dc] for one direction

    Returns:
        Vector of StepData structures with precomputed movement information

    Performance Notes:
        - Single pass computation during initialization
        - Results cached for entire pathfinding session
        - Memory pre-allocation with reserve() for efficiency
    """
    cdef vector[StepData] directions
    cdef StepData direction
    cdef int s, dr, dc
    cdef int intermediates_count
    cdef int steps_count = <int>steps_arr.shape[0]

    directions.reserve(steps_count)

    for s in range(steps_count):
        dr = steps_arr[s, 0]
        dc = steps_arr[s, 1]

        # Count intermediate steps for this direction
        intermediates_count = <int>(
            _calculate_intermediate_steps_cython(dr, dc).size())

        # Store precomputed direction data
        direction.dr = dr
        direction.dc = dc
        direction.cost_factor = <float>_get_cost_factor_cython(
            dr, dc, intermediates_count)

        directions.push_back(direction)

    return directions

# ==================== UTILITY FUNCTIONS ====================

cpdef np.ndarray[uint8_t, ndim=2] create_exclude_mask(
        np.ndarray[uint16_t, ndim=2] raster_arr, uint16_t max_value):
    """
    Create a binary mask identifying traversable cells in the raster.

    This function generates a boolean mask where 1 indicates a traversable cell
    and 0 indicates an obstacle or excluded area. Cells with the maximum value
    are treated as barriers and marked as non-traversable.

    Parameters:
        raster_arr: 2D numpy array containing cost values for each cell
        max_value: Value representing obstacles/barriers (typically 65535)

    Returns:
        2D numpy array of uint8 values (1=traversable, 0=obstacle)

    Performance Notes:
        - Uses efficient nested loops for direct memory access
        - Single pass through the raster data
        - Minimal memory allocation with pre-sized output array
    """
    cdef int rows = <int>raster_arr.shape[0]
    cdef int cols = <int>raster_arr.shape[1]
    cdef uint16_t[:, :] raster = raster_arr

    # Initialize mask with all cells marked as traversable
    cdef np.ndarray[uint8_t, ndim=2] exclude_mask_arr = np.ones((rows, cols),
                                                                dtype=np.uint8)
    cdef uint8_t[:, :] exclude_mask = exclude_mask_arr

    cdef int i, j
    for i in range(rows):
        for j in range(cols):
            if raster[i, j] == max_value:
                exclude_mask[i, j] = 0  # Mark as obstacle

    return exclude_mask_arr


cpdef double path_cost(np.ndarray[uint64_t, ndim=1] path,
                       np.ndarray[uint16_t, ndim=2] raster_arr, uint64_t cols):
    """
    Calculate the total traversal cost for a given path through the raster.

    This utility function sums the raster costs of all cells in a path,
    providing the total cost metric for path comparison and analysis.

    Parameters:
        path: 1D array of linear indices representing the path sequence
        raster_arr: 2D cost raster containing per-cell traversal costs
        cols: Number of columns in raster (for index conversion)

    Returns:
        Total cost as sum of individual cell costs along the path

    Performance Notes:
        - Linear time complexity O(path_length)
        - Efficient coordinate conversion using integer arithmetic
        - Direct memory access for cost lookup
    """
    cdef int i
    cdef uint64_t idx, row, col
    cdef double cost = 0.0
    cdef int path_len = <int> path.shape[0]

    for i in range(path_len):
        idx = path[i]
        row = idx // cols
        col = idx % cols
        cost += <double> raster_arr[row, col]

    return cost


cpdef double path_cost_uint32(np.ndarray[uint32_t, ndim=1] path,
                              np.ndarray[uint16_t, ndim=2] raster_arr, int cols):
    """
    Calculate the total traversal cost for a given path through the raster.

    This is the uint32_t version for compatibility with Dijkstra algorithms
    that use 32-bit vertex indices.

    Parameters:
        path: 1D array of linear indices (uint32) representing the path sequence
        raster_arr: 2D cost raster containing per-cell traversal costs
        cols: Number of columns in raster (int, matching Dijkstra's internal type)

    Returns:
        Total cost as sum of individual cell costs along the path
    """
    cdef int i
    cdef uint32_t idx
    cdef int row, col
    cdef double cost = 0.0
    cdef int path_len = <int> path.shape[0]

    for i in range(path_len):
        idx = path[i]
        row = <int> (idx // cols)
        col = <int> (idx % cols)
        cost += <double> raster_arr[row, col]

    return cost

# ==================== CIRCULAR BUFFER UTILITIES ====================

cdef inline size_t get_circular_index(size_t logical_bucket, size_t buffer_size) noexcept nogil:
    """
    Map logical bucket index to physical position in circular buffer.
    
    Parameters:
        logical_bucket: The logical bucket index (can be any size)
        buffer_size: Size of the circular buffer (must be power of 2)
    
    Returns:
        Physical index in the circular buffer array
    """
    # Use bitwise AND for fast modulo with power-of-2 sizes
    return logical_bucket & (buffer_size - 1)


cdef inline bint is_bucket_in_window(size_t logical_bucket, size_t window_start,
                                     size_t window_size) noexcept nogil:
    """
    Check if a logical bucket index is within the current processing window.
    
    Parameters:
        logical_bucket: Bucket index to check
        window_start: Start of current processing window
        window_size: Size of the window (typically same as buffer size)
    
    Returns:
        True if bucket is in current window, False otherwise
    """
    return logical_bucket >= window_start and logical_bucket < window_start + window_size


cdef size_t round_up_power_of_two(size_t n) noexcept nogil:
    """
    Round up to the nearest power of two for efficient modulo operations.
    
    Parameters:
        n: Number to round up
    
    Returns:
        Next power of two >= n
    """
    if n <= 1:
        return 1

    # Bit manipulation trick to find next power of 2
    n -= 1
    n |= n >> 1
    n |= n >> 2
    n |= n >> 4
    n |= n >> 8
    n |= n >> 16
    if sizeof(size_t) > 4:  # 64-bit systems
        n |= n >> 32
    n += 1

    return n


