import pytest
from pathlib import Path
from emcp.fs import fs_mkdir
from emcp.utils import PathAlreadyExists, PathOutsideWorkDir


def test_fs_mkdir_simple(tmp_wd):
    """fs_mkdir creates a simple directory."""
    new_dir = tmp_wd / "new_directory"

    result = fs_mkdir(str(new_dir))

    assert new_dir.exists()
    assert new_dir.is_dir()
    assert "Successfully created directory" in result


def test_fs_mkdir_nested(tmp_wd):
    """fs_mkdir creates nested directories recursively."""
    nested = tmp_wd / "level1" / "level2" / "level3"

    result = fs_mkdir(str(nested))

    assert nested.exists()
    assert nested.is_dir()
    assert (tmp_wd / "level1").exists()
    assert (tmp_wd / "level1" / "level2").exists()


def test_fs_mkdir_already_exists(tmp_wd):
    """fs_mkdir raises error when path already exists."""
    existing = tmp_wd / "existing"
    existing.mkdir()

    with pytest.raises(PathAlreadyExists):
        fs_mkdir(str(existing))


def test_fs_mkdir_file_exists(tmp_wd):
    """fs_mkdir raises error when a file exists at path."""
    existing_file = tmp_wd / "file.txt"
    existing_file.write_text("content")

    with pytest.raises(PathAlreadyExists):
        fs_mkdir(str(existing_file))


def test_fs_mkdir_outside_wd(tmp_path):
    """fs_mkdir raises error for paths outside working directory."""
    outside = tmp_path / "outside"

    with pytest.raises(PathOutsideWorkDir):
        fs_mkdir(str(outside))

    assert not outside.exists()
