from setuptools import find_packages, setup

version_file = 'siatune/version.py'


def parse_requirements(fname: str = 'requirements.txt',
                       with_version: bool = True):
    """Parse the package dependencies listed in a requirements file but strips
    specific versioning information.

    Args:
        fname (str): path to requirements file
        with_version (bool, default=False): if True include version specs
    Returns:
        List[str]: list of requirements items
    CommandLine:
        python -c "import setup; print(setup.parse_requirements())"
    """
    import re
    import sys
    from os.path import exists
    require_fpath = fname

    def parse_line(line: str):
        """Parse information from a line in a requirements text file.

        Args:
            line (str): a line in a requirements text file
        Yields:
            Dict: a dictionary of parsed information
        """
        if line.startswith('-r '):
            # Allow specifying requirements in other files
            target = line.split(' ')[1]
            for info in parse_require_file(target):
                yield info
        else:
            info = {'line': line}
            if line.startswith('-e '):
                info['package'] = line.split('#egg=')[1]
            elif '@git+' in line:
                info['package'] = line
            else:
                # Remove versioning from the package
                pat = '(' + '|'.join(['>=', '==', '>']) + ')'
                parts = re.split(pat, line, maxsplit=1)
                parts = [p.strip() for p in parts]

                info['package'] = parts[0]
                if len(parts) > 1:
                    op, rest = parts[1:]
                    if ';' in rest:
                        # Handle platform specific dependencies
                        # http://setuptools.readthedocs.io/en/latest/setuptools.html#declaring-platform-specific-dependencies
                        version, platform_deps = map(str.strip,
                                                     rest.split(';'))
                        info['platform_deps'] = platform_deps
                    else:
                        version = rest  # NOQA
                    info['version'] = (op, version)
            yield info

    def parse_require_file(fpath: str):
        """Parse the package dependencies in a requirements file.

        Args:
            fpath (str): path to requirements file

        Yields:
            Dict: a dictionary of parsed information
        """
        with open(fpath, 'r') as f:
            for line in f.readlines():
                line = line.strip()
                if line and not line.startswith(('#', '-i')):
                    for info in parse_line(line):
                        yield info

    def gen_packages_items() -> str:
        """Generated package items.

        Yields:
            str: package items
        """

        if exists(require_fpath):
            for info in parse_require_file(require_fpath):
                parts = [info['package']]
                if with_version and 'version' in info:
                    parts.extend(info['version'])
                if not sys.version.startswith('3.4'):
                    # apparently package_deps are broken in 3.4
                    platform_deps = info.get('platform_deps')
                    if platform_deps is not None:
                        parts.append(';' + platform_deps)
                item = ''.join(parts)
                yield item

    packages = list(gen_packages_items())
    return packages


def get_version() -> str:
    """Get the version string from the package.

    Returns:
        str: The version string.
    """
    with open(version_file, 'r') as f:
        exec(compile(f.read(), version_file, 'exec'))
    return locals()['__version__']


if __name__ == '__main__':
    setup(
        name='siatune',
        version=get_version(),
        author='SIA',
        packages=find_packages(exclude=('configs', 'tools')),
        install_requires=parse_requirements('requirements/runtime.txt'),
        extras_require={
            'optional': parse_requirements('requirements/optional.txt'),
        },
    )
