import abc
from typing import Any, Generic, Literal, TypeVar, ParamSpec
from dataclasses import dataclass

from kirin import ir

Params = ParamSpec("Params")
RetType = TypeVar("RetType")


class BatchFuture(abc.ABC, Generic[RetType]):
    """Protocol for future objects that can be awaited."""

    MISSING_RESULT = Literal[None]

    @abc.abstractmethod
    def result(self, timeout: float | None) -> list[RetType]:
        """Returns the result of the future, blocking until it is available or the timeout expires."""

    @abc.abstractmethod
    def partial_result(self) -> list[RetType | MISSING_RESULT]:
        """Return all results that are available so far, or MISSING_RESULT for those that are not yet available."""

    @abc.abstractmethod
    def fetch(self) -> None:
        """Fetches the result of the future that are currently available."""

    def cancel(self):
        """Attempts to cancel the execution of the future."""
        raise NotImplementedError(
            f"cancel method not implemented for {self.__class__.__name__}"
        )

    def cancelled(self) -> bool:
        """Returns True if the future was cancelled, False otherwise."""
        raise NotImplementedError(
            f"cancelled method not implemented for {self.__class__.__name__}"
        )

    def done(self) -> bool:
        self.fetch()
        return not any(
            result is self.MISSING_RESULT for result in self.partial_result()
        )


ClassRetType = TypeVar("ClassRetType")


@dataclass
class AbstractTask(abc.ABC, Generic[Params, RetType]):
    kernel: ir.Method[Params, RetType]
    args: tuple[Any, ...]
    kwargs: dict[str, Any]


class DeviceTaskExpectMixin(AbstractTask):

    ObsType = TypeVar("ObsType")

    @abc.abstractmethod
    def expect(self, observable: ir.Method[[RetType], ObsType], shots: int) -> ObsType:
        """Returns the expectation value of the given observable after running the task."""


@dataclass
class AbstractRemoteTask(AbstractTask[Params, RetType]):
    """Base class for tasks generated by the devices."""

    def run(self, *, shots: int = 1, timeout: float | None = None) -> list[RetType]:
        return self.run_async(shots=shots).result(timeout=timeout)

    @abc.abstractmethod
    def run_async(self, *, shots: int = 1) -> BatchFuture[RetType]:
        """Executes the kernel asynchronously and returns a Future object."""


StateType = TypeVar("StateType")


@dataclass
class AbstractSimulatorTask(
    AbstractTask[Params, RetType], Generic[Params, RetType, StateType]
):
    """Base class for tasks generated by local simulators."""

    @abc.abstractmethod
    def run(self) -> RetType:
        """Executes the kernel and returns the result."""

    @property
    @abc.abstractmethod
    def state(self) -> StateType:
        """Returns the state of the simulator after running the task."""
