"""
The Low Autocorrelation Binary Sequence problem.

The `k`-step autocorrelations of a bit string `x` are defined as

`C_k(x) = sum(i = 0, n - k - 1, 1 if x[i] == x[i + k] else -1)`

The energy of the bit string, which is equivalent to the objective value of
this benchmark problem, is then:

`E(x) = sum(k = 1, n - 1, C_k(x) ** 2)`

The low-autocorrelation binary sequence (LABS) problem is to find a minimum
autocorrelated bit string is equivalent to minimizing `E`. We use the energy
directly as objective function. Others who define LABS as a maximization
problem try to maximize the merit factor `F` instead, which is equivalent to
minimizing the energy:

`F(x) = n² / (n * E(X))`

This problem is different from the other bit-string based problems, because we
only know the optimal solutions for few, smaller instances. For larger
instances, we only have lower bounds of the energy.

1. Carola Doerr and Furong Ye and Naama Horesh and Hao Wang and Ofer M. Shir
   and Thomas Bäck. Benchmarking Discrete Optimization Heuristics with
   IOHprofiler. Applied Soft Computing 88:106027, March 2020,
   doi: https://doi.org/10.1016/j.asoc.2019.106027.
2. Tom Packebusch and Stephan Mertens. Low Autocorrelation Binary Sequences.
   arXiv:1512.02475v2 [cond-mat.stat-mech] 24 Mar 2016
   https://arxiv.org/pdf/1512.02475
3. Burkhard Militzer, Michele Zamparelli, and Dieter Beule. Evolutionary
   Search for Low Autocorrelated Binary Sequences. IEEE Transactions on
   Evolutionary Computation. 2(1):34-39}. 1998.
   doi: https://doi.org/10.1109/4235.728212
4. Wai Ho Mow and Ke-Lin Du. New Evolutionary Search for Long Low
   Autocorrelation Binary Sequences. IEEE Transactions on Aerospace and
   Electronic Systems. 51(1):290-303. January 2015.
   doi: https://doi.org/10.1109/TAES.2014.130518
5. Borko Bošković, Franc Brglez, and Janez Brest. Low-Autocorrelation Binary
   Sequences: On Improved Merit Factors and Runtime Predictions to Achieve
   Them. arXiv:1406.5301v6 [cs.DS] 6 May 2017. See [6] below.
   https://arxiv.org/pdf/1406.5301
6. Borko Bošković, Franc Brglez, and Janez Brest. Low-Autocorrelation Binary
   Sequences: On Improved Merit Factors and Runtime Predictions to Achieve
   Them. Applied Soft Computing. 56:262-285, July 2017.
   doi: https://doi.org/10.1016/j.asoc.2017.02.024

This is code is part of the research work of Mr. Jiazheng ZENG (曾嘉政),
a Master's student at the Institute of Applied Optimization
(应用优化研究所) of the School of Artificial
Intelligence and Big Data (人工智能与大数据学院) at
Hefei University (合肥大学) in
Hefei, Anhui, China (中国安徽省合肥市) under the supervision of
Prof. Dr. Thomas Weise (汤卫思教授).
"""
from math import isfinite
from typing import Callable, Final, Iterator, cast

import numba  # type: ignore
import numpy as np
from pycommons.ds.sequences import merge_sorted_and_return_unique
from pycommons.math.int_math import try_int_div
from pycommons.types import check_int_range, type_error

from moptipy.examples.bitstrings.bitstring_problem import (
    BitStringProblem,
    default_scale_sequence,
)


@numba.njit(nogil=True, cache=True)
def labs(x: np.ndarray) -> int:
    """
    Calculate the objective value of the LABS problem.

    :param x: the flat numpy array representing the bitstring
    :return: the LABS objective

    # N = 3, E = 1, FN = 4.50000
    >>> labs(np.array([0, 0, 1]))
    1

    # N = 3, E = 1, FN = 4.50000
    >>> labs(np.array([1, 1, 0]))
    1

    # N = 4, E = 2, FN = 4.00000
    >>> labs(np.array([0, 1, 0, 0]))
    2

    # N = 4, E = 2, FN = 4.00000
    >>> labs(np.array([1, 0, 1, 1]))
    2

    # N = 5, E = 2, FN = 6.25000
    >>> labs(np.array([0, 0, 0, 1, 0]))
    2

    # N = 5, E = 2, FN = 6.25000
    >>> labs(np.array([1, 1, 1, 0, 1]))
    2

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([0, 1, 1, 1, 1, 0]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([1, 0, 0, 0, 0, 1]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([0, 1, 1, 0, 0, 0]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([1, 0, 0, 1, 1, 1]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([0, 0, 0, 1, 0, 0]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([1, 1, 1, 0, 1, 1]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([0, 1, 0, 1, 1, 1]))
    7

    # N = 6, E = 7, FN = 2.57143
    >>> labs(np.array([1, 0, 1, 0, 0, 0]))
    7

    # N = 7, E = 3, FN = 8.16667
    >>> labs(np.array([0, 1, 0, 0, 1, 1, 1]))
    3

    # N = 7, E = 3, FN = 8.16667
    >>> labs(np.array([1, 0, 1, 1, 0, 0, 0]))
    3

    # N = 8, E = 8, FN = 4.00000
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 1, 0]))
    8

    # N = 8, E = 8, FN = 4.00000
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 0, 1]))
    8

    # N = 8, E = 8, FN = 4.00000
    >>> labs(np.array([0, 0, 0, 1, 0, 1, 1, 0]))
    8

    # N = 8, E = 8, FN = 4.00000
    >>> labs(np.array([1, 1, 1, 0, 1, 0, 0, 1]))
    8

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([0, 0, 0, 1, 0, 1, 0, 0, 1]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([1, 1, 1, 0, 1, 0, 1, 1, 0]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 0, 1, 0]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 1, 0, 1]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 0, 1, 0]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 1, 0, 1]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([0, 0, 0, 1, 0, 1, 1, 0, 0]))
    12

    # N = 9, E = 12, FN = 3.37500
    >>> labs(np.array([1, 1, 1, 0, 1, 0, 0, 1, 1]))
    12

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 0, 0, 1, 0]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 1, 1, 0, 1]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 0, 1, 0]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 1, 0, 1]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([0, 0, 0, 1, 0, 1, 0, 0, 1, 1]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([1, 1, 1, 0, 1, 0, 1, 1, 0, 0]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([0, 0, 0, 0, 1, 0, 1, 1, 0, 0]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([1, 1, 1, 1, 0, 1, 0, 0, 1, 1]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 0, 1, 1, 0]))
    13

    # N = 10, E = 13, FN = 3.84615
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 1, 0, 0, 1]))
    13

    # N = 11, E = 5, FN = 12.10000
    >>> labs(np.array([0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 1]))
    5

    # N = 11, E = 5, FN = 12.10000
    >>> labs(np.array([1, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0]))
    5

    # N = 12, E = 10, FN = 7.20000
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0]))
    10

    # N = 12, E = 10, FN = 7.20000
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1]))
    10

    # N = 12, E = 10, FN = 7.20000
    >>> labs(np.array([0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0]))
    10

    # N = 12, E = 10, FN = 7.20000
    >>> labs(np.array([1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1]))
    10

    # N = 13, E = 6, FN = 14.08333
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0]))
    6

    # N = 13, E = 6, FN = 14.08333
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1]))
    6

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 1]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 0]))
    19

    # N = 14, E = 19, FN = 5.15789
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1]))
    19

    # N = 15, E = 15, FN = 7.50000
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1]))
    15

    # N = 15, E = 15, FN = 7.50000
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0]))
    15

    # N = 15, E = 15, FN = 7.50000
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1]))
    15

    # N = 15, E = 15, FN = 7.50000
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0]))
    15

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 0]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 1]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([0, 0, 0, 1, 0, 0, 0, 1, 1, 1, 0, 1, 0, 0, 1, 0]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([1, 1, 1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 1, 1, 0, 1]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0]))
    24

    # N = 16, E = 24, FN = 5.33333
    >>> labs(np.array([1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1]))
    24

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 1, 0]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0, 1]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1]))
    32

    # N = 17, E = 32, FN = 4.51562
    >>> labs(np.array([1, 1, 0, 1, 0, 0, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 0]))
    32

    # N = 18, E = 25, FN = 6.48000
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0]))
    25

    # N = 18, E = 25, FN = 6.48000
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1]))
    25

    # N = 18, E = 25, FN = 6.48000
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 1, 0, 0]))
    25

    # N = 18, E = 25, FN = 6.48000
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1, 1, 0, 0, 1, 1]))
    25

    # N = 19, E = 29, FN = 6.22414
    >>> labs(np.array([0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1,
    ...     1]))
    29

    # N = 19, E = 29, FN = 6.22414
    >>> labs(np.array([1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0,
    ...     0]))
    29

    # N = 20, E = 26, FN = 7.69231
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 1, 0, 0, 1, 1, 1, 0,
    ...     0, 1]))
    26

    # N = 20, E = 26, FN = 7.69231
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1,
    ...     1, 0]))
    26

    # N = 21, E = 26, FN = 8.48077
    >>> labs(np.array([0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0,
    ...     1, 1, 0]))
    26

    # N = 21, E = 26, FN = 8.48077
    >>> labs(np.array([1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1,
    ...     0, 0, 1]))
    26

    # N = 22, E = 39, FN = 6.20513
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 1, 1,
    ...     1, 0, 0, 0]))
    39

    # N = 22, E = 39, FN = 6.20513
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 1, 0, 0,
    ...     0, 1, 1, 1]))
    39

    # N = 22, E = 39, FN = 6.20513
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0,
    ...     1, 1, 0, 1]))
    39

    # N = 22, E = 39, FN = 6.20513
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1,
    ...     0, 0, 1, 0]))
    39

    # N = 22, E = 39, FN = 6.20513
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 0,
    ...     0, 0, 1, 1]))
    39

    # N = 22, E = 39, FN = 6.20513
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1,
    ...     1, 1, 0, 0]))
    39

    # N = 23, E = 47, FN = 5.62766
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1,
    ...     0, 0, 0, 1, 1]))
    47

    # N = 23, E = 47, FN = 5.62766
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0,
    ...     1, 1, 1, 0, 0]))
    47

    # N = 23, E = 47, FN = 5.62766
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0, 1,
    ...     1, 0, 0, 1, 0]))
    47

    # N = 23, E = 47, FN = 5.62766
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0,
    ...     0, 1, 1, 0, 1]))
    47

    # N = 23, E = 47, FN = 5.62766
    >>> labs(np.array([0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 1, 0, 1,
    ...     0, 0, 0, 1, 1]))
    47

    # N = 23, E = 47, FN = 5.62766
    >>> labs(np.array([1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0,
    ...     1, 1, 1, 0, 0]))
    47

    # N = 24, E = 36, FN = 8.00000
    >>> labs(np.array([0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0,
    ...     1, 1, 0, 1, 1, 0]))
    36

    # N = 24, E = 36, FN = 8.00000
    >>> labs(np.array([1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1,
    ...     0, 0, 1, 0, 0, 1]))
    36

    # N = 25, E = 36, FN = 8.68056
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1,
    ...     1, 0, 1, 1, 0, 0, 1]))
    36

    # N = 25, E = 36, FN = 8.68056
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0,
    ...     0, 1, 0, 0, 1, 1, 0]))
    36

    # N = 26, E = 45, FN = 7.51111
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0,
    ...     0, 1, 1, 1, 0, 0, 1, 1]))
    45

    # N = 26, E = 45, FN = 7.51111
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1,
    ...     1, 0, 0, 0, 1, 1, 0, 0]))
    45

    # N = 26, E = 45, FN = 7.51111
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 1, 0, 1,
    ...     0, 1, 1, 0, 1, 1, 0, 1]))
    45

    # N = 26, E = 45, FN = 7.51111
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0,
    ...     1, 0, 0, 1, 0, 0, 1, 0]))
    45

    # N = 26, E = 45, FN = 7.51111
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1,
    ...     0, 1, 1, 0, 1, 1, 0, 1]))
    45

    # N = 26, E = 45, FN = 7.51111
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0,
    ...     1, 0, 0, 1, 0, 0, 1, 0]))
    45

    # N = 27, E = 37, FN = 9.85135
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0,
    ...     1, 0, 0, 1, 0, 1, 1, 0, 1]))
    37

    # N = 27, E = 37, FN = 9.85135
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1,
    ...     0, 1, 1, 0, 1, 0, 0, 1, 0]))
    37

    # N = 28, E = 50, FN = 7.84000
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0,
    ...     1, 0, 0, 1, 0, 1, 1, 0, 1, 1]))
    50

    # N = 28, E = 50, FN = 7.84000
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1,
    ...     0, 1, 1, 0, 1, 0, 0, 1, 0, 0]))
    50

    # N = 29, E = 62, FN = 6.78226
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0,
    ...     1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0]))
    62

    # N = 29, E = 62, FN = 6.78226
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1,
    ...     0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1]))
    62

    # N = 29, E = 62, FN = 6.78226
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
    ...     1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0]))
    62

    # N = 29, E = 62, FN = 6.78226
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1,
    ...     0, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1]))
    62

    # N = 30, E = 59, FN = 7.62712
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1,
    ...     0, 1, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0]))
    59

    # N = 30, E = 59, FN = 7.62712
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0,
    ...     1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1]))
    59

    # N = 30, E = 59, FN = 7.62712
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1,
    ...     0, 1, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0]))
    59

    # N = 30, E = 59, FN = 7.62712
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0,
    ...     1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1]))
    59

    # N = 31, E = 67, FN = 7.17164
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1,
    ...     0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 1, 0, 1]))
    67

    # N = 31, E = 67, FN = 7.17164
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 0,
    ...     1, 1, 0, 0, 1, 0, 1, 0, 1, 1, 0, 1, 0]))
    67

    # N = 32, E = 64, FN = 8.00000
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1,
    ...     1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0]))
    64

    # N = 32, E = 64, FN = 8.00000
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0,
    ...     0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1]))
    64

    # N = 33, E = 64, FN = 8.50781
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 0, 1, 1, 0,
    ...     1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1]))
    64

    # N = 33, E = 64, FN = 8.50781
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1,
    ...     0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0]))
    64

    # N = 34, E = 65, FN = 8.89231
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 0, 1, 1,
    ...     0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1]))
    65

    # N = 34, E = 65, FN = 8.89231
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0,
    ...     1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0]))
    65

    # N = 35, E = 73, FN = 8.39041
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1,
    ...     0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 1, 1, 0, 0]))
    73

    # N = 35, E = 73, FN = 8.39041
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0,
    ...     1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 0, 0, 1, 1]))
    73

    # N = 36, E = 82, FN = 7.90244
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1,
    ...     0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0]))
    82

    # N = 36, E = 82, FN = 7.90244
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0,
    ...     1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 1, 1, 0, 1]))
    82

    # N = 37, E = 86, FN = 7.95930
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1,
    ...     0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1]))
    86

    # N = 37, E = 86, FN = 7.95930
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0,
    ...     1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0]))
    86

    # N = 38, E = 87, FN = 8.29885
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1,
    ...     0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0]))
    87

    # N = 38, E = 87, FN = 8.29885
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0,
    ...     1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1]))
    87

    # N = 39, E = 99, FN = 7.68182
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1,
    ...     0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1]))
    99

    # N = 39, E = 99, FN = 7.68182
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0,
    ...     1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0]))
    99

    # N = 39, E = 99, FN = 7.68182
    >>> labs(np.array([0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0,
    ...     0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1]))
    99

    # N = 39, E = 99, FN = 7.68182
    >>> labs(np.array([1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1,
    ...     1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0]))
    99

    # N = 40, E = 108, FN = 7.40741
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1,
    ...     1, 0, 1, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0,
    ...     1]))
    108

    # N = 40, E = 108, FN = 7.40741
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0,
    ...     0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1,
    ...     0]))
    108

    # N = 41, E = 108, FN = 7.78241
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1,
    ...     0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 0,
    ...     1, 0]))
    108

    # N = 41, E = 108, FN = 7.78241
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0,
    ...     1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1,
    ...     0, 1]))
    108

    # N = 42, E = 101, FN = 8.73267
    >>> labs(np.array([0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1, 1,
    ...     1, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1, 0,
    ...     1, 0, 0]))
    101

    # N = 42, E = 101, FN = 8.73267
    >>> labs(np.array([1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 0, 0,
    ...     0, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 1,
    ...     0, 1, 1]))
    101

    # N = 43, E = 109, FN = 8.48165
    >>> labs(np.array([0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0,
    ...     1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 1,
    ...     0, 1, 1, 1]))
    109

    # N = 43, E = 109, FN = 8.48165
    >>> labs(np.array([1, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1,
    ...     0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0,
    ...     1, 0, 0, 0]))
    109

    # N = 44, E = 122, FN = 7.93443
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1,
    ...     1, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0, 1,
    ...     0, 1, 0, 0, 1]))
    122

    # N = 44, E = 122, FN = 7.93443
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0, 0,
    ...     0, 1, 0, 1, 1, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 1, 0,
    ...     1, 0, 1, 1, 0]))
    122

    # N = 45, E = 118, FN = 8.58051
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1,
    ...     0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0,
    ...     1, 0, 1, 0, 1, 0]))
    118

    # N = 45, E = 118, FN = 8.58051
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0,
    ...     1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1,
    ...     0, 1, 0, 1, 0, 1]))
    118

    # N = 46, E = 131, FN = 8.07634
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0,
    ...     0, 0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 0,
    ...     1, 0, 1, 0, 1, 0, 1]))
    131

    # N = 46, E = 131, FN = 8.07634
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1,
    ...     1, 1, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1,
    ...     0, 1, 0, 1, 0, 1, 0]))
    131

    # N = 46, E = 131, FN = 8.07634
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1,
    ...     0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0,
    ...     1, 0, 1, 0, 1, 0, 1]))
    131

    # N = 46, E = 131, FN = 8.07634
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0,
    ...     1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1,
    ...     0, 1, 0, 1, 0, 1, 0]))
    131

    # N = 46, E = 131, FN = 8.07634
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0,
    ...     0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1,
    ...     0, 1, 1, 0, 1, 1, 0]))
    131

    # N = 46, E = 131, FN = 8.07634
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1,
    ...     1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0,
    ...     1, 0, 0, 1, 0, 0, 1]))
    131

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1,
    ...     0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1,
    ...     0, 1, 0, 1, 0, 1, 0, 1]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0,
    ...     1, 1, 1, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0, 0,
    ...     1, 0, 1, 0, 1, 0, 1, 0]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
    ...     1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 1,
    ...     0, 1, 1, 0, 0, 1, 0, 1]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
    ...     0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0,
    ...     1, 0, 0, 1, 1, 0, 1, 0]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 0,
    ...     0, 0, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0,
    ...     1, 1, 0, 0, 0, 0, 1, 1]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 1, 1, 0, 1,
    ...     1, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 1, 0, 0, 1,
    ...     0, 0, 1, 1, 1, 1, 0, 0]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1,
    ...     0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 0,
    ...     1, 0, 1, 0, 1, 1, 0, 1]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
    ...     1, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0, 1,
    ...     0, 1, 0, 1, 0, 0, 1, 0]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1,
    ...     0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1,
    ...     0, 1, 0, 0, 1, 0, 0, 1]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0,
    ...     1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0,
    ...     1, 0, 1, 1, 0, 1, 1, 0]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1,
    ...     0, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0,
    ...     1, 0, 1, 1, 0, 0, 0, 1]))
    135

    # N = 47, E = 135, FN = 8.18148
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0,
    ...     1, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1,
    ...     0, 1, 0, 0, 1, 1, 1, 0]))
    135

    # N = 48, E = 140, FN = 8.22857
    >>> labs(np.array([0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0,
    ...     0, 0, 1, 1, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 1,
    ...     0, 1, 0, 0, 1, 0, 1, 1, 0]))
    140

    # N = 48, E = 140, FN = 8.22857
    >>> labs(np.array([1, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1,
    ...     1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 0,
    ...     1, 0, 1, 1, 0, 1, 0, 0, 1]))
    140

    # N = 49, E = 136, FN = 8.82721
    >>> labs(np.array([0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0,
    ...     1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1,
    ...     1, 1, 1, 0, 1, 0, 0, 0, 0, 1]))
    136

    # N = 49, E = 136, FN = 8.82721
    >>> labs(np.array([1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1,
    ...     0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0,
    ...     0, 0, 0, 1, 0, 1, 1, 1, 1, 0]))
    136

    # N = 49, E = 136, FN = 8.82721
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0,
    ...     0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1,
    ...     0, 0, 1, 0, 0, 1, 0, 0, 1, 0]))
    136

    # N = 49, E = 136, FN = 8.82721
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1,
    ...     1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 0, 1, 0,
    ...     1, 1, 0, 1, 1, 0, 1, 1, 0, 1]))
    136

    # N = 50, E = 153, FN = 8.16993
    >>> labs(np.array([0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0,
    ...     1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1,
    ...     1, 1, 1, 0, 1, 0, 0, 0, 0, 1, 1]))
    153

    # N = 50, E = 153, FN = 8.16993
    >>> labs(np.array([1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1,
    ...     0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0,
    ...     0, 0, 0, 1, 0, 1, 1, 1, 1, 0, 0]))
    153

    # N = 50, E = 153, FN = 8.16993
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1,
    ...     0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, 0,
    ...     1, 0, 1, 0, 0, 1, 0, 1, 1, 0, 1]))
    153

    # N = 50, E = 153, FN = 8.16993
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0,
    ...     1, 1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 1,
    ...     0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0]))
    153

    # N = 50, E = 153, FN = 8.16993
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0,
    ...     0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0,
    ...     1, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0]))
    153

    # N = 50, E = 153, FN = 8.16993
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1,
    ...     1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 0, 1,
    ...     0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1]))
    153

    # N = 51, E = 153, FN = 8.50000
    >>> labs(np.array([0, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0,
    ...     1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1,
    ...     1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1]))
    153

    # N = 51, E = 153, FN = 8.50000
    >>> labs(np.array([1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1,
    ...     0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0,
    ...     0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0]))
    153

    # N = 52, E = 166, FN = 8.14458
    >>> labs(np.array([0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1,
    ...     1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1, 1, 0,
    ...     1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1]))
    166

    # N = 52, E = 166, FN = 8.14458
    >>> labs(np.array([1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0,
    ...     0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 1,
    ...     0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0]))
    166

    # N = 53, E = 170, FN = 8.26176
    >>> labs(np.array([0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0,
    ...     0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0,
    ...     1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0, 1]))
    170

    # N = 53, E = 170, FN = 8.26176
    >>> labs(np.array([1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 0, 1, 0, 1,
    ...     1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1,
    ...     0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 0]))
    170

    # N = 53, E = 170, FN = 8.26176
    >>> labs(np.array([0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
    ...     1, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0,
    ...     1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0]))
    170

    # N = 53, E = 170, FN = 8.26176
    >>> labs(np.array([1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
    ...     0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1,
    ...     0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1]))
    170

    # N = 54, E = 175, FN = 8.33143
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0,
    ...     1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 1,
    ...     1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0]))
    175

    # N = 54, E = 175, FN = 8.33143
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1,
    ...     0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0,
    ...     0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1]))
    175

    # N = 55, E = 171, FN = 8.84503
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0,
    ...     0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 0,
    ...     1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1]))
    171

    # N = 55, E = 171, FN = 8.84503
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 0, 1,
    ...     1, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 1,
    ...     0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0]))
    171

    # N = 55, E = 171, FN = 8.84503
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 0, 1,
    ...     0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1]))
    171

    # N = 55, E = 171, FN = 8.84503
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0,
    ...     1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0]))
    171

    # N = 56, E = 192, FN = 8.16667
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0,
    ...     1, 1, 1, 0, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1, 0, 1, 0,
    ...     1, 1, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 1]))
    192

    # N = 56, E = 192, FN = 8.16667
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 1,
    ...     0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 1,
    ...     0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1, 0]))
    192

    # N = 57, E = 188, FN = 8.64096
    >>> labs(np.array([0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1,
    ...     1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1,
    ...     0, 1, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 0]))
    188

    # N = 57, E = 188, FN = 8.64096
    >>> labs(np.array([1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
    ...     0, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
    ...     1, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 1]))
    188

    # N = 58, E = 197, FN = 8.53807
    >>> labs(np.array([0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 1, 1, 0, 1,
    ...     1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0,
    ...     1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 1]))
    197

    # N = 58, E = 197, FN = 8.53807
    >>> labs(np.array([1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 1, 0,
    ...     0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1,
    ...     0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0, 0]))
    197

    # N = 59, E = 205, FN = 8.49024
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1,
    ...     1, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1,
    ...     1, 0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 1]))
    205

    # N = 59, E = 205, FN = 8.49024
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0,
    ...     0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0,
    ...     0, 1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0]))
    205

    # N = 59, E = 205, FN = 8.49024
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0,
    ...     1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 0, 0, 1, 1, 1, 0, 0,
    ...     0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 0, 0]))
    205

    # N = 59, E = 205, FN = 8.49024
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1,
    ...     0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1,
    ...     1, 1, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 1]))
    205

    # N = 60, E = 218, FN = 8.25688
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1,
    ...     0, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 0, 0, 1, 1, 1, 1,
    ...     0, 0, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1]))
    218

    # N = 60, E = 218, FN = 8.25688
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0,
    ...     1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0,
    ...     1, 1, 0, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0]))
    218

    # N = 60, E = 218, FN = 8.25688
    >>> labs(np.array([0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 1, 1,
    ...     0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 1,
    ...     1, 1, 1, 1, 0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0, 0, 1]))
    218

    # N = 60, E = 218, FN = 8.25688
    >>> labs(np.array([1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 0, 0,
    ...     1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 0, 0,
    ...     0, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0]))
    218

    # N = 61, E = 226, FN = 8.23230
    >>> labs(np.array([0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0,
    ...     0, 0, 1, 1, 1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0,
    ...     0, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0,
    ...     1]))
    226

    # N = 61, E = 226, FN = 8.23230
    >>> labs(np.array([1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1,
    ...     1, 1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 0, 1,
    ...     1, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1,
    ...     0]))
    226

    # N = 62, E = 235, FN = 8.17872
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 1, 0,
    ...     1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0,
    ...     1, 0, 1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 0,
    ...     1, 1]))
    235

    # N = 62, E = 235, FN = 8.17872
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1,
    ...     0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1,
    ...     0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 1, 1,
    ...     0, 0]))
    235

    # N = 62, E = 235, FN = 8.17872
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0,
    ...     0, 1, 1, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0,
    ...     1, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 0, 0, 0, 1, 0,
    ...     1, 1]))
    235

    # N = 62, E = 235, FN = 8.17872
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1,
    ...     1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1,
    ...     0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 1, 1, 1, 0, 1,
    ...     0, 0]))
    235

    # N = 63, E = 207, FN = 9.58696
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1,
    ...     1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1,
    ...     0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 0,
    ...     1, 1, 0]))
    207

    # N = 63, E = 207, FN = 9.58696
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0,
    ...     0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0,
    ...     1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1,
    ...     0, 0, 1]))
    207

    # N = 64, E = 208, FN = 9.84615
    >>> labs(np.array([0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1,
    ...     1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1,
    ...     0, 0, 0, 0, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 0,
    ...     1, 1, 0, 0]))
    208

    # N = 64, E = 208, FN = 9.84615
    >>> labs(np.array([1, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 0, 0, 0,
    ...     0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0,
    ...     1, 1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1,
    ...     0, 0, 1, 1]))
    208

    # N = 65, E = 240, FN = 8.80208
    >>> labs(np.array([0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 1,
    ...     0, 1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 0,
    ...     1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1,
    ...     0, 0, 1, 0, 0]))
    240

    # N = 65, E = 240, FN = 8.80208
    >>> labs(np.array([1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0,
    ...     1, 0, 0, 0, 1, 1, 1, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 1,
    ...     0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0,
    ...     1, 1, 0, 1, 1]))
    240

    # N = 66, E = 257, FN = 8.47471
    >>> labs(np.array([0, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0,
    ...     0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1,
    ...     0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0,
    ...     1, 1, 1, 1, 0, 0]))
    257

    # N = 66, E = 257, FN = 8.47471
    >>> labs(np.array([1, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1,
    ...     1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0,
    ...     1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1,
    ...     0, 0, 0, 0, 1, 1]))
    257

    # N = 3, all bits are True
    >>> labs(np.array([1, 1, 1]))
    5

    # N = 4, all bits are True
    >>> labs(np.array([1, 1, 1, 1]))
    14

    # N = 5, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1]))
    30

    # N = 6, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1]))
    55

    # N = 7, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1]))
    91

    # N = 8, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0]))
    140

    # N = 9, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0]))
    204

    # N = 10, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    285

    # N = 11, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    385

    # N = 12, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    506

    # N = 13, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    650

    # N = 14, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    819

    # N = 15, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    1015

    # N = 16, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    1240

    # N = 17, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    1496

    # N = 18, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    1785

    # N = 19, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0]))
    2109

    # N = 20, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0]))
    2470

    # N = 21, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1]))
    2870

    # N = 22, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1]))
    3311

    # N = 23, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1]))
    3795

    # N = 24, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0]))
    4324

    # N = 25, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1]))
    4900

    # N = 26, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1]))
    5525

    # N = 27, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1]))
    6201

    # N = 28, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    6930

    # N = 29, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    7714

    # N = 30, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    8555

    # N = 31, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    9455

    # N = 32, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    10416

    # N = 33, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    11440

    # N = 34, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    12529

    # N = 35, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    13685

    # N = 36, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    14910

    # N = 37, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    16206

    # N = 38, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    17575

    # N = 39, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    19019

    # N = 40, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0]))
    20540

    # N = 41, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0]))
    22140

    # N = 42, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1]))
    23821

    # N = 43, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1]))
    25585

    # N = 44, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0]))
    27434

    # N = 45, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0]))
    29370

    # N = 46, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1]))
    31395

    # N = 47, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1]))
    33511

    # N = 48, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1]))
    35720

    # N = 49, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    38024

    # N = 50, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    40425

    # N = 51, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    42925

    # N = 52, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    45526

    # N = 53, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    48230

    # N = 54, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    51039

    # N = 55, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    53955

    # N = 56, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    56980

    # N = 57, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    60116

    # N = 58, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    63365

    # N = 59, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    66729

    # N = 60, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    70210

    # N = 61, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0]))
    73810

    # N = 62, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1]))
    77531

    # N = 63, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1]))
    81375

    # N = 64, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1]))
    85344

    # N = 65, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1]))
    89440

    # N = 66, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1]))
    93665

    # N = 67, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0]))
    98021

    # N = 68, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0]))
    102510

    # N = 69, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0]))
    107134

    # N = 70, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    111895

    # N = 71, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    116795

    # N = 72, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    121836

    # N = 73, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    127020

    # N = 74, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    132349

    # N = 75, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    137825

    # N = 76, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    143450

    # N = 77, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    149226

    # N = 78, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    155155

    # N = 79, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    161239

    # N = 80, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    167480

    # N = 81, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    173880

    # N = 82, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0]))
    180441

    # N = 83, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0]))
    187165

    # N = 84, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0]))
    194054

    # N = 85, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0]))
    201110

    # N = 86, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1]))
    208335

    # N = 87, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0]))
    215731

    # N = 88, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1]))
    223300

    # N = 89, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0]))
    231044

    # N = 90, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0]))
    238965

    # N = 91, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    247065

    # N = 92, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    255346

    # N = 93, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    263810

    # N = 94, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    272459

    # N = 95, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    281295

    # N = 96, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    290320

    # N = 97, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    299536

    # N = 98, all bits are False
    >>> labs(np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    ...     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]))
    308945

    # N = 99, all bits are True
    >>> labs(np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
    ...     1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]))
    318549
    """
    n: Final[int] = len(x)  # Get the length of the bit string.
    result: int = 0
    for k in range(1, n):
        sqr: int = 0
        for i in range(n - k):
            sqr += 1 if x[i] == x[i + k] else -1
        result += sqr * sqr
    return result


def energy_to_merit(n: int, energy: int) -> int | float:
    """
    Convert an energy value to a merit factor.

    :param n: the length of the bit sequence
    :param energy: the integer energy factor, computed based on :func:`labs`.
    :returns: a merit factor

    >>> try:
    ...     energy_to_merit(0, 1)
    ... except ValueError as ve:
    ...     print(ve)
    n=0 is invalid, must be in 3..1000000000.

    >>> try:
    ...     energy_to_merit(4, 0)
    ... except ValueError as ve:
    ...     print(ve)
    energy=0 is invalid, must be in 1..1000000000.
    >>> energy_to_merit(3, 1)
    4.5
    >>> energy_to_merit(4, 2)
    4
    >>> energy_to_merit(5, 2)
    6.25
    >>> print(f"{energy_to_merit(6, 7):.4}")
    2.571
    >>> print(f"{energy_to_merit(7, 3):.4}")
    8.167
    >>> energy_to_merit(8, 8)
    4
    >>> print(f"{energy_to_merit(9, 12):.4}")
    3.375
    >>> print(f"{energy_to_merit(10, 13):.4}")
    3.846
    >>> print(f"{energy_to_merit(11, 5):.4}")
    12.1
    >>> print(f"{energy_to_merit(12, 10):.4}")
    7.2
    >>> print(f"{energy_to_merit(13, 6):.5}")
    14.083
    >>> print(f"{energy_to_merit(14, 19):.4}")
    5.158
    >>> print(f"{energy_to_merit(15, 15):.4}")
    7.5
    >>> print(f"{energy_to_merit(16, 24):.4}")
    5.333
    >>> print(f"{energy_to_merit(17, 32):.4}")
    4.516
    >>> print(f"{energy_to_merit(18, 25):.4}")
    6.48
    >>> print(f"{energy_to_merit(47, 135):.4}")
    8.181
    >>> print(f"{energy_to_merit(66, 257):.4}")
    8.475
    >>> print(f"{energy_to_merit(19, 33):.4}")
    5.47
    >>> print(f"{energy_to_merit(119, 835):.4}")
    8.48
    >>> print(f"{energy_to_merit(401, 11888):.5}")
    6.7632
    """
    n = check_int_range(n, "n", 3)
    energy = check_int_range(energy, "energy", 1)
    return try_int_div(n * n, energy + energy)


def merit_to_energy(n: int, merit: int | float) -> int:
    """
    Convert a merit factor back to an energy.

    :param n: the energy
    :param merit: the merit factor
    :returns: the energy

    >>> try:
    ...     merit_to_energy(0, 1)
    ... except ValueError as ve:
    ...     print(ve)
    n=0 is invalid, must be in 3..1000000000.

    >>> try:
    ...     merit_to_energy(3, 0)
    ... except ValueError as ve:
    ...     print(ve)
    merit factor cannot be 0.

    >>> try:
    ...     merit_to_energy(3, "x")
    ... except TypeError as te:
    ...     print(str(te)[:50])
    merit should be an instance of any in {float, int}

    >>> merit_to_energy(3, 4.5)
    1
    >>> merit_to_energy(4, 4)
    2
    >>> merit_to_energy(5, 6.25)
    2
    >>> merit_to_energy(6, 2.571)
    7
    >>> merit_to_energy(7, 8.167)
    3
    >>> merit_to_energy(8, 4)
    8
    >>> merit_to_energy(9, 3.375)
    12
    >>> merit_to_energy(10, 3.846)
    13
    >>> merit_to_energy(11, 12.100)
    5
    >>> merit_to_energy(12, 7.200)
    10
    >>> merit_to_energy(13, 14.083)
    6
    >>> merit_to_energy(14, 5.158)
    19
    >>> merit_to_energy(15, 7.5)
    15
    >>> merit_to_energy(16, 5.333)
    24
    >>> merit_to_energy(17, 4.516)
    32
    >>> merit_to_energy(18, 6.48)
    25
    >>> merit_to_energy(47, 8.181)
    135
    >>> merit_to_energy(66, 8.475)
    257
    >>> merit_to_energy(19, 5.47)
    33
    >>> merit_to_energy(119, 8.48)
    835
    >>> merit_to_energy(401, 6.7632)
    11888
    >>> merit_to_energy(181, 8.9316)
    1834
    >>> merit_to_energy(201, 8.4876)
    2380
    >>> merit_to_energy(215, 8.5888)
    2691
    >>> merit_to_energy(221, 8.8544)
    2758
    >>> merit_to_energy(241, 8.0668)
    3600
    >>> merit_to_energy(249, 8.1323)
    3812
    >>> merit_to_energy(259, 8.0918)
    4145
    >>> merit_to_energy(261, 7.8517)
    4338
    >>> merit_to_energy(271, 7.5386)
    4871
    >>> merit_to_energy(281, 7.5058)
    5260
    >>> merit_to_energy(283, 7.5088)
    5333
    >>> merit_to_energy(301, 7.4827)
    6054
    >>> merit_to_energy(303, 7.2462)
    6335
    >>> merit_to_energy(341, 6.9397)
    8378
    >>> merit_to_energy(381, 7.0893)
    10238
    >>> merit_to_energy(401, 6.7632)
    11888
    """
    n = check_int_range(n, "n", 3)
    if not isinstance(merit, int | float):
        raise type_error(merit, "merit", (int, float))
    if (not isfinite(merit)) or (merit <= 0):
        raise ValueError(f"merit factor cannot be {merit}.")
    return int(0.5 + ((n * n) / merit) / 2)


#: The lower bounds of the LABS problem, i.e., the objective values of the
#: known global optima also used in the unit tests of the :func:`labs`
#: function. We use the data from
#:
#: Tom Packebusch and Stephan Mertens. Low Autocorrelation Binary Sequences.
#: arXiv:1512.02475v2 [cond-mat.stat-mech] 24 Mar 2016
#: https://arxiv.org/pdf/1512.02475
_LBS: Final[dict[int, int]] = {
    3: 1, 4: 2, 5: 2, 6: 7, 7: 3, 8: 8, 9: 12, 10: 13, 11: 5, 12: 10, 13: 6,
    14: 19, 15: 15, 16: 24, 17: 32, 18: 25, 19: 29, 20: 26, 21: 26, 22: 39,
    23: 47, 24: 36, 25: 36, 26: 45, 27: 37, 28: 50, 29: 62, 30: 59, 31: 67,
    32: 64, 33: 64, 34: 65, 35: 73, 36: 82, 37: 86, 38: 87, 39: 99, 40: 108,
    41: 108, 42: 101, 43: 109, 44: 122, 45: 118, 46: 131, 47: 135, 48: 140,
    49: 136, 50: 153, 51: 153, 52: 166, 53: 170, 54: 175, 55: 171, 56: 192,
    57: 188, 58: 197, 59: 205, 60: 218, 61: 226, 62: 235, 63: 207, 64: 208,
    65: 240, 66: 257}


#: the internal set of related work solutions
__RELATED_WORKS: Final[dict[tuple[str, str], dict[int, int]]] = {
    ("PM2016LABS",
     "Tom Packebusch and Stephan Mertens. Low Autocorrelation Binary "
     "Sequences. arXiv:1512.02475v2 [cond-mat.stat-mech] 24 Mar 2016"): {
        3: 1, 4: 2, 5: 2, 6: 7, 7: 3, 8: 8, 9: 12, 10: 13, 11: 5, 12: 10,
        13: 6, 14: 19, 15: 15, 16: 24, 17: 32, 18: 25, 19: 29, 20: 26, 21: 26,
        22: 39, 23: 47, 24: 36, 25: 36, 26: 45, 27: 37, 28: 50, 29: 62,
        30: 59, 31: 67, 32: 64, 33: 64, 34: 65, 35: 73, 36: 82, 37: 86,
        38: 87, 39: 99, 40: 108, 41: 108, 42: 101, 43: 109, 44: 122, 45: 118,
        46: 131, 47: 135, 48: 140, 49: 136, 50: 153, 51: 153, 52: 166,
        53: 170, 54: 175, 55: 171, 56: 192, 57: 188, 58: 197, 59: 205,
        60: 218, 61: 226, 62: 235, 63: 207, 64: 208, 65: 240, 66: 257,
        67: 241, 69: 282, 71: 275, 73: 348, 75: 341, 77: 358, 79: 407,
        81: 400, 83: 377, 85: 442, 87: 451, 89: 484, 91: 477, 93: 502,
        95: 479, 97: 536, 99: 577, 101: 578, 103: 555, 105: 620, 107: 677,
        109: 662, 111: 687, 113: 752, 115: 745, 117: 786, 119: 835,
    },
    ("BBB2017LABSOIMFARPTAT",
     "Borko Bošković, Franc Brglez, and Janez Brest. Low-Autocorrelation "
     "Binary Sequences: On Improved Merit Factors and Runtime Predictions"
     " to Achieve Them. arXiv:1406.5301v6 [cs.DS] 6 May 2017."
     "https://arxiv.org/pdf/1406.5301"): {
        181: 1834, 201: 2380, 215: 2691, 221: 2758, 241: 3600, 249: 3812,
        259: 4145, 261: 4338, 271: 4871, 281: 5260, 283: 5333, 301: 6054,
        303: 6335, 341: 8378, 381: 10238, 401: 11888,
        107: merit_to_energy(107, 8.4557), 109: merit_to_energy(109, 8.9736),
        111: merit_to_energy(111, 8.9672), 113: merit_to_energy(113, 8.49),
        115: merit_to_energy(115, 8.8758), 117: merit_to_energy(117, 8.708),
        119: merit_to_energy(119, 8.4796), 121: merit_to_energy(121, 8.6736),
        141: merit_to_energy(141, 8.8282), 149: merit_to_energy(149, 9.1137),
        157: merit_to_energy(157, 9.0223), 161: merit_to_energy(161, 8.5718),
        165: merit_to_energy(165, 9.2351), 169: merit_to_energy(169, 9.3215),
        173: merit_to_energy(173, 9.3645), 175: merit_to_energy(175, 9.0768),
        177: merit_to_energy(177, 9.5052), 179: merit_to_energy(179, 9.0974),
        183: merit_to_energy(183, 9.0073), 189: merit_to_energy(189, 9.0847),
    },
    ("MD2015NESFLLABS",
     "Wai Ho Mow and Ke-Lin Du. New Evolutionary Search for Long Low "
     "Autocorrelation Binary Sequences. IEEE Transactions on Aerospace and "
     "Electronic Systems. 51(1):290-303. January 2015. "
     "doi:10.1109/TAES.2014.130518"): {
        106: merit_to_energy(106, 5.0295), 107: merit_to_energy(107, 5.1805),
        108: merit_to_energy(108, 4.6957), 109: merit_to_energy(109, 5.0429),
        110: merit_to_energy(110, 4.9631), 111: merit_to_energy(111, 5.626),
        112: merit_to_energy(112, 5.3153), 113: merit_to_energy(113, 4.8514),
        114: merit_to_energy(114, 4.4114), 115: merit_to_energy(115, 4.8729),
        116: merit_to_energy(116, 4.3974), 117: merit_to_energy(117, 4.2832),
        118: merit_to_energy(118, 4.5355), 119: merit_to_energy(119, 4.7235),
        120: merit_to_energy(120, 5.8632), 121: merit_to_energy(121, 4.4421),
        122: merit_to_energy(122, 4.6368), 123: merit_to_energy(123, 4.6897),
        124: merit_to_energy(124, 4.9987), 125: merit_to_energy(125, 4.8646),
        126: merit_to_energy(126, 5.0272), 127: merit_to_energy(127, 4.8965),
        128: merit_to_energy(128, 4.8075), 129: merit_to_energy(129, 4.6328),
        130: merit_to_energy(130, 4.8872), 131: merit_to_energy(131, 4.9627),
        132: merit_to_energy(132, 4.343), 133: merit_to_energy(133, 4.6995),
        134: merit_to_energy(134, 4.438), 135: merit_to_energy(135, 4.5134),
        136: merit_to_energy(136, 4.472), 137: merit_to_energy(137, 4.2273),
        138: merit_to_energy(138, 4.3341), 139: merit_to_energy(139, 4.6602),
        140: merit_to_energy(140, 4.6009), 141: merit_to_energy(141, 4.418),
        142: merit_to_energy(142, 4.4789), 143: merit_to_energy(143, 4.5584),
        144: merit_to_energy(144, 4.2492), 145: merit_to_energy(145, 4.4696),
        146: merit_to_energy(146, 4.6239), 147: merit_to_energy(147, 4.3409),
        148: merit_to_energy(148, 4.3703), 149: merit_to_energy(149, 4.5531),
        150: merit_to_energy(150, 4.7209), 151: merit_to_energy(151, 4.3663),
        152: merit_to_energy(152, 5.2509), 153: merit_to_energy(153, 4.8206),
        154: merit_to_energy(154, 4.2517), 155: merit_to_energy(155, 4.2704),
        156: merit_to_energy(156, 4.5986), 157: merit_to_energy(157, 4.6368),
        158: merit_to_energy(158, 4.9473), 159: merit_to_energy(159, 4.7396),
        160: merit_to_energy(160, 4.878), 161: merit_to_energy(161, 4.7789),
        162: merit_to_energy(162, 4.5674), 163: merit_to_energy(163, 4.676),
        164: merit_to_energy(164, 5.0367), 165: merit_to_energy(165, 4.2834),
        166: merit_to_energy(166, 4.5337), 167: merit_to_energy(167, 4.719),
        168: merit_to_energy(168, 4.6057), 169: merit_to_energy(169, 4.7099),
        170: merit_to_energy(170, 4.6598), 171: merit_to_energy(171, 4.4077),
        172: merit_to_energy(172, 4.7685), 173: merit_to_energy(173, 4.7932),
        174: merit_to_energy(174, 4.6564), 175: merit_to_energy(175, 4.2832),
        176: merit_to_energy(176, 4.2549), 177: merit_to_energy(177, 4.3416),
        178: merit_to_energy(178, 4.1331), 179: merit_to_energy(179, 4.3287),
        180: merit_to_energy(180, 4.3385), 181: merit_to_energy(181, 4.5832),
        182: merit_to_energy(182, 4.2347), 183: merit_to_energy(183, 4.4879),
        184: merit_to_energy(184, 3.9331), 185: merit_to_energy(185, 4.5464),
        186: merit_to_energy(186, 4.6239), 187: merit_to_energy(187, 4.5285),
        188: merit_to_energy(188, 4.7176), 189: merit_to_energy(189, 4.809),
        190: merit_to_energy(190, 4.4645), 191: merit_to_energy(191, 5.0125),
        192: merit_to_energy(192, 4.795), 193: merit_to_energy(193, 4.5205),
        194: merit_to_energy(194, 4.4288), 195: merit_to_energy(195, 4.2316),
        196: merit_to_energy(196, 4.3048), 197: merit_to_energy(197, 4.6267),
        198: merit_to_energy(198, 4.5052), 199: merit_to_energy(199, 4.4889),
        200: merit_to_energy(200, 4.5496), 201: merit_to_energy(201, 4.008),
        202: merit_to_energy(202, 4.3033), 203: merit_to_energy(203, 4.3033),
        204: merit_to_energy(204, 4.231), 205: merit_to_energy(205, 4.705),
        206: merit_to_energy(206, 4.1288), 207: merit_to_energy(207, 4.1886),
        208: merit_to_energy(208, 4.3126), 209: merit_to_energy(209, 4.3542),
        210: merit_to_energy(210, 4.6392), 211: merit_to_energy(211, 4.2899),
        212: merit_to_energy(212, 4.3567), 213: merit_to_energy(213, 4.517),
        214: merit_to_energy(214, 4.4872), 215: merit_to_energy(215, 4.3584),
        216: merit_to_energy(216, 4.7725), 217: merit_to_energy(217, 4.1364),
        218: merit_to_energy(218, 4.5373), 219: merit_to_energy(219, 4.7798),
        220: merit_to_energy(220, 4.6414), 221: merit_to_energy(221, 4.4096),
        222: merit_to_energy(222, 4.5273), 223: merit_to_energy(223, 4.5216),
        224: merit_to_energy(224, 4.5253), 225: merit_to_energy(225, 4.6462),
        226: merit_to_energy(226, 4.599), 227: merit_to_energy(227, 4.5641),
        228: merit_to_energy(228, 4.6365), 229: merit_to_energy(229, 4.5664),
        230: merit_to_energy(230, 4.5299), 231: merit_to_energy(231, 4.5819),
        232: merit_to_energy(232, 4.4556), 233: merit_to_energy(233, 4.5423),
        234: merit_to_energy(234, 4.3122), 235: merit_to_energy(235, 4.4529),
        236: merit_to_energy(236, 4.3418), 237: merit_to_energy(237, 4.3488),
        238: merit_to_energy(238, 4.6992), 239: merit_to_energy(239, 4.5457),
        240: merit_to_energy(240, 4.6512), 241: merit_to_energy(241, 4.3921),
        242: merit_to_energy(242, 4.3464), 243: merit_to_energy(243, 4.6415),
        244: merit_to_energy(244, 4.2224), 245: merit_to_energy(245, 4.3994),
        246: merit_to_energy(246, 4.498), 247: merit_to_energy(247, 4.7024),
        248: merit_to_energy(248, 4.4414), 249: merit_to_energy(249, 4.6491),
        250: merit_to_energy(250, 4.4816), 251: merit_to_energy(251, 4.7291),
        252: merit_to_energy(252, 4.002), 253: merit_to_energy(253, 4.7163),
        254: merit_to_energy(254, 4.544), 255: merit_to_energy(255, 4.5902),
        256: merit_to_energy(256, 4.8075), 257: merit_to_energy(257, 4.8338),
        258: merit_to_energy(258, 4.3421), 259: merit_to_energy(259, 4.4596),
        260: merit_to_energy(260, 4.6492), 261: merit_to_energy(261, 4.2672),
        262: merit_to_energy(262, 4.4557), 263: merit_to_energy(263, 4.3344),
        264: merit_to_energy(264, 4.6814), 265: merit_to_energy(265, 4.6007),
        266: merit_to_energy(266, 4.3118), 267: merit_to_energy(267, 4.2652),
        268: merit_to_energy(268, 4.5912), 269: merit_to_energy(269, 4.5903),
        270: merit_to_energy(270, 4.3836), 271: merit_to_energy(271, 4.5028),
        272: merit_to_energy(272, 4.8267), 273: merit_to_energy(273, 4.4617),
        274: merit_to_energy(274, 4.4683), 275: merit_to_energy(275, 4.3881),
        276: merit_to_energy(276, 4.5032), 277: merit_to_energy(277, 4.4352),
        278: merit_to_energy(278, 4.579), 279: merit_to_energy(279, 4.3192),
        280: merit_to_energy(280, 4.2535), 281: merit_to_energy(281, 4.2325),
        282: merit_to_energy(282, 4.4273), 283: merit_to_energy(283, 4.4747),
        284: merit_to_energy(284, 4.4307), 285: merit_to_energy(285, 4.3735),
        286: merit_to_energy(286, 4.383), 287: merit_to_energy(287, 4.1706),
        288: merit_to_energy(288, 4.3858), 289: merit_to_energy(289, 4.2981),
        290: merit_to_energy(290, 4.5622), 291: merit_to_energy(291, 4.5621),
        292: merit_to_energy(292, 4.641), 293: merit_to_energy(293, 4.7599),
        294: merit_to_energy(294, 4.3961), 295: merit_to_energy(295, 4.3992),
        296: merit_to_energy(296, 4.3203), 297: merit_to_energy(297, 4.2953),
        298: merit_to_energy(298, 4.5771), 299: merit_to_energy(299, 4.4166),
        300: merit_to_energy(300, 4.4074), 303: merit_to_energy(303, 4.3507),
        304: merit_to_energy(304, 4.0676), 350: merit_to_energy(350, 3.9458),
        353: merit_to_energy(353, 4.2075), 400: merit_to_energy(400, 4.3908),
        449: merit_to_energy(449, 4.0547), 450: merit_to_energy(450, 4.4235),
        500: merit_to_energy(500, 4.3442), 512: merit_to_energy(512, 4.2656),
        547: merit_to_energy(547, 4.3408), 550: merit_to_energy(550, 4.0695),
        600: merit_to_energy(600, 3.6753), 650: merit_to_energy(650, 3.9239),
        653: merit_to_energy(653, 4.2287), 700: merit_to_energy(700, 4.1524),
        750: merit_to_energy(750, 3.7603), 751: merit_to_energy(751, 4.1537),
        800: merit_to_energy(800, 3.7481), 850: merit_to_energy(850, 3.8096),
        853: merit_to_energy(853, 4.0854), 900: merit_to_energy(900, 3.7623),
        950: merit_to_energy(950, 4.0438), 953: merit_to_energy(953, 3.8493),
        1000: merit_to_energy(1000, 3.7873),
        1019: merit_to_energy(1019, 4.139),
        1024: merit_to_energy(1024, 3.9683),
        1500: merit_to_energy(1500, 3.7316),
        2000: merit_to_energy(2000, 3.6193),
        2048: merit_to_energy(2048, 3.5387),
        2197: merit_to_energy(2197, 3.6423),
        3000: merit_to_energy(3000, 3.3608),
        4096: merit_to_energy(4096, 3.4589),
    },
    ("BB2018AHAFALABSPWOLAHMF",
     "Borko Bošković and Janez Brest. A Heuristic Algorithm for a Low "
     "Autocorrelation Binary Sequence Problem with Odd Length and High "
     "Merit Factor. IEEE Access 6:4127-4134. January 2018. "
     "doi:10.1109/ACCESS.2018.2789916 "): {
        191: 1903, 193: 2144, 195: 2105, 197: 2202, 199: 2195, 201: 2220,
        203: 2317, 205: 2430, 207: 2351, 209: 2528, 211: 2457, 213: 2378,
        215: 2595, 217: 2684, 219: 2733, 221: 2734, 223: 2751, 225: 2808,
    },
    ("BHB2024PSAWALABSP",
     "Borko Bošković, Jana Herzog, and Janez Brest. Parallel Self-Avoiding "
     "Walks for a Low-Autocorrelation Binary Sequences Problem. Journal of "
     "Computational Science 77 (2024) 102260. "
     "doi:10.1016/j.jocs.2024.102260"): {
        171: 1669, 185: 1932, 193: 2040, 197: 2162, 199: 2187, 219: 2605,
        223: 2727, 225: 2768, 229: 2810, 231: 2963, 235: 2965, 237: 3118,
        239: 3055, 241: 3216, 243: 3233, 245: 3226, 247: 3259,
    },
}


def __make_bkses() -> dict[int, tuple[int, tuple[str, str]]]:
    """Make the BKSes."""
    d: Final[dict[int, tuple[int, tuple[str, str]]]] = {}
    for citation, data in __RELATED_WORKS.items():
        for n, ener in data.items():
            # We sanity check the BKSes, too.
            if ((n in _LBS) and (ener < _LBS[n])) or (ener < (n // 2)):
                raise ValueError("Invalid BKS!")
            if (n not in d) or (d[n][0] > ener):
                d[n] = ener, citation
    return d


#: The internal lookup table for best-known solutions
_BKS: Final[dict[int, tuple[int, tuple[str, str]]]] = __make_bkses()
del __make_bkses


def get_bks(n: int) -> tuple[int, tuple[str, str] | None]:
    """
    Get the best currently known solution for the given dimension `n`.

    :param n: the dimension
    :returns: a tuple of the energy value and a tuple of a citation. If
        no solution is found in literature, the lower bound is returned and
        `None` is returned as literature tuple

    >>> try:
    ...     get_bks(-1)
    ... except ValueError as ve:
    ...     print(ve)
    n=-1 is invalid, must be in 3..1000000000.

    >>> get_bks(20)[0]
    26

    >>> get_bks(401)[0]
    11888
    >>> get_bks(401)[1][0]
    'BBB2017LABSOIMFARPTAT'

    >>> get_bks(10000)
    (5000, None)
    """
    n = check_int_range(n, "n", 3)
    if n in _BKS:
        return _BKS[n]
    return n // 2, None


def is_labs_objective_value_optimal(n: int, f: int) -> bool | None:
    """
    Check whether a given objective value is optimal on the LABS problem.

    We use the lower and upper bound of the LABS problem to check whether a
    given objective value can be optimal. We also use the objective values of
    the known global optima from

    Tom Packebusch and Stephan Mertens. Low Autocorrelation Binary Sequences.
    arXiv:1512.02475v2 [cond-mat.stat-mech] 24 Mar 2016
    https://arxiv.org/pdf/1512.02475

    :param n: the scale of the problem
    :param f: the objective value
    :returns: `True` if the objective value `f` is optimal, `False` if it
        cannot be optimal, and `None` if the situation is unclear
    :raises TypeError: if `n` or `f` are not integers
    :raises ValueError: if `n` or `f` are invalid integer values

    >>> print(is_labs_objective_value_optimal(49, 136))
    True

    >>> print(is_labs_objective_value_optimal(49, 137))
    False

    >>> print(is_labs_objective_value_optimal(100, 50))
    True

    >>> print(is_labs_objective_value_optimal(100, 51))
    None

    >>> try:
    ...     is_labs_objective_value_optimal("x", 3)
    ... except TypeError as te:
    ...     print(te)
    n should be an instance of int but is str, namely 'x'.

    >>> try:
    ...     is_labs_objective_value_optimal(7, "3")
    ... except TypeError as te:
    ...     print(te)
    f should be an instance of int but is str, namely '3'.

    >>> try:
    ...     is_labs_objective_value_optimal(0, 0)
    ... except ValueError as ve:
    ...     print(ve)
    n=0 is invalid, must be in 3..1000000000.

    >>> try:
    ...     is_labs_objective_value_optimal(3, 0)
    ... except ValueError as ve:
    ...     print(ve)
    f=0 is invalid, must be in 1..5.

    >>> try:
    ...     is_labs_objective_value_optimal(3, 234240)
    ... except ValueError as ve:
    ...     print(ve)
    f=234240 is invalid, must be in 1..5.
    """
    check_int_range(n, "n", 3)
    lb: int = n // 2
    ub: int = ((n - 1) * n * (2 * n - 1)) // 6
    if check_int_range(f, "f", lb, ub) <= lb:
        return True
    if n in _LBS:  # We know the optimum, so if it is not it, we can stop.
        lb = _LBS[n]
        return check_int_range(f, "f", lb) <= lb
    if f >= ub:
        return False
    return None


class LABS(BitStringProblem):
    """The Low-Autocorrelation Sequence Problem."""

    def __init__(self, n: int) -> None:
        """
        Initialize the LABS objective function.

        :param n: the dimension of the problem
        """
        super().__init__(check_int_range(n, "n", 3))
        self.evaluate = labs  # type: ignore

    def lower_bound(self) -> int:
        """
        Get the lower bound of the LABS problem.

        The minimum amount of energy that a sequence could possibly have.
        For each correlation length `k`, this is `(n - k) % 2`.
        In total, this is just equivalent to `n // 2`.

        For some instances of :class:`LABS`, the global optima are known.
        For these instances, we return the objective values of these optima,
        which obviously are `>= n // 2`.

        :return: the lower bound of the :class:`LABS` problem

        >>> LABS(16).lower_bound()
        24

        >>> LABS(61).lower_bound()
        226

        >>> LABS(200).lower_bound()
        100

        >>> LABS(124200).lower_bound()
        62100
        """
        n: Final[int] = self.n
        if n in _LBS:
            return _LBS[n]
        return n // 2

    def upper_bound(self) -> int:
        """
        Get the upper bound of the :class:`LABS` problem.

        The energy of a sequence where all bits are the same is the highest.
        This means that each correlation basically produces its length to the
        square as contribution to the result. The upper bound then is the sum
        of all of these squares from `1..(n-1)`.

        :return: the upper bound of LABS

        >>> LABS(16).upper_bound()
        1240

        >>> LABS(63).upper_bound()
        81375

        >>> LABS(200).upper_bound()
        2646700

        >>> LABS(12316).upper_bound()
        622636950290
        """
        return ((self.n - 1) * self.n * (2 * self.n - 1)) // 6

    def __str__(self) -> str:
        """
        Get the name of the LABS objective function.

        :return: labs_ + length of string

        >>> LABS(16)
        labs_16
        """
        return f"labs_{self.n}"

    @classmethod
    def default_instances(
            cls: type, scale_min: int = 3,
            scale_max: int = min(1024, max(_BKS.keys()))) \
            -> Iterator[Callable[[], "LABS"]]:
        """
        Get the default instances of the :class:`LABS` problem.

        :param scale_min: the minimum permitted scale, by default `3`
        :param scale_max: the maximum permitted scale, by default `625`
        :returns: a sequence of default :class:`LABS` instances

        >>> len(list(LABS.default_instances()))
        81

        >>> [x() for x in LABS.default_instances()]
        [labs_3, labs_4, labs_5, labs_6, labs_7, labs_8, labs_9, labs_10, \
labs_11, labs_12, labs_13, labs_14, labs_15, labs_16, labs_17, labs_18, \
labs_19, labs_20, labs_21, labs_22, labs_23, labs_24, labs_25, labs_26, \
labs_27, labs_28, labs_29, labs_30, labs_31, labs_32, labs_33, labs_36, \
labs_40, labs_41, labs_42, labs_44, labs_48, labs_49, labs_50, labs_55, \
labs_59, labs_60, labs_64, labs_66, labs_77, labs_79, labs_81, labs_85, \
labs_99, labs_100, labs_107, labs_111, labs_121, labs_125, labs_128, \
labs_144, labs_149, labs_169, labs_170, labs_192, labs_196, labs_199, \
labs_200, labs_222, labs_225, labs_243, labs_256, labs_269, labs_289, \
labs_300, labs_341, labs_400, labs_500, labs_512, labs_600, labs_625, \
labs_700, labs_800, labs_900, labs_1000, labs_1024]
        """
        check_int_range(scale_max, "scale_max", check_int_range(
            scale_min, "scale_min", 1, 1_000_000_000) + 1, 1_000_000_000)
        return (cast("Callable[[], LABS]", lambda __i=i: cls(__i))
                for i in merge_sorted_and_return_unique((
                    16, 64, 100, 625), (k for k in default_scale_sequence(
                        scale_min, scale_max) if k in _BKS))
                if scale_min <= i <= scale_max)
