import appscaling = require('@aws-cdk/aws-applicationautoscaling');
import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import ec2 = require('@aws-cdk/aws-ec2');
import elb = require('@aws-cdk/aws-elasticloadbalancing');
import elbv2 = require('@aws-cdk/aws-elasticloadbalancingv2');
import cloudmap = require('@aws-cdk/aws-servicediscovery');
import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
import { LoadBalancerTargetOptions, TaskDefinition } from '../base/task-definition';
import { ICluster } from '../cluster';
import { Protocol } from '../container-definition';
import { CfnService } from '../ecs.generated';
import { ScalableTaskCount } from './scalable-task-count';
/**
 * The interface for a service.
 */
export interface IService extends IResource {
    /**
     * The Amazon Resource Name (ARN) of the service.
     *
     * @attribute
     */
    readonly serviceArn: string;
}
export interface EcsTarget {
    /**
     * The name of the container.
     */
    readonly containerName: string;
    /**
     * The port number of the container. Only applicable when using application/network load balancers.
     *
     * @default - Container port of the first added port mapping.
     */
    readonly containerPort?: number;
    /**
     * The protocol used for the port mapping. Only applicable when using application load balancers.
     *
     * @default Protocol.TCP
     */
    readonly protocol?: Protocol;
    /**
     * ID for a target group to be created.
     */
    readonly newTargetGroupId: string;
    /**
     * Listener and properties for adding target group to the listener.
     */
    readonly listener: ListenerConfig;
}
/**
 * Interface for ECS load balancer target.
 */
export interface IEcsLoadBalancerTarget extends elbv2.IApplicationLoadBalancerTarget, elbv2.INetworkLoadBalancerTarget, elb.ILoadBalancerTarget {
}
/**
 * The properties for the base Ec2Service or FargateService service.
 */
export interface BaseServiceOptions {
    /**
     * The name of the cluster that hosts the service.
     */
    readonly cluster: ICluster;
    /**
     * The desired number of instantiations of the task definition to keep running on the service.
     *
     * @default 1
     */
    readonly desiredCount?: number;
    /**
     * The name of the service.
     *
     * @default - CloudFormation-generated name.
     */
    readonly serviceName?: string;
    /**
     * The maximum number of tasks, specified as a percentage of the Amazon ECS
     * service's DesiredCount value, that can run in a service during a
     * deployment.
     *
     * @default - 100 if daemon, otherwise 200
     */
    readonly maxHealthyPercent?: number;
    /**
     * The minimum number of tasks, specified as a percentage of
     * the Amazon ECS service's DesiredCount value, that must
     * continue to run and remain healthy during a deployment.
     *
     * @default - 0 if daemon, otherwise 50
     */
    readonly minHealthyPercent?: number;
    /**
     * The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy
     * Elastic Load Balancing target health checks after a task has first started.
     *
     * @default - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
     */
    readonly healthCheckGracePeriod?: Duration;
    /**
     * The options for configuring an Amazon ECS service to use service discovery.
     *
     * @default - AWS Cloud Map service discovery is not enabled.
     */
    readonly cloudMapOptions?: CloudMapOptions;
    /**
     * Specifies whether to propagate the tags from the task definition or the service to the tasks in the service
     *
     * Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE
     *
     * @default PropagatedTagSource.NONE
     */
    readonly propagateTags?: PropagatedTagSource;
    /**
     * Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see
     * [Tagging Your Amazon ECS Resources](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html)
     *
     * @default false
     */
    readonly enableECSManagedTags?: boolean;
}
/**
 * Complete base service properties that are required to be supplied by the implementation
 * of the BaseService class.
 */
export interface BaseServiceProps extends BaseServiceOptions {
    /**
     * The launch type on which to run your service.
     *
     * Valid values are: LaunchType.ECS or LaunchType.FARGATE
     */
    readonly launchType: LaunchType;
}
/**
 * Base class for configuring listener when registering targets.
 */
export declare abstract class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener: elbv2.ApplicationListener, props?: elbv2.AddApplicationTargetsProps): ListenerConfig;
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener: elbv2.NetworkListener, props?: elbv2.AddNetworkTargetsProps): ListenerConfig;
    /**
     * Create and attach a target group to listener.
     */
    abstract addTargets(id: string, target: LoadBalancerTargetOptions, service: BaseService): void;
}
/**
 * The base class for Ec2Service and FargateService services.
 */
export declare abstract class BaseService extends Resource implements IService, elbv2.IApplicationLoadBalancerTarget, elbv2.INetworkLoadBalancerTarget, elb.ILoadBalancerTarget {
    /**
     * The security groups which manage the allowed network traffic for the service.
     */
    readonly connections: ec2.Connections;
    /**
     * The Amazon Resource Name (ARN) of the service.
     */
    readonly serviceArn: string;
    /**
     * The name of the service.
     *
     * @attribute
     */
    readonly serviceName: string;
    /**
     * The task definition to use for tasks in the service.
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * The cluster that hosts the service.
     */
    readonly cluster: ICluster;
    /**
     * The details of the AWS Cloud Map service.
     */
    protected cloudmapService?: cloudmap.Service;
    /**
     * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
     * name (as it appears in a container definition), and the container port to access from the load balancer.
     */
    protected loadBalancers: CfnService.LoadBalancerProperty[];
    /**
     * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
     * name (as it appears in a container definition), and the container port to access from the load balancer.
     */
    protected networkConfiguration?: CfnService.NetworkConfigurationProperty;
    /**
     * The details of the service discovery registries to assign to this service.
     * For more information, see Service Discovery.
     */
    protected serviceRegistries: CfnService.ServiceRegistryProperty[];
    private readonly resource;
    private scalableTaskCount?;
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope: Construct, id: string, props: BaseServiceProps, additionalProps: any, taskDefinition: TaskDefinition);
    /**
     * The CloudMap service created for this service, if any.
     */
    readonly cloudMapService: cloudmap.IService | undefined;
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup: elbv2.IApplicationTargetGroup): elbv2.LoadBalancerTargetProps;
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer: elb.LoadBalancer): void;
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * listener.addTargets(service.loadBalancerTarget({
     *   containerName: 'MyContainer',
     *   containerPort: 1234
     * }));
     */
    loadBalancerTarget(options: LoadBalancerTargetOptions): IEcsLoadBalancerTarget;
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * service.registerLoadBalancerTargets(
     *   {
     *     containerTarget: {
     *       containerName: 'web',
     *       containerPort: 80,
     *     },
     *     targetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets: EcsTarget[]): void;
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup: elbv2.INetworkTargetGroup): elbv2.LoadBalancerTargetProps;
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props: appscaling.EnableScalingProps): ScalableTaskCount;
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options: CloudMapOptions): cloudmap.Service;
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method is called to create a networkConfiguration.
     */
    protected configureAwsVpcNetworking(vpc: ec2.IVpc, assignPublicIp?: boolean, vpcSubnets?: ec2.SubnetSelection, securityGroup?: ec2.ISecurityGroup): void;
    private renderServiceRegistry;
    /**
     * Shared logic for attaching to an ELB
     */
    private attachToELB;
    /**
     * Shared logic for attaching to an ELBv2
     */
    private attachToELBv2;
    private readonly defaultLoadBalancerTarget;
    /**
     * Generate the role that will be used for autoscaling this service
     */
    private makeAutoScalingRole;
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    private addServiceRegistry;
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    private evaluateHealthGracePeriod;
}
/**
 * The options to enabling AWS Cloud Map for an Amazon ECS service.
 */
export interface CloudMapOptions {
    /**
     * The name of the Cloud Map service to attach to the ECS service.
     *
     * @default CloudFormation-generated name
     */
    readonly name?: string;
    /**
     * The service discovery namespace for the Cloud Map service to attach to the ECS service.
     *
     * @default - the defaultCloudMapNamespace associated to the cluster
     */
    readonly cloudMapNamespace?: cloudmap.INamespace;
    /**
     * The DNS record type that you want AWS Cloud Map to create. The supported record types are A or SRV.
     *
     * @default DnsRecordType.A
     */
    readonly dnsRecordType?: cloudmap.DnsRecordType.A | cloudmap.DnsRecordType.SRV;
    /**
     * The amount of time that you want DNS resolvers to cache the settings for this record.
     *
     * @default 60
     */
    readonly dnsTtl?: Duration;
    /**
     * The number of 30-second intervals that you want Cloud Map to wait after receiving an UpdateInstanceCustomHealthStatus
     * request before it changes the health status of a service instance.
     *
     * NOTE: This is used for HealthCheckCustomConfig
     */
    readonly failureThreshold?: number;
}
/**
 * The launch type of an ECS service
 */
export declare enum LaunchType {
    /**
     * The service will be launched using the EC2 launch type
     */
    EC2 = "EC2",
    /**
     * The service will be launched using the FARGATE launch type
     */
    FARGATE = "FARGATE"
}
/**
 * Propagate tags from either service or task definition
 */
export declare enum PropagatedTagSource {
    /**
     * Propagate tags from service
     */
    SERVICE = "SERVICE",
    /**
     * Propagate tags from task definition
     */
    TASK_DEFINITION = "TASK_DEFINITION",
    /**
     * Do not propagate
     */
    NONE = "NONE"
}
