"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const core_1 = require("@aws-cdk/core");
const task_definition_1 = require("../base/task-definition");
const ecs_generated_1 = require("../ecs.generated");
const scalable_task_count_1 = require("./scalable-task-count");
/**
 * Base class for configuring listener when registering targets.
 */
class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener, props) {
        return new ApplicationListenerConfig(listener, props);
    }
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener, props) {
        return new NetworkListenerConfig(listener, props);
    }
}
exports.ListenerConfig = ListenerConfig;
/**
 * Class for configuring application load balancer listener when registering targets.
 */
class ApplicationListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const props = this.props || {};
        const protocol = props.protocol;
        const port = props.port !== undefined ? props.port : (protocol === undefined ? 80 :
            (protocol === elbv2.ApplicationProtocol.HTTPS ? 443 : 80));
        this.listener.addTargets(id, Object.assign(Object.assign({}, props), { targets: [
                service.loadBalancerTarget(Object.assign({}, target))
            ], port }));
    }
}
/**
 * Class for configuring network load balancer listener when registering targets.
 */
class NetworkListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const port = this.props !== undefined ? this.props.port : 80;
        this.listener.addTargets(id, Object.assign(Object.assign({}, this.props), { targets: [
                service.loadBalancerTarget(Object.assign({}, target))
            ], port }));
    }
}
/**
 * The base class for Ec2Service and FargateService services.
 */
class BaseService extends core_1.Resource {
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope, id, props, additionalProps, taskDefinition) {
        super(scope, id, {
            physicalName: props.serviceName,
        });
        /**
         * The security groups which manage the allowed network traffic for the service.
         */
        this.connections = new ec2.Connections();
        /**
         * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
         * name (as it appears in a container definition), and the container port to access from the load balancer.
         */
        this.loadBalancers = new Array();
        /**
         * The details of the service discovery registries to assign to this service.
         * For more information, see Service Discovery.
         */
        this.serviceRegistries = new Array();
        this.taskDefinition = taskDefinition;
        this.resource = new ecs_generated_1.CfnService(this, "Service", Object.assign({ desiredCount: props.desiredCount, serviceName: this.physicalName, loadBalancers: core_1.Lazy.anyValue({ produce: () => this.loadBalancers }, { omitEmptyArray: true }), deploymentConfiguration: {
                maximumPercent: props.maxHealthyPercent || 200,
                minimumHealthyPercent: props.minHealthyPercent === undefined ? 50 : props.minHealthyPercent
            }, propagateTags: props.propagateTags === PropagatedTagSource.NONE ? undefined : props.propagateTags, enableEcsManagedTags: props.enableECSManagedTags === undefined ? false : props.enableECSManagedTags, launchType: props.launchType, healthCheckGracePeriodSeconds: this.evaluateHealthGracePeriod(props.healthCheckGracePeriod), 
            /* role: never specified, supplanted by Service Linked Role */
            networkConfiguration: core_1.Lazy.anyValue({ produce: () => this.networkConfiguration }, { omitEmptyArray: true }), serviceRegistries: core_1.Lazy.anyValue({ produce: () => this.serviceRegistries }, { omitEmptyArray: true }) }, additionalProps));
        this.serviceArn = this.getResourceArnAttribute(this.resource.ref, {
            service: 'ecs',
            resource: 'service',
            resourceName: `${props.cluster.clusterName}/${this.physicalName}`,
        });
        this.serviceName = this.getResourceNameAttribute(this.resource.attrName);
        this.cluster = props.cluster;
        if (props.cloudMapOptions) {
            this.enableCloudMap(props.cloudMapOptions);
        }
    }
    /**
     * The CloudMap service created for this service, if any.
     */
    get cloudMapService() {
        return this.cloudmapService;
    }
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToApplicationTargetGroup(targetGroup);
    }
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer) {
        return this.defaultLoadBalancerTarget.attachToClassicLB(loadBalancer);
    }
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * listener.addTargets(service.loadBalancerTarget({
     *   containerName: 'MyContainer',
     *   containerPort: 1234
     * }));
     */
    loadBalancerTarget(options) {
        const self = this;
        const target = this.taskDefinition._validateTarget(options);
        const connections = self.connections;
        return {
            attachToApplicationTargetGroup(targetGroup) {
                targetGroup.registerConnectable(self, self.taskDefinition._portRangeFromPortMapping(target.portMapping));
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            attachToNetworkTargetGroup(targetGroup) {
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            connections,
            attachToClassicLB(loadBalancer) {
                return self.attachToELB(loadBalancer, target.containerName, target.portMapping.containerPort);
            }
        };
    }
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * service.registerLoadBalancerTargets(
     *   {
     *     containerTarget: {
     *       containerName: 'web',
     *       containerPort: 80,
     *     },
     *     targetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets) {
        for (const target of targets) {
            target.listener.addTargets(target.newTargetGroupId, {
                containerName: target.containerName,
                containerPort: target.containerPort,
                protocol: target.protocol
            }, this);
        }
    }
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToNetworkTargetGroup(targetGroup);
    }
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props) {
        if (this.scalableTaskCount) {
            throw new Error('AutoScaling of task count already enabled for this service');
        }
        return this.scalableTaskCount = new scalable_task_count_1.ScalableTaskCount(this, 'TaskCount', Object.assign({ serviceNamespace: appscaling.ServiceNamespace.ECS, resourceId: `service/${this.cluster.clusterName}/${this.serviceName}`, dimension: 'ecs:service:DesiredCount', role: this.makeAutoScalingRole() }, props));
    }
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options) {
        const sdNamespace = options.cloudMapNamespace !== undefined ? options.cloudMapNamespace : this.cluster.defaultCloudMapNamespace;
        if (sdNamespace === undefined) {
            throw new Error("Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster.");
        }
        // Determine DNS type based on network mode
        const networkMode = this.taskDefinition.networkMode;
        if (networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error("Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead.");
        }
        // Bridge or host network mode requires SRV records
        let dnsRecordType = options.dnsRecordType;
        if (networkMode === task_definition_1.NetworkMode.BRIDGE || networkMode === task_definition_1.NetworkMode.HOST) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.SRV;
            }
            if (dnsRecordType !== cloudmap.DnsRecordType.SRV) {
                throw new Error("SRV records must be used when network mode is Bridge or Host.");
            }
        }
        // Default DNS record type for AwsVpc network mode is A Records
        if (networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.A;
            }
        }
        // If the task definition that your service task specifies uses the AWSVPC network mode and a type SRV DNS record is
        // used, you must specify a containerName and containerPort combination
        const containerName = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerName : undefined;
        const containerPort = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerPort : undefined;
        const cloudmapService = new cloudmap.Service(this, 'CloudmapService', {
            namespace: sdNamespace,
            name: options.name,
            dnsRecordType: dnsRecordType,
            customHealthCheck: { failureThreshold: options.failureThreshold || 1 }
        });
        const serviceArn = cloudmapService.serviceArn;
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: serviceArn,
            containerName,
            containerPort
        });
        this.cloudmapService = cloudmapService;
        return cloudmapService;
    }
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric(Object.assign({ namespace: 'AWS/ECS', metricName, dimensions: { ClusterName: this.cluster.clusterName, ServiceName: this.serviceName } }, props));
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        return this.metric('MemoryUtilization', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('CPUUtilization', props);
    }
    /**
     * This method is called to create a networkConfiguration.
     */
    // tslint:disable-next-line:max-line-length
    configureAwsVpcNetworking(vpc, assignPublicIp, vpcSubnets, securityGroup) {
        if (vpcSubnets === undefined) {
            vpcSubnets = { subnetType: assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        }
        if (securityGroup === undefined) {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc });
        }
        this.connections.addSecurityGroup(securityGroup);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: core_1.Lazy.listValue({ produce: () => [securityGroup.securityGroupId] }),
            }
        };
    }
    renderServiceRegistry(registry) {
        return {
            registryArn: registry.arn,
            containerName: registry.containerName,
            containerPort: registry.containerPort,
        };
    }
    /**
     * Shared logic for attaching to an ELB
     */
    attachToELB(loadBalancer, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            throw new Error("Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead.");
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error("Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead.");
        }
        this.loadBalancers.push({
            loadBalancerName: loadBalancer.loadBalancerName,
            containerName,
            containerPort
        });
    }
    /**
     * Shared logic for attaching to an ELBv2
     */
    attachToELBv2(targetGroup, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error("Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead.");
        }
        this.loadBalancers.push({
            targetGroupArn: targetGroup.targetGroupArn,
            containerName,
            containerPort,
        });
        // Service creation can only happen after the load balancer has
        // been associated with our target group(s), so add ordering dependency.
        this.resource.node.addDependency(targetGroup.loadBalancerAttached);
        const targetType = this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC ? elbv2.TargetType.IP : elbv2.TargetType.INSTANCE;
        return { targetType };
    }
    get defaultLoadBalancerTarget() {
        return this.loadBalancerTarget({
            containerName: this.taskDefinition.defaultContainer.containerName
        });
    }
    /**
     * Generate the role that will be used for autoscaling this service
     */
    makeAutoScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            region: '',
            service: 'iam',
            resource: 'role/aws-service-role/ecs.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_ECSService',
        }));
    }
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    addServiceRegistry(registry) {
        const sr = this.renderServiceRegistry(registry);
        this.serviceRegistries.push(sr);
    }
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    evaluateHealthGracePeriod(providedHealthCheckGracePeriod) {
        return core_1.Lazy.anyValue({
            produce: () => providedHealthCheckGracePeriod !== undefined ? providedHealthCheckGracePeriod.toSeconds() :
                this.loadBalancers.length > 0 ? 60 :
                    undefined
        });
    }
}
exports.BaseService = BaseService;
/**
 * The launch type of an ECS service
 */
var LaunchType;
(function (LaunchType) {
    /**
     * The service will be launched using the EC2 launch type
     */
    LaunchType["EC2"] = "EC2";
    /**
     * The service will be launched using the FARGATE launch type
     */
    LaunchType["FARGATE"] = "FARGATE";
})(LaunchType = exports.LaunchType || (exports.LaunchType = {}));
/**
 * Propagate tags from either service or task definition
 */
var PropagatedTagSource;
(function (PropagatedTagSource) {
    /**
     * Propagate tags from service
     */
    PropagatedTagSource["SERVICE"] = "SERVICE";
    /**
     * Propagate tags from task definition
     */
    PropagatedTagSource["TASK_DEFINITION"] = "TASK_DEFINITION";
    /**
     * Do not propagate
     */
    PropagatedTagSource["NONE"] = "NONE";
})(PropagatedTagSource = exports.PropagatedTagSource || (exports.PropagatedTagSource = {}));
//# sourceMappingURL=data:application/json;base64,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