"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
/**
 * The scalable attribute representing task count.
 */
class ScalableTaskCount extends appscaling.BaseScalableAttribute {
    /**
     * Constructs a new instance of the ScalableTaskCount class.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Scales in or out based on a specified scheduled time.
     */
    scaleOnSchedule(id, props) {
        return super.doScaleOnSchedule(id, props);
    }
    /**
     * Scales in or out based on a specified metric value.
     */
    scaleOnMetric(id, props) {
        return super.doScaleOnMetric(id, props);
    }
    /**
     * Scales in or out to achieve a target CPU utilization.
     */
    scaleOnCpuUtilization(id, props) {
        return super.doScaleToTrackMetric(id, {
            predefinedMetric: appscaling.PredefinedMetric.ECS_SERVICE_AVERAGE_CPU_UTILIZATION,
            policyName: props.policyName,
            disableScaleIn: props.disableScaleIn,
            targetValue: props.targetUtilizationPercent,
            scaleInCooldown: props.scaleInCooldown,
            scaleOutCooldown: props.scaleOutCooldown
        });
    }
    /**
     * Scales in or out to achieve a target memory utilization.
     */
    scaleOnMemoryUtilization(id, props) {
        return super.doScaleToTrackMetric(id, {
            predefinedMetric: appscaling.PredefinedMetric.ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            policyName: props.policyName,
            disableScaleIn: props.disableScaleIn,
            scaleInCooldown: props.scaleInCooldown,
            scaleOutCooldown: props.scaleOutCooldown
        });
    }
    /**
     * Scales in or out to achieve a target Application Load Balancer request count per target.
     */
    scaleOnRequestCount(id, props) {
        const resourceLabel = props.targetGroup.firstLoadBalancerFullName +
            '/' + props.targetGroup.targetGroupFullName;
        return super.doScaleToTrackMetric(id, {
            predefinedMetric: appscaling.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            resourceLabel,
            targetValue: props.requestsPerTarget,
            policyName: props.policyName,
            disableScaleIn: props.disableScaleIn,
            scaleInCooldown: props.scaleInCooldown,
            scaleOutCooldown: props.scaleOutCooldown
        });
    }
    /**
     * Scales in or out to achieve a target on a custom metric.
     */
    scaleToTrackCustomMetric(id, props) {
        return super.doScaleToTrackMetric(id, {
            customMetric: props.metric,
            targetValue: props.targetValue,
            policyName: props.policyName,
            disableScaleIn: props.disableScaleIn,
            scaleInCooldown: props.scaleInCooldown,
            scaleOutCooldown: props.scaleOutCooldown,
        });
    }
}
exports.ScalableTaskCount = ScalableTaskCount;
//# sourceMappingURL=data:application/json;base64,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