"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || this.node.uniqueId;
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.containers.map(x => x.renderContainerDefinition(this)) }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.volumes }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ["EC2"] : []),
                ...(isFargateCompatible(props.compatibility) ? ["FARGATE"] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({ produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    get executionRole() {
        return this._executionRole;
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer._findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMapping()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, Object.assign({ taskDefinition: this }, props));
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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