"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const autoscaling = require("@aws-cdk/aws-autoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_generated_1 = require("./ecs.generated");
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
class Cluster extends core_1.Resource {
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * Manage the allowed network connections for the cluster with Security Groups.
         */
        this.connections = new ec2.Connections();
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity("DefaultAutoScalingGroup", props.capacity)
            : undefined;
    }
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options) {
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error("Can only add default namespace once.");
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id, options) {
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, Object.assign(Object.assign({}, options), { vpc: this.vpc, machineImage: options.machineImage || new EcsOptimizedAmi(), updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE, instanceType: options.instanceType }));
        this.addAutoScalingGroup(autoScalingGroup, options);
        return autoScalingGroup;
    }
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        // Tie instances to cluster
        autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
        if (!options.canContainersAccessInstanceRole) {
            // Deny containers access to instance metadata service
            // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
            autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
            autoScalingGroup.addUserData('sudo service iptables save');
            // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
            autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
        }
        if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
            autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                "ecs:CreateCluster",
                "ecs:DeregisterContainerInstance",
                "ecs:DiscoverPollEndpoint",
                "ecs:Poll",
                "ecs:RegisterContainerInstance",
                "ecs:StartTelemetrySession",
                "ecs:Submit*",
                "ecr:GetAuthorizationToken",
                "logs:CreateLogStream",
                "logs:PutLogEvents"
            ],
            resources: ['*']
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime
            });
        }
    }
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props) {
        return this.metric('CPUReservation', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props) {
        return this.metric('MemoryReservation', props);
    }
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric(Object.assign({ namespace: 'AWS/ECS', metricName, dimensions: { ClusterName: this.clusterName } }, props));
    }
}
exports.Cluster = Cluster;
/**
 * ECS-optimized Windows version list
 */
var WindowsOptimizedVersion;
(function (WindowsOptimizedVersion) {
    WindowsOptimizedVersion["SERVER_2019"] = "2019";
    WindowsOptimizedVersion["SERVER_2016"] = "2016";
})(WindowsOptimizedVersion = exports.WindowsOptimizedVersion || (exports.WindowsOptimizedVersion = {}));
/*
 * TODO:v2.0.0 remove EcsOptimizedAmi
 */
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
class EcsOptimizedAmi {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = (props && props.hardwareType) || AmiHardwareType.STANDARD;
        if (props && props.generation) { // generation defined in the props object
            if (props.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX && this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error(`Amazon Linux does not support special hardware type. Use Amazon Linux 2 instead`);
            }
            else if (props.windowsVersion) {
                throw new Error('"windowsVersion" and Linux image "generation" cannot be both set');
            }
            else {
                this.generation = props.generation;
            }
        }
        else if (props && props.windowsVersion) {
            if (this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Windows Server does not support special hardware type');
            }
            else {
                this.windowsVersion = props.windowsVersion;
            }
        }
        else { // generation not defined in props object
            // always default to Amazon Linux v2 regardless of HW
            this.generation = ec2.AmazonLinuxGeneration.AMAZON_LINUX_2;
        }
        // set the SSM parameter name
        this.amiParameterName = "/aws/service/ecs/optimized-ami/"
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? "amazon-linux/" : "")
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? "amazon-linux-2/" : "")
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : "")
            + (this.hwType === AmiHardwareType.GPU ? "gpu/" : "")
            + (this.hwType === AmiHardwareType.ARM ? "arm64/" : "")
            + "recommended/image_id";
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX
        };
    }
}
exports.EcsOptimizedAmi = EcsOptimizedAmi;
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 */
class EcsOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = props && props.hardwareType;
        if (props.windowsVersion) {
            this.windowsVersion = props.windowsVersion;
        }
        else if (props.generation) {
            this.generation = props.generation;
        }
        else {
            throw new Error('This error should never be thrown');
        }
        // set the SSM parameter name
        this.amiParameterName = "/aws/service/ecs/optimized-ami/"
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? "amazon-linux/" : "")
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? "amazon-linux-2/" : "")
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : "")
            + (this.hwType === AmiHardwareType.GPU ? "gpu/" : "")
            + (this.hwType === AmiHardwareType.ARM ? "arm64/" : "")
            + "recommended/image_id";
    }
    /**
     * Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM
     *
     * @param hardwareType ECS-optimized AMI variant to use
     */
    static amazonLinux2(hardwareType = AmiHardwareType.STANDARD) {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2, hardwareType });
    }
    /**
     * Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM
     */
    static amazonLinux() {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX });
    }
    /**
     * Construct a Windows image from the latest ECS Optimized AMI published in SSM
     *
     * @param windowsVersion Windows Version to use
     */
    static windows(windowsVersion) {
        return new EcsOptimizedImage({ windowsVersion });
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX
        };
    }
}
exports.EcsOptimizedImage = EcsOptimizedImage;
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this.clusterArn = props.clusterArn !== undefined ? props.clusterArn : core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
}
/**
 * The ECS-optimized AMI variant to use. For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 */
var AmiHardwareType;
(function (AmiHardwareType) {
    /**
     * Use the standard Amazon ECS-optimized AMI.
     */
    AmiHardwareType["STANDARD"] = "Standard";
    /**
     * Use the Amazon ECS GPU-optimized AMI.
     */
    AmiHardwareType["GPU"] = "GPU";
    /**
     * Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.
     */
    AmiHardwareType["ARM"] = "ARM64";
})(AmiHardwareType = exports.AmiHardwareType || (exports.AmiHardwareType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsdXN0ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3REFBeUQ7QUFDekQsc0RBQXVEO0FBQ3ZELHdDQUF5QztBQUN6Qyx3Q0FBeUM7QUFDekMsMERBQTJEO0FBQzNELHdDQUF5QztBQUN6Qyx3Q0FBZ0Y7QUFDaEYsMEVBQXFFO0FBQ3JFLG1EQUE2QztBQW9DN0M7O0dBRUc7QUFDSCxNQUFhLE9BQVEsU0FBUSxlQUFRO0lBMkNuQzs7T0FFRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBc0IsRUFBRTtRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVztTQUNoQyxDQUFDLENBQUM7UUF6Q0w7O1dBRUc7UUFDYSxnQkFBVyxHQUFvQixJQUFJLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQXNCckU7O1dBRUc7UUFDSyxvQkFBZSxHQUFZLEtBQUssQ0FBQztRQWV2QyxNQUFNLE9BQU8sR0FBRyxJQUFJLDBCQUFVLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMvQyxXQUFXLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDL0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRTtZQUM5RCxPQUFPLEVBQUUsS0FBSztZQUNkLFFBQVEsRUFBRSxTQUFTO1lBQ25CLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFOUQsSUFBSSxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsR0FBRyxJQUFJLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLEVBQUUsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFaEUsSUFBSSxDQUFDLHlCQUF5QixHQUFHLEtBQUssQ0FBQyx3QkFBd0IsS0FBSyxTQUFTO1lBQzNFLENBQUMsQ0FBQyxJQUFJLENBQUMsMkJBQTJCLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDO1lBQ2xFLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFFZCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLFFBQVEsS0FBSyxTQUFTO1lBQ25ELENBQUMsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLHlCQUF5QixFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUM7WUFDN0QsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUNoQixDQUFDO0lBdEVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLHFCQUFxQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXdCO1FBQ3hGLE9BQU8sSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBbUVEOzs7O09BSUc7SUFDSSwyQkFBMkIsQ0FBQyxPQUFpQztRQUNsRSxJQUFJLElBQUksQ0FBQyx5QkFBeUIsS0FBSyxTQUFTLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLElBQUksS0FBSyxTQUFTO1lBQzlDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSTtZQUNkLENBQUMsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQztRQUV2QyxNQUFNLFdBQVcsR0FBRyxhQUFhLEtBQUssUUFBUSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN4RSxJQUFJLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsa0NBQWtDLEVBQUU7Z0JBQ3pFLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2FBQ2QsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsa0NBQWtDLEVBQUU7Z0JBQ3hFLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTthQUNuQixDQUFDLENBQUM7UUFFTCxJQUFJLENBQUMseUJBQXlCLEdBQUcsV0FBVyxDQUFDO1FBRTdDLE9BQU8sV0FBVyxDQUFDO0lBQ3JCLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsd0JBQXdCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLHlCQUF5QixDQUFDO0lBQ3hDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksV0FBVyxDQUFDLEVBQVUsRUFBRSxPQUEyQjtRQUN4RCxNQUFNLGdCQUFnQixHQUFHLElBQUksV0FBVyxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxFQUFFLGtDQUM3RCxPQUFPLEtBQ1YsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQ2IsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZLElBQUksSUFBSSxlQUFlLEVBQUUsRUFDM0QsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVLElBQUksV0FBVyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsRUFDekUsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZLElBQ2xDLENBQUM7UUFFSCxJQUFJLENBQUMsbUJBQW1CLENBQUMsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFcEQsT0FBTyxnQkFBZ0IsQ0FBQztJQUMxQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CLENBQUMsZ0JBQThDLEVBQUUsVUFBOEMsRUFBRTtRQUN6SCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQztRQUM1QixJQUFJLENBQUMsV0FBVyxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLGdCQUFnQixDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUU5RiwyQkFBMkI7UUFDM0IsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLG9CQUFvQixJQUFJLENBQUMsV0FBVyx5QkFBeUIsQ0FBQyxDQUFDO1FBRTVGLElBQUksQ0FBQyxPQUFPLENBQUMsK0JBQStCLEVBQUU7WUFDNUMsc0RBQXNEO1lBQ3RELDZGQUE2RjtZQUM3RixnQkFBZ0IsQ0FBQyxXQUFXLENBQUMsc0dBQXNHLENBQUMsQ0FBQztZQUNySSxnQkFBZ0IsQ0FBQyxXQUFXLENBQUMsNEJBQTRCLENBQUMsQ0FBQztZQUMzRCwwRkFBMEY7WUFDMUYsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDeEY7UUFFRCxJQUFJLGdCQUFnQixDQUFDLFNBQVMsSUFBSSxPQUFPLENBQUMsb0JBQW9CLEVBQUU7WUFDOUQsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLG9FQUFvRSxDQUFDLENBQUM7U0FDcEc7UUFFRCxnREFBZ0Q7UUFDaEQsNkZBQTZGO1FBQzdGLGdCQUFnQixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdkQsT0FBTyxFQUFFO2dCQUNQLG1CQUFtQjtnQkFDbkIsaUNBQWlDO2dCQUNqQywwQkFBMEI7Z0JBQzFCLFVBQVU7Z0JBQ1YsK0JBQStCO2dCQUMvQiwyQkFBMkI7Z0JBQzNCLGFBQWE7Z0JBQ2IsMkJBQTJCO2dCQUMzQixzQkFBc0I7Z0JBQ3RCLG1CQUFtQjthQUNwQjtZQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUVKLDBGQUEwRjtRQUMxRixJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsSUFBSSxPQUFPLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsRUFBRTtZQUNyRSxJQUFJLHVDQUFpQixDQUFDLGdCQUFnQixFQUFFLGNBQWMsRUFBRTtnQkFDdEQsZ0JBQWdCO2dCQUNoQixPQUFPLEVBQUUsSUFBSTtnQkFDYixTQUFTLEVBQUUsT0FBTyxDQUFDLGFBQWE7YUFDakMsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLGdCQUFnQjtRQUN6QixPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUFDO0lBQzlCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksb0JBQW9CLENBQUMsS0FBZ0M7UUFDMUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksdUJBQXVCLENBQUMsS0FBZ0M7UUFDN0QsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxVQUFrQixFQUFFLEtBQWdDO1FBQ2hFLE9BQU8sSUFBSSxVQUFVLENBQUMsTUFBTSxpQkFDMUIsU0FBUyxFQUFFLFNBQVMsRUFDcEIsVUFBVSxFQUNWLFVBQVUsRUFBRSxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVyxFQUFFLElBQzFDLEtBQUssRUFDUixDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBak9ELDBCQWlPQztBQUVEOztHQUVHO0FBQ0gsSUFBWSx1QkFHWDtBQUhELFdBQVksdUJBQXVCO0lBQ2pDLCtDQUFvQixDQUFBO0lBQ3BCLCtDQUFvQixDQUFBO0FBQ3RCLENBQUMsRUFIVyx1QkFBdUIsR0FBdkIsK0JBQXVCLEtBQXZCLCtCQUF1QixRQUdsQztBQW1DRDs7R0FFRztBQUNIOzs7O0dBSUc7QUFDSCxNQUFhLGVBQWU7SUFPMUI7O09BRUc7SUFDSCxZQUFZLEtBQTRCO1FBQ3RDLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxLQUFLLElBQUksS0FBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLGVBQWUsQ0FBQyxRQUFRLENBQUM7UUFDeEUsSUFBSSxLQUFLLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRSxFQUFPLHlDQUF5QztZQUM3RSxJQUFJLEtBQUssQ0FBQyxVQUFVLEtBQUssR0FBRyxDQUFDLHFCQUFxQixDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLGVBQWUsQ0FBQyxRQUFRLEVBQUU7Z0JBQzNHLE1BQU0sSUFBSSxLQUFLLENBQUMsaUZBQWlGLENBQUMsQ0FBQzthQUNwRztpQkFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7Z0JBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQzthQUNyRjtpQkFBTTtnQkFDTCxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7YUFDcEM7U0FDRjthQUFNLElBQUksS0FBSyxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDeEMsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLGVBQWUsQ0FBQyxRQUFRLEVBQUU7Z0JBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQzthQUMxRTtpQkFBTTtnQkFDTCxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7YUFDNUM7U0FDRjthQUFNLEVBQStCLHlDQUF5QztZQUM3RSxxREFBcUQ7WUFDckQsSUFBSSxDQUFDLFVBQVUsR0FBRyxHQUFHLENBQUMscUJBQXFCLENBQUMsY0FBYyxDQUFDO1NBQzVEO1FBRUQsNkJBQTZCO1FBQzdCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxpQ0FBaUM7Y0FDckQsQ0FBQyxJQUFJLENBQUMsVUFBVSxLQUFLLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2NBQ25GLENBQUMsSUFBSSxDQUFDLFVBQVUsS0FBSyxHQUFHLENBQUMscUJBQXFCLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2NBQ3ZGLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsa0JBQWtCLElBQUksQ0FBQyxjQUFjLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Y0FDbEYsQ0FBQyxJQUFJLENBQUMsTUFBTSxLQUFLLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2NBQ25ELENBQUMsSUFBSSxDQUFDLE1BQU0sS0FBSyxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztjQUNyRCxzQkFBc0IsQ0FBQztJQUM3QixDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRLENBQUMsS0FBZ0I7UUFDOUIsTUFBTSxHQUFHLEdBQUcsR0FBRyxDQUFDLGVBQWUsQ0FBQyw0QkFBNEIsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUMvSCxPQUFPO1lBQ0wsT0FBTyxFQUFFLEdBQUc7WUFDWixNQUFNLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLEtBQUs7U0FDOUYsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQW5ERCwwQ0FtREM7QUFFRDs7R0FFRztBQUNILE1BQWEsaUJBQWlCO0lBZ0M1Qjs7T0FFRztJQUNILFlBQW9CLEtBQTJCO1FBQzdDLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFFMUMsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQ3hCLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztTQUM1QzthQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUMzQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7U0FDcEM7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsbUNBQW1DLENBQUMsQ0FBQztTQUN0RDtRQUVELDZCQUE2QjtRQUM3QixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsaUNBQWlDO2NBQ3JELENBQUMsSUFBSSxDQUFDLFVBQVUsS0FBSyxHQUFHLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztjQUNuRixDQUFDLElBQUksQ0FBQyxVQUFVLEtBQUssR0FBRyxDQUFDLHFCQUFxQixDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztjQUN2RixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLGtCQUFrQixJQUFJLENBQUMsY0FBYyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2NBQ2xGLENBQUMsSUFBSSxDQUFDLE1BQU0sS0FBSyxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztjQUNuRCxDQUFDLElBQUksQ0FBQyxNQUFNLEtBQUssZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Y0FDckQsc0JBQXNCLENBQUM7SUFDN0IsQ0FBQztJQXJERDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxZQUFZLEdBQUcsZUFBZSxDQUFDLFFBQVE7UUFDaEUsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxjQUFjLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztJQUN2RyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsV0FBVztRQUN2QixPQUFPLElBQUksaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLFlBQVksRUFBRSxDQUFDLENBQUM7SUFDdkYsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQXVDO1FBQzNELE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQWdDRDs7T0FFRztJQUNJLFFBQVEsQ0FBQyxLQUFnQjtRQUM5QixNQUFNLEdBQUcsR0FBRyxHQUFHLENBQUMsZUFBZSxDQUFDLDRCQUE0QixDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQy9ILE9BQU87WUFDTCxPQUFPLEVBQUUsR0FBRztZQUNaLE1BQU0sRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsS0FBSztTQUM5RixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBbEVELDhDQWtFQztBQTRGRDs7R0FFRztBQUNILE1BQU0sZUFBZ0IsU0FBUSxlQUFRO0lBK0JwQzs7T0FFRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQW5CbkI7O1dBRUc7UUFDYSxnQkFBVyxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBaUJsRCxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsSUFBSSxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDO1FBQ3JCLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsS0FBSyxLQUFLLENBQUM7UUFDckQsSUFBSSxDQUFDLHlCQUF5QixHQUFHLEtBQUssQ0FBQyx3QkFBd0IsQ0FBQztRQUVoRSxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUM3RixPQUFPLEVBQUUsS0FBSztZQUNkLFFBQVEsRUFBRSxTQUFTO1lBQ25CLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVztTQUNoQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQztZQUNyQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7U0FDckMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELElBQVcsd0JBQXdCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLHlCQUF5QixDQUFDO0lBQ3hDLENBQUM7Q0FDRjtBQTZFRDs7O0dBR0c7QUFDSCxJQUFZLGVBZ0JYO0FBaEJELFdBQVksZUFBZTtJQUV6Qjs7T0FFRztJQUNILHdDQUFxQixDQUFBO0lBRXJCOztPQUVHO0lBQ0gsOEJBQVcsQ0FBQTtJQUVYOztPQUVHO0lBQ0gsZ0NBQWEsQ0FBQTtBQUNmLENBQUMsRUFoQlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUFnQjFCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGF1dG9zY2FsaW5nID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWF1dG9zY2FsaW5nJyk7XG5pbXBvcnQgY2xvdWR3YXRjaCA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJyk7XG5pbXBvcnQgZWMyID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWVjMicpO1xuaW1wb3J0IGlhbSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1pYW0nKTtcbmltcG9ydCBjbG91ZG1hcCA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1zZXJ2aWNlZGlzY292ZXJ5Jyk7XG5pbXBvcnQgc3NtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLXNzbScpO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiwgSVJlc291cmNlLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEluc3RhbmNlRHJhaW5Ib29rIH0gZnJvbSAnLi9kcmFpbi1ob29rL2luc3RhbmNlLWRyYWluLWhvb2snO1xuaW1wb3J0IHsgQ2ZuQ2x1c3RlciB9IGZyb20gJy4vZWNzLmdlbmVyYXRlZCc7XG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgdXNlZCB0byBkZWZpbmUgYW4gRUNTIGNsdXN0ZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2x1c3RlclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIGZvciB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQGRlZmF1bHQgQ2xvdWRGb3JtYXRpb24tZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXJOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHdoZXJlIHlvdXIgRUNTIGluc3RhbmNlcyB3aWxsIGJlIHJ1bm5pbmcgb3IgeW91ciBFTklzIHdpbGwgYmUgZGVwbG95ZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBjcmVhdGVzIGEgbmV3IFZQQyB3aXRoIHR3byBBWnNcbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgc2VydmljZSBkaXNjb3ZlcnkgbmFtZXNwYWNlIGNyZWF0ZWQgaW4gdGhpcyBjbHVzdGVyXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gc2VydmljZSBkaXNjb3ZlcnkgbmFtZXNwYWNlIGNyZWF0ZWQsIHlvdSBjYW4gdXNlIGBhZGREZWZhdWx0Q2xvdWRNYXBOYW1lc3BhY2VgIHRvIGFkZCBhXG4gICAqIGRlZmF1bHQgc2VydmljZSBkaXNjb3ZlcnkgbmFtZXNwYWNlIGxhdGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlPzogQ2xvdWRNYXBOYW1lc3BhY2VPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBUaGUgZWMyIGNhcGFjaXR5IHRvIGFkZCB0byB0aGUgY2x1c3RlclxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIEVDMiBjYXBhY2l0eSB3aWxsIGJlIGFkZGVkLCB5b3UgY2FuIHVzZSBgYWRkQ2FwYWNpdHlgIHRvIGFkZCBjYXBhY2l0eSBsYXRlci5cbiAgICovXG4gIHJlYWRvbmx5IGNhcGFjaXR5PzogQWRkQ2FwYWNpdHlPcHRpb25zO1xufVxuXG4vKipcbiAqIEEgcmVnaW9uYWwgZ3JvdXBpbmcgb2Ygb25lIG9yIG1vcmUgY29udGFpbmVyIGluc3RhbmNlcyBvbiB3aGljaCB5b3UgY2FuIHJ1biB0YXNrcyBhbmQgc2VydmljZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbHVzdGVyIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQ2x1c3RlciB7XG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgY2x1c3RlciB0byB0aGUgc3RhY2sgZnJvbSBpdHMgYXR0cmlidXRlcy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNsdXN0ZXJBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBDbHVzdGVyQXR0cmlidXRlcyk6IElDbHVzdGVyIHtcbiAgICByZXR1cm4gbmV3IEltcG9ydGVkQ2x1c3RlcihzY29wZSwgaWQsIGF0dHJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYW5hZ2UgdGhlIGFsbG93ZWQgbmV0d29yayBjb25uZWN0aW9ucyBmb3IgdGhlIGNsdXN0ZXIgd2l0aCBTZWN1cml0eSBHcm91cHMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoKTtcblxuICAvKipcbiAgICogVGhlIFZQQyBhc3NvY2lhdGVkIHdpdGggdGhlIGNsdXN0ZXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIHRoYXQgaWRlbnRpZmllcyB0aGUgY2x1c3Rlci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjbHVzdGVyQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBjbHVzdGVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNsdXN0ZXJOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBV1MgQ2xvdWQgTWFwIG5hbWVzcGFjZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgY2x1c3Rlci5cbiAgICovXG4gIHByaXZhdGUgX2RlZmF1bHRDbG91ZE1hcE5hbWVzcGFjZT86IGNsb3VkbWFwLklOYW1lc3BhY2U7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRoZSBjbHVzdGVyIGhhcyBFQzIgaW5zdGFuY2UgY2FwYWNpdHkuXG4gICAqL1xuICBwcml2YXRlIF9oYXNFYzJDYXBhY2l0eTogYm9vbGVhbiA9IGZhbHNlO1xuXG4gIC8qKlxuICAgKiBUaGUgYXV0b3NjYWxpbmcgZ3JvdXAgZm9yIGFkZGVkIEVjMiBjYXBhY2l0eVxuICAgKi9cbiAgcHJpdmF0ZSBfYXV0b3NjYWxpbmdHcm91cD86IGF1dG9zY2FsaW5nLklBdXRvU2NhbGluZ0dyb3VwO1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBDbHVzdGVyIGNsYXNzLlxuICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENsdXN0ZXJQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmNsdXN0ZXJOYW1lLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY2x1c3RlciA9IG5ldyBDZm5DbHVzdGVyKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNsdXN0ZXJOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuY2x1c3RlckFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoY2x1c3Rlci5hdHRyQXJuLCB7XG4gICAgICBzZXJ2aWNlOiAnZWNzJyxcbiAgICAgIHJlc291cmNlOiAnY2x1c3RlcicsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuICAgIHRoaXMuY2x1c3Rlck5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShjbHVzdGVyLnJlZik7XG5cbiAgICB0aGlzLnZwYyA9IHByb3BzLnZwYyB8fCBuZXcgZWMyLlZwYyh0aGlzLCAnVnBjJywgeyBtYXhBenM6IDIgfSk7XG5cbiAgICB0aGlzLl9kZWZhdWx0Q2xvdWRNYXBOYW1lc3BhY2UgPSBwcm9wcy5kZWZhdWx0Q2xvdWRNYXBOYW1lc3BhY2UgIT09IHVuZGVmaW5lZFxuICAgICAgPyB0aGlzLmFkZERlZmF1bHRDbG91ZE1hcE5hbWVzcGFjZShwcm9wcy5kZWZhdWx0Q2xvdWRNYXBOYW1lc3BhY2UpXG4gICAgICA6IHVuZGVmaW5lZDtcblxuICAgIHRoaXMuX2F1dG9zY2FsaW5nR3JvdXAgPSBwcm9wcy5jYXBhY2l0eSAhPT0gdW5kZWZpbmVkXG4gICAgICA/IHRoaXMuYWRkQ2FwYWNpdHkoXCJEZWZhdWx0QXV0b1NjYWxpbmdHcm91cFwiLCBwcm9wcy5jYXBhY2l0eSlcbiAgICAgIDogdW5kZWZpbmVkO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhbiBBV1MgQ2xvdWQgTWFwIEROUyBuYW1lc3BhY2UgZm9yIHRoaXMgY2x1c3Rlci5cbiAgICogTk9URTogSHR0cE5hbWVzcGFjZXMgYXJlIG5vdCBzdXBwb3J0ZWQsIGFzIEVDUyBhbHdheXMgcmVxdWlyZXMgYSBETlNDb25maWcgd2hlbiByZWdpc3RlcmluZyBhbiBpbnN0YW5jZSB0byBhIENsb3VkXG4gICAqIE1hcCBzZXJ2aWNlLlxuICAgKi9cbiAgcHVibGljIGFkZERlZmF1bHRDbG91ZE1hcE5hbWVzcGFjZShvcHRpb25zOiBDbG91ZE1hcE5hbWVzcGFjZU9wdGlvbnMpOiBjbG91ZG1hcC5JTmFtZXNwYWNlIHtcbiAgICBpZiAodGhpcy5fZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIkNhbiBvbmx5IGFkZCBkZWZhdWx0IG5hbWVzcGFjZSBvbmNlLlwiKTtcbiAgICB9XG5cbiAgICBjb25zdCBuYW1lc3BhY2VUeXBlID0gb3B0aW9ucy50eXBlICE9PSB1bmRlZmluZWRcbiAgICAgID8gb3B0aW9ucy50eXBlXG4gICAgICA6IGNsb3VkbWFwLk5hbWVzcGFjZVR5cGUuRE5TX1BSSVZBVEU7XG5cbiAgICBjb25zdCBzZE5hbWVzcGFjZSA9IG5hbWVzcGFjZVR5cGUgPT09IGNsb3VkbWFwLk5hbWVzcGFjZVR5cGUuRE5TX1BSSVZBVEUgP1xuICAgICAgbmV3IGNsb3VkbWFwLlByaXZhdGVEbnNOYW1lc3BhY2UodGhpcywgJ0RlZmF1bHRTZXJ2aWNlRGlzY292ZXJ5TmFtZXNwYWNlJywge1xuICAgICAgICBuYW1lOiBvcHRpb25zLm5hbWUsXG4gICAgICAgIHZwYzogdGhpcy52cGNcbiAgICAgIH0pIDpcbiAgICAgIG5ldyBjbG91ZG1hcC5QdWJsaWNEbnNOYW1lc3BhY2UodGhpcywgJ0RlZmF1bHRTZXJ2aWNlRGlzY292ZXJ5TmFtZXNwYWNlJywge1xuICAgICAgICBuYW1lOiBvcHRpb25zLm5hbWUsXG4gICAgICB9KTtcblxuICAgIHRoaXMuX2RlZmF1bHRDbG91ZE1hcE5hbWVzcGFjZSA9IHNkTmFtZXNwYWNlO1xuXG4gICAgcmV0dXJuIHNkTmFtZXNwYWNlO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHRlciBmb3IgbmFtZXNwYWNlIGFkZGVkIHRvIGNsdXN0ZXJcbiAgICovXG4gIHB1YmxpYyBnZXQgZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlKCk6IGNsb3VkbWFwLklOYW1lc3BhY2UgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLl9kZWZhdWx0Q2xvdWRNYXBOYW1lc3BhY2U7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBtZXRob2QgYWRkcyBjb21wdXRlIGNhcGFjaXR5IHRvIGEgY2x1c3RlciBieSBjcmVhdGluZyBhbiBBdXRvU2NhbGluZ0dyb3VwIHdpdGggdGhlIHNwZWNpZmllZCBvcHRpb25zLlxuICAgKlxuICAgKiBSZXR1cm5zIHRoZSBBdXRvU2NhbGluZ0dyb3VwIHNvIHlvdSBjYW4gYWRkIGF1dG9zY2FsaW5nIHNldHRpbmdzIHRvIGl0LlxuICAgKi9cbiAgcHVibGljIGFkZENhcGFjaXR5KGlkOiBzdHJpbmcsIG9wdGlvbnM6IEFkZENhcGFjaXR5T3B0aW9ucyk6IGF1dG9zY2FsaW5nLkF1dG9TY2FsaW5nR3JvdXAge1xuICAgIGNvbnN0IGF1dG9TY2FsaW5nR3JvdXAgPSBuZXcgYXV0b3NjYWxpbmcuQXV0b1NjYWxpbmdHcm91cCh0aGlzLCBpZCwge1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICBtYWNoaW5lSW1hZ2U6IG9wdGlvbnMubWFjaGluZUltYWdlIHx8IG5ldyBFY3NPcHRpbWl6ZWRBbWkoKSxcbiAgICAgIHVwZGF0ZVR5cGU6IG9wdGlvbnMudXBkYXRlVHlwZSB8fCBhdXRvc2NhbGluZy5VcGRhdGVUeXBlLlJFUExBQ0lOR19VUERBVEUsXG4gICAgICBpbnN0YW5jZVR5cGU6IG9wdGlvbnMuaW5zdGFuY2VUeXBlLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hZGRBdXRvU2NhbGluZ0dyb3VwKGF1dG9TY2FsaW5nR3JvdXAsIG9wdGlvbnMpO1xuXG4gICAgcmV0dXJuIGF1dG9TY2FsaW5nR3JvdXA7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBtZXRob2QgYWRkcyBjb21wdXRlIGNhcGFjaXR5IHRvIGEgY2x1c3RlciB1c2luZyB0aGUgc3BlY2lmaWVkIEF1dG9TY2FsaW5nR3JvdXAuXG4gICAqXG4gICAqIEBwYXJhbSBhdXRvU2NhbGluZ0dyb3VwIHRoZSBBU0cgdG8gYWRkIHRvIHRoaXMgY2x1c3Rlci5cbiAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV0gaXMgbmVlZGVkIGluIG9yZGVyIHRvIGluc3RhbGwgdGhlIEVDU1xuICAgKiBhZ2VudCBieSB1cGRhdGluZyB0aGUgQVNHcyB1c2VyIGRhdGEuXG4gICAqL1xuICBwdWJsaWMgYWRkQXV0b1NjYWxpbmdHcm91cChhdXRvU2NhbGluZ0dyb3VwOiBhdXRvc2NhbGluZy5BdXRvU2NhbGluZ0dyb3VwLCBvcHRpb25zOiBBZGRBdXRvU2NhbGluZ0dyb3VwQ2FwYWNpdHlPcHRpb25zID0ge30pIHtcbiAgICB0aGlzLl9oYXNFYzJDYXBhY2l0eSA9IHRydWU7XG4gICAgdGhpcy5jb25uZWN0aW9ucy5jb25uZWN0aW9ucy5hZGRTZWN1cml0eUdyb3VwKC4uLmF1dG9TY2FsaW5nR3JvdXAuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMpO1xuXG4gICAgLy8gVGllIGluc3RhbmNlcyB0byBjbHVzdGVyXG4gICAgYXV0b1NjYWxpbmdHcm91cC5hZGRVc2VyRGF0YShgZWNobyBFQ1NfQ0xVU1RFUj0ke3RoaXMuY2x1c3Rlck5hbWV9ID4+IC9ldGMvZWNzL2Vjcy5jb25maWdgKTtcblxuICAgIGlmICghb3B0aW9ucy5jYW5Db250YWluZXJzQWNjZXNzSW5zdGFuY2VSb2xlKSB7XG4gICAgICAvLyBEZW55IGNvbnRhaW5lcnMgYWNjZXNzIHRvIGluc3RhbmNlIG1ldGFkYXRhIHNlcnZpY2VcbiAgICAgIC8vIFNvdXJjZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvaW5zdGFuY2VfSUFNX3JvbGUuaHRtbFxuICAgICAgYXV0b1NjYWxpbmdHcm91cC5hZGRVc2VyRGF0YSgnc3VkbyBpcHRhYmxlcyAtLWluc2VydCBGT1JXQVJEIDEgLS1pbi1pbnRlcmZhY2UgZG9ja2VyKyAtLWRlc3RpbmF0aW9uIDE2OS4yNTQuMTY5LjI1NC8zMiAtLWp1bXAgRFJPUCcpO1xuICAgICAgYXV0b1NjYWxpbmdHcm91cC5hZGRVc2VyRGF0YSgnc3VkbyBzZXJ2aWNlIGlwdGFibGVzIHNhdmUnKTtcbiAgICAgIC8vIFRoZSBmb2xsb3dpbmcgaXMgb25seSBmb3IgQXdzVnBjIG5ldHdvcmtpbmcgbW9kZSwgYnV0IGRvZXNuJ3QgaHVydCBmb3IgdGhlIG90aGVyIG1vZGVzLlxuICAgICAgYXV0b1NjYWxpbmdHcm91cC5hZGRVc2VyRGF0YSgnZWNobyBFQ1NfQVdTVlBDX0JMT0NLX0lNRFM9dHJ1ZSA+PiAvZXRjL2Vjcy9lY3MuY29uZmlnJyk7XG4gICAgfVxuXG4gICAgaWYgKGF1dG9TY2FsaW5nR3JvdXAuc3BvdFByaWNlICYmIG9wdGlvbnMuc3BvdEluc3RhbmNlRHJhaW5pbmcpIHtcbiAgICAgIGF1dG9TY2FsaW5nR3JvdXAuYWRkVXNlckRhdGEoJ2VjaG8gRUNTX0VOQUJMRV9TUE9UX0lOU1RBTkNFX0RSQUlOSU5HPXRydWUgPj4gL2V0Yy9lY3MvZWNzLmNvbmZpZycpO1xuICAgIH1cblxuICAgIC8vIEVDUyBpbnN0YW5jZXMgbXVzdCBiZSBhYmxlIHRvIGRvIHRoZXNlIHRoaW5nc1xuICAgIC8vIFNvdXJjZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvaW5zdGFuY2VfSUFNX3JvbGUuaHRtbFxuICAgIGF1dG9TY2FsaW5nR3JvdXAuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgXCJlY3M6Q3JlYXRlQ2x1c3RlclwiLFxuICAgICAgICBcImVjczpEZXJlZ2lzdGVyQ29udGFpbmVySW5zdGFuY2VcIixcbiAgICAgICAgXCJlY3M6RGlzY292ZXJQb2xsRW5kcG9pbnRcIixcbiAgICAgICAgXCJlY3M6UG9sbFwiLFxuICAgICAgICBcImVjczpSZWdpc3RlckNvbnRhaW5lckluc3RhbmNlXCIsXG4gICAgICAgIFwiZWNzOlN0YXJ0VGVsZW1ldHJ5U2Vzc2lvblwiLFxuICAgICAgICBcImVjczpTdWJtaXQqXCIsXG4gICAgICAgIFwiZWNyOkdldEF1dGhvcml6YXRpb25Ub2tlblwiLFxuICAgICAgICBcImxvZ3M6Q3JlYXRlTG9nU3RyZWFtXCIsXG4gICAgICAgIFwibG9nczpQdXRMb2dFdmVudHNcIlxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogWycqJ11cbiAgICB9KSk7XG5cbiAgICAvLyAwIGRpc2FibGVzLCBvdGhlcndpc2UgZm9yd2FyZCB0byB1bmRlcmx5aW5nIGltcGxlbWVudGF0aW9uIHdoaWNoIHBpY2tzIHRoZSBzYW5lIGRlZmF1bHRcbiAgICBpZiAoIW9wdGlvbnMudGFza0RyYWluVGltZSB8fCBvcHRpb25zLnRhc2tEcmFpblRpbWUudG9TZWNvbmRzKCkgIT09IDApIHtcbiAgICAgIG5ldyBJbnN0YW5jZURyYWluSG9vayhhdXRvU2NhbGluZ0dyb3VwLCAnRHJhaW5FQ1NIb29rJywge1xuICAgICAgICBhdXRvU2NhbGluZ0dyb3VwLFxuICAgICAgICBjbHVzdGVyOiB0aGlzLFxuICAgICAgICBkcmFpblRpbWU6IG9wdGlvbnMudGFza0RyYWluVGltZVxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEdldHRlciBmb3IgYXV0b3NjYWxpbmcgZ3JvdXAgYWRkZWQgdG8gY2x1c3RlclxuICAgKi9cbiAgcHVibGljIGdldCBhdXRvc2NhbGluZ0dyb3VwKCk6IGF1dG9zY2FsaW5nLklBdXRvU2NhbGluZ0dyb3VwIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fYXV0b3NjYWxpbmdHcm91cDtcbiAgfVxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBjbHVzdGVyIGhhcyBFQzIgY2FwYWNpdHkgYXNzb2NpYXRlZCB3aXRoIGl0XG4gICAqL1xuICBwdWJsaWMgZ2V0IGhhc0VjMkNhcGFjaXR5KCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLl9oYXNFYzJDYXBhY2l0eTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGlzIG1ldGhvZCByZXR1cm5zIHRoZSBDbG91ZFdhdGNoIG1ldHJpYyBmb3IgdGhpcyBjbHVzdGVycyBDUFUgcmVzZXJ2YXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNDcHVSZXNlcnZhdGlvbihwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0NQVVJlc2VydmF0aW9uJywgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoaXMgbWV0aG9kIHJldHVybnMgdGhlIENsb3VkV2F0Y2ggbWV0cmljIGZvciB0aGlzIGNsdXN0ZXJzIG1lbW9yeSByZXNlcnZhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgYXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY01lbW9yeVJlc2VydmF0aW9uKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnTWVtb3J5UmVzZXJ2YXRpb24nLCBwcm9wcyk7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBtZXRob2QgcmV0dXJucyB0aGUgc3BlY2lmZWQgQ2xvdWRXYXRjaCBtZXRyaWMgZm9yIHRoaXMgY2x1c3Rlci5cbiAgICovXG4gIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9FQ1MnLFxuICAgICAgbWV0cmljTmFtZSxcbiAgICAgIGRpbWVuc2lvbnM6IHsgQ2x1c3Rlck5hbWU6IHRoaXMuY2x1c3Rlck5hbWUgfSxcbiAgICAgIC4uLnByb3BzXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBFQ1Mtb3B0aW1pemVkIFdpbmRvd3MgdmVyc2lvbiBsaXN0XG4gKi9cbmV4cG9ydCBlbnVtIFdpbmRvd3NPcHRpbWl6ZWRWZXJzaW9uIHtcbiAgU0VSVkVSXzIwMTkgPSAnMjAxOScsXG4gIFNFUlZFUl8yMDE2ID0gJzIwMTYnLFxufVxuXG4vKlxuICogVE9ETzp2Mi4wLjBcbiAqICAqIHJlbW92ZSBgZXhwb3J0YCBrZXl3b3JkXG4gKiAgKiByZW1vdmUgQGRlcHJhY3RlZFxuICovXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIHRoYXQgZGVmaW5lIHdoaWNoIEVDUy1vcHRpbWl6ZWQgQU1JIGlzIHVzZWQuXG4gKlxuICogQGRlcHJlY2F0ZWQgc2VlIHtAbGluayBFY3NPcHRpbWl6ZWRJbWFnZX1cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFY3NPcHRpbWl6ZWRBbWlQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIExpbnV4IGdlbmVyYXRpb24gdG8gdXNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBBbWF6b25MaW51eEdlbmVyYXRpb24uQW1hem9uTGludXgyXG4gICAqL1xuICByZWFkb25seSBnZW5lcmF0aW9uPzogZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIFdpbmRvd3MgU2VydmVyIHZlcnNpb24gdG8gdXNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBub25lLCB1c2VzIExpbnV4IGdlbmVyYXRpb25cbiAgICovXG4gIHJlYWRvbmx5IHdpbmRvd3NWZXJzaW9uPzogV2luZG93c09wdGltaXplZFZlcnNpb247XG5cbiAgLyoqXG4gICAqIFRoZSBFQ1Mtb3B0aW1pemVkIEFNSSB2YXJpYW50IHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgQW1pSGFyZHdhcmVUeXBlLlN0YW5kYXJkXG4gICAqL1xuICByZWFkb25seSBoYXJkd2FyZVR5cGU/OiBBbWlIYXJkd2FyZVR5cGU7XG59XG5cbi8qXG4gKiBUT0RPOnYyLjAuMCByZW1vdmUgRWNzT3B0aW1pemVkQW1pXG4gKi9cbi8qKlxuICogQ29uc3RydWN0IGEgTGludXggb3IgV2luZG93cyBtYWNoaW5lIGltYWdlIGZyb20gdGhlIGxhdGVzdCBFQ1MgT3B0aW1pemVkIEFNSSBwdWJsaXNoZWQgaW4gU1NNXG4gKlxuICogQGRlcHJlY2F0ZWQgc2VlIHtAbGluayBFY3NPcHRpbWl6ZWRJbWFnZSNhbWF6b25MaW51eH0sIHtAbGluayBFY3NPcHRpbWl6ZWRJbWFnZSNhbWF6b25MaW51eH0gYW5kIHtAbGluayBFY3NPcHRpbWl6ZWRJbWFnZSN3aW5kb3dzfVxuICovXG5leHBvcnQgY2xhc3MgRWNzT3B0aW1pemVkQW1pIGltcGxlbWVudHMgZWMyLklNYWNoaW5lSW1hZ2Uge1xuICBwcml2YXRlIHJlYWRvbmx5IGdlbmVyYXRpb24/OiBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uO1xuICBwcml2YXRlIHJlYWRvbmx5IHdpbmRvd3NWZXJzaW9uPzogV2luZG93c09wdGltaXplZFZlcnNpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgaHdUeXBlOiBBbWlIYXJkd2FyZVR5cGU7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBhbWlQYXJhbWV0ZXJOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIEVjc09wdGltaXplZEFtaSBjbGFzcy5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogRWNzT3B0aW1pemVkQW1pUHJvcHMpIHtcbiAgICB0aGlzLmh3VHlwZSA9IChwcm9wcyAmJiBwcm9wcy5oYXJkd2FyZVR5cGUpIHx8IEFtaUhhcmR3YXJlVHlwZS5TVEFOREFSRDtcbiAgICBpZiAocHJvcHMgJiYgcHJvcHMuZ2VuZXJhdGlvbikgeyAgICAgIC8vIGdlbmVyYXRpb24gZGVmaW5lZCBpbiB0aGUgcHJvcHMgb2JqZWN0XG4gICAgICBpZiAocHJvcHMuZ2VuZXJhdGlvbiA9PT0gZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVggJiYgdGhpcy5od1R5cGUgIT09IEFtaUhhcmR3YXJlVHlwZS5TVEFOREFSRCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEFtYXpvbiBMaW51eCBkb2VzIG5vdCBzdXBwb3J0IHNwZWNpYWwgaGFyZHdhcmUgdHlwZS4gVXNlIEFtYXpvbiBMaW51eCAyIGluc3RlYWRgKTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMud2luZG93c1ZlcnNpb24pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdcIndpbmRvd3NWZXJzaW9uXCIgYW5kIExpbnV4IGltYWdlIFwiZ2VuZXJhdGlvblwiIGNhbm5vdCBiZSBib3RoIHNldCcpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy5nZW5lcmF0aW9uID0gcHJvcHMuZ2VuZXJhdGlvbjtcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKHByb3BzICYmIHByb3BzLndpbmRvd3NWZXJzaW9uKSB7XG4gICAgICBpZiAodGhpcy5od1R5cGUgIT09IEFtaUhhcmR3YXJlVHlwZS5TVEFOREFSRCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1dpbmRvd3MgU2VydmVyIGRvZXMgbm90IHN1cHBvcnQgc3BlY2lhbCBoYXJkd2FyZSB0eXBlJyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aGlzLndpbmRvd3NWZXJzaW9uID0gcHJvcHMud2luZG93c1ZlcnNpb247XG4gICAgICB9XG4gICAgfSBlbHNlIHsgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAvLyBnZW5lcmF0aW9uIG5vdCBkZWZpbmVkIGluIHByb3BzIG9iamVjdFxuICAgICAgLy8gYWx3YXlzIGRlZmF1bHQgdG8gQW1hem9uIExpbnV4IHYyIHJlZ2FyZGxlc3Mgb2YgSFdcbiAgICAgIHRoaXMuZ2VuZXJhdGlvbiA9IGVjMi5BbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzI7XG4gICAgfVxuXG4gICAgLy8gc2V0IHRoZSBTU00gcGFyYW1ldGVyIG5hbWVcbiAgICB0aGlzLmFtaVBhcmFtZXRlck5hbWUgPSBcIi9hd3Mvc2VydmljZS9lY3Mvb3B0aW1pemVkLWFtaS9cIlxuICAgICAgKyAodGhpcy5nZW5lcmF0aW9uID09PSBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWCA/IFwiYW1hem9uLWxpbnV4L1wiIDogXCJcIilcbiAgICAgICsgKHRoaXMuZ2VuZXJhdGlvbiA9PT0gZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiA/IFwiYW1hem9uLWxpbnV4LTIvXCIgOiBcIlwiKVxuICAgICAgKyAodGhpcy53aW5kb3dzVmVyc2lvbiA/IGB3aW5kb3dzX3NlcnZlci8ke3RoaXMud2luZG93c1ZlcnNpb259L2VuZ2xpc2gvZnVsbC9gIDogXCJcIilcbiAgICAgICsgKHRoaXMuaHdUeXBlID09PSBBbWlIYXJkd2FyZVR5cGUuR1BVID8gXCJncHUvXCIgOiBcIlwiKVxuICAgICAgKyAodGhpcy5od1R5cGUgPT09IEFtaUhhcmR3YXJlVHlwZS5BUk0gPyBcImFybTY0L1wiIDogXCJcIilcbiAgICAgICsgXCJyZWNvbW1lbmRlZC9pbWFnZV9pZFwiO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgY29ycmVjdCBpbWFnZVxuICAgKi9cbiAgcHVibGljIGdldEltYWdlKHNjb3BlOiBDb25zdHJ1Y3QpOiBlYzIuTWFjaGluZUltYWdlQ29uZmlnIHtcbiAgICBjb25zdCBhbWkgPSBzc20uU3RyaW5nUGFyYW1ldGVyLnZhbHVlRm9yVHlwZWRTdHJpbmdQYXJhbWV0ZXIoc2NvcGUsIHRoaXMuYW1pUGFyYW1ldGVyTmFtZSwgc3NtLlBhcmFtZXRlclR5cGUuQVdTX0VDMl9JTUFHRV9JRCk7XG4gICAgcmV0dXJuIHtcbiAgICAgIGltYWdlSWQ6IGFtaSxcbiAgICAgIG9zVHlwZTogdGhpcy53aW5kb3dzVmVyc2lvbiA/IGVjMi5PcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1MgOiBlYzIuT3BlcmF0aW5nU3lzdGVtVHlwZS5MSU5VWFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3QgYSBMaW51eCBvciBXaW5kb3dzIG1hY2hpbmUgaW1hZ2UgZnJvbSB0aGUgbGF0ZXN0IEVDUyBPcHRpbWl6ZWQgQU1JIHB1Ymxpc2hlZCBpbiBTU01cbiAqL1xuZXhwb3J0IGNsYXNzIEVjc09wdGltaXplZEltYWdlIGltcGxlbWVudHMgZWMyLklNYWNoaW5lSW1hZ2Uge1xuICAvKipcbiAgICogQ29uc3RydWN0IGFuIEFtYXpvbiBMaW51eCAyIGltYWdlIGZyb20gdGhlIGxhdGVzdCBFQ1MgT3B0aW1pemVkIEFNSSBwdWJsaXNoZWQgaW4gU1NNXG4gICAqXG4gICAqIEBwYXJhbSBoYXJkd2FyZVR5cGUgRUNTLW9wdGltaXplZCBBTUkgdmFyaWFudCB0byB1c2VcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYW1hem9uTGludXgyKGhhcmR3YXJlVHlwZSA9IEFtaUhhcmR3YXJlVHlwZS5TVEFOREFSRCk6IEVjc09wdGltaXplZEltYWdlIHtcbiAgICByZXR1cm4gbmV3IEVjc09wdGltaXplZEltYWdlKHsgZ2VuZXJhdGlvbjogZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiwgaGFyZHdhcmVUeXBlIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdCBhbiBBbWF6b24gTGludXggQU1JIGltYWdlIGZyb20gdGhlIGxhdGVzdCBFQ1MgT3B0aW1pemVkIEFNSSBwdWJsaXNoZWQgaW4gU1NNXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFtYXpvbkxpbnV4KCk6IEVjc09wdGltaXplZEltYWdlIHtcbiAgICByZXR1cm4gbmV3IEVjc09wdGltaXplZEltYWdlKHsgZ2VuZXJhdGlvbjogZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVggfSk7XG4gIH1cblxuICAvKipcbiAgICogQ29uc3RydWN0IGEgV2luZG93cyBpbWFnZSBmcm9tIHRoZSBsYXRlc3QgRUNTIE9wdGltaXplZCBBTUkgcHVibGlzaGVkIGluIFNTTVxuICAgKlxuICAgKiBAcGFyYW0gd2luZG93c1ZlcnNpb24gV2luZG93cyBWZXJzaW9uIHRvIHVzZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB3aW5kb3dzKHdpbmRvd3NWZXJzaW9uOiBXaW5kb3dzT3B0aW1pemVkVmVyc2lvbik6IEVjc09wdGltaXplZEltYWdlIHtcbiAgICByZXR1cm4gbmV3IEVjc09wdGltaXplZEltYWdlKHsgd2luZG93c1ZlcnNpb24gfSk7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IGdlbmVyYXRpb24/OiBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uO1xuICBwcml2YXRlIHJlYWRvbmx5IHdpbmRvd3NWZXJzaW9uPzogV2luZG93c09wdGltaXplZFZlcnNpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgaHdUeXBlPzogQW1pSGFyZHdhcmVUeXBlO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgYW1pUGFyYW1ldGVyTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBFY3NPcHRpbWl6ZWRBbWkgY2xhc3MuXG4gICAqL1xuICBwcml2YXRlIGNvbnN0cnVjdG9yKHByb3BzOiBFY3NPcHRpbWl6ZWRBbWlQcm9wcykge1xuICAgIHRoaXMuaHdUeXBlID0gcHJvcHMgJiYgcHJvcHMuaGFyZHdhcmVUeXBlO1xuXG4gICAgaWYgKHByb3BzLndpbmRvd3NWZXJzaW9uKSB7XG4gICAgICB0aGlzLndpbmRvd3NWZXJzaW9uID0gcHJvcHMud2luZG93c1ZlcnNpb247XG4gICAgfSBlbHNlIGlmIChwcm9wcy5nZW5lcmF0aW9uKSB7XG4gICAgICB0aGlzLmdlbmVyYXRpb24gPSBwcm9wcy5nZW5lcmF0aW9uO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoaXMgZXJyb3Igc2hvdWxkIG5ldmVyIGJlIHRocm93bicpO1xuICAgIH1cblxuICAgIC8vIHNldCB0aGUgU1NNIHBhcmFtZXRlciBuYW1lXG4gICAgdGhpcy5hbWlQYXJhbWV0ZXJOYW1lID0gXCIvYXdzL3NlcnZpY2UvZWNzL29wdGltaXplZC1hbWkvXCJcbiAgICAgICsgKHRoaXMuZ2VuZXJhdGlvbiA9PT0gZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVggPyBcImFtYXpvbi1saW51eC9cIiA6IFwiXCIpXG4gICAgICArICh0aGlzLmdlbmVyYXRpb24gPT09IGVjMi5BbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgPyBcImFtYXpvbi1saW51eC0yL1wiIDogXCJcIilcbiAgICAgICsgKHRoaXMud2luZG93c1ZlcnNpb24gPyBgd2luZG93c19zZXJ2ZXIvJHt0aGlzLndpbmRvd3NWZXJzaW9ufS9lbmdsaXNoL2Z1bGwvYCA6IFwiXCIpXG4gICAgICArICh0aGlzLmh3VHlwZSA9PT0gQW1pSGFyZHdhcmVUeXBlLkdQVSA/IFwiZ3B1L1wiIDogXCJcIilcbiAgICAgICsgKHRoaXMuaHdUeXBlID09PSBBbWlIYXJkd2FyZVR5cGUuQVJNID8gXCJhcm02NC9cIiA6IFwiXCIpXG4gICAgICArIFwicmVjb21tZW5kZWQvaW1hZ2VfaWRcIjtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIGNvcnJlY3QgaW1hZ2VcbiAgICovXG4gIHB1YmxpYyBnZXRJbWFnZShzY29wZTogQ29uc3RydWN0KTogZWMyLk1hY2hpbmVJbWFnZUNvbmZpZyB7XG4gICAgY29uc3QgYW1pID0gc3NtLlN0cmluZ1BhcmFtZXRlci52YWx1ZUZvclR5cGVkU3RyaW5nUGFyYW1ldGVyKHNjb3BlLCB0aGlzLmFtaVBhcmFtZXRlck5hbWUsIHNzbS5QYXJhbWV0ZXJUeXBlLkFXU19FQzJfSU1BR0VfSUQpO1xuICAgIHJldHVybiB7XG4gICAgICBpbWFnZUlkOiBhbWksXG4gICAgICBvc1R5cGU6IHRoaXMud2luZG93c1ZlcnNpb24gPyBlYzIuT3BlcmF0aW5nU3lzdGVtVHlwZS5XSU5ET1dTIDogZWMyLk9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVhcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogQSByZWdpb25hbCBncm91cGluZyBvZiBvbmUgb3IgbW9yZSBjb250YWluZXIgaW5zdGFuY2VzIG9uIHdoaWNoIHlvdSBjYW4gcnVuIHRhc2tzIGFuZCBzZXJ2aWNlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ2x1c3RlciBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY2x1c3Rlci5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3Rlck5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIHRoYXQgaWRlbnRpZmllcyB0aGUgY2x1c3Rlci5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlckFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIGFzc29jaWF0ZWQgd2l0aCB0aGUgY2x1c3Rlci5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIE1hbmFnZSB0aGUgYWxsb3dlZCBuZXR3b3JrIGNvbm5lY3Rpb25zIGZvciB0aGUgY2x1c3RlciB3aXRoIFNlY3VyaXR5IEdyb3Vwcy5cbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRoZSBjbHVzdGVyIGhhcyBFQzIgaW5zdGFuY2UgY2FwYWNpdHkuXG4gICAqL1xuICByZWFkb25seSBoYXNFYzJDYXBhY2l0eTogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIEFXUyBDbG91ZCBNYXAgbmFtZXNwYWNlIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBjbHVzdGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlPzogY2xvdWRtYXAuSU5hbWVzcGFjZTtcblxuICAvKipcbiAgICogVGhlIGF1dG9zY2FsaW5nIGdyb3VwIGFkZGVkIHRvIHRoZSBjbHVzdGVyIGlmIGNhcGFjaXR5IGlzIGFzc29jaWF0ZWQgdG8gdGhlIGNsdXN0ZXJcbiAgICovXG4gIHJlYWRvbmx5IGF1dG9zY2FsaW5nR3JvdXA/OiBhdXRvc2NhbGluZy5JQXV0b1NjYWxpbmdHcm91cDtcbn1cblxuLyoqXG4gKiBUaGUgcHJvcGVydGllcyB0byBpbXBvcnQgZnJvbSB0aGUgRUNTIGNsdXN0ZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2x1c3RlckF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGNsdXN0ZXIuXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgdGhhdCBpZGVudGlmaWVzIHRoZSBjbHVzdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEZXJpdmVkIGZyb20gY2x1c3Rlck5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXJBcm4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgYXNzb2NpYXRlZCB3aXRoIHRoZSBjbHVzdGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGdyb3VwcyBhc3NvY2lhdGVkIHdpdGggdGhlIGNvbnRhaW5lciBpbnN0YW5jZXMgcmVnaXN0ZXJlZCB0byB0aGUgY2x1c3Rlci5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzOiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdGhlIGNsdXN0ZXIgaGFzIEVDMiBpbnN0YW5jZSBjYXBhY2l0eS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgaGFzRWMyQ2FwYWNpdHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgQVdTIENsb3VkIE1hcCBuYW1lc3BhY2UgdG8gYXNzb2NpYXRlIHdpdGggdGhlIGNsdXN0ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZGVmYXVsdCBuYW1lc3BhY2VcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRDbG91ZE1hcE5hbWVzcGFjZT86IGNsb3VkbWFwLklOYW1lc3BhY2U7XG5cbiAgLyoqXG4gICAqIEF1dG9zY2FsaW5nIGdyb3VwIGFkZGVkIHRvIHRoZSBjbHVzdGVyIGlmIGNhcGFjaXR5IGlzIGFkZGVkXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZGVmYXVsdCBhdXRvc2NhbGluZyBncm91cFxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b3NjYWxpbmdHcm91cD86IGF1dG9zY2FsaW5nLklBdXRvU2NhbGluZ0dyb3VwO1xufVxuXG4vKipcbiAqIEFuIENsdXN0ZXIgdGhhdCBoYXMgYmVlbiBpbXBvcnRlZFxuICovXG5jbGFzcyBJbXBvcnRlZENsdXN0ZXIgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDbHVzdGVyIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGNsdXN0ZXJcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjbHVzdGVyTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBUk4gb2YgdGhlIGNsdXN0ZXJcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjbHVzdGVyQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFZQQyB0aGF0IHRoZSBjbHVzdGVyIGluc3RhbmNlcyBhcmUgcnVubmluZyBpblxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFNlY3VyaXR5IGdyb3VwIG9mIHRoZSBjbHVzdGVyIGluc3RhbmNlc1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucygpO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBjbHVzdGVyIGhhcyBFQzIgY2FwYWNpdHlcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBoYXNFYzJDYXBhY2l0eTogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ2xvdWRtYXAgbmFtZXNwYWNlIGNyZWF0ZWQgaW4gdGhlIGNsdXN0ZXJcbiAgICovXG4gIHByaXZhdGUgX2RlZmF1bHRDbG91ZE1hcE5hbWVzcGFjZT86IGNsb3VkbWFwLklOYW1lc3BhY2U7XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIEltcG9ydGVkQ2x1c3RlciBjbGFzcy5cbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDbHVzdGVyQXR0cmlidXRlcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgdGhpcy5jbHVzdGVyTmFtZSA9IHByb3BzLmNsdXN0ZXJOYW1lO1xuICAgIHRoaXMudnBjID0gcHJvcHMudnBjO1xuICAgIHRoaXMuaGFzRWMyQ2FwYWNpdHkgPSBwcm9wcy5oYXNFYzJDYXBhY2l0eSAhPT0gZmFsc2U7XG4gICAgdGhpcy5fZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlID0gcHJvcHMuZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlO1xuXG4gICAgdGhpcy5jbHVzdGVyQXJuID0gcHJvcHMuY2x1c3RlckFybiAhPT0gdW5kZWZpbmVkID8gcHJvcHMuY2x1c3RlckFybiA6IFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnZWNzJyxcbiAgICAgIHJlc291cmNlOiAnY2x1c3RlcicsXG4gICAgICByZXNvdXJjZU5hbWU6IHByb3BzLmNsdXN0ZXJOYW1lXG4gICAgfSk7XG5cbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucyh7XG4gICAgICBzZWN1cml0eUdyb3VwczogcHJvcHMuc2VjdXJpdHlHcm91cHNcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgZGVmYXVsdENsb3VkTWFwTmFtZXNwYWNlKCk6IGNsb3VkbWFwLklOYW1lc3BhY2UgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLl9kZWZhdWx0Q2xvdWRNYXBOYW1lc3BhY2U7XG4gIH1cbn1cblxuLyoqXG4gKiBUaGUgcHJvcGVydGllcyBmb3IgYWRkaW5nIGFuIEF1dG9TY2FsaW5nR3JvdXAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkQXV0b1NjYWxpbmdHcm91cENhcGFjaXR5T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgd2hldGhlciB0aGUgY29udGFpbmVycyBjYW4gYWNjZXNzIHRoZSBjb250YWluZXIgaW5zdGFuY2Ugcm9sZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNhbkNvbnRhaW5lcnNBY2Nlc3NJbnN0YW5jZVJvbGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgdGltZSBwZXJpb2QgdG8gd2FpdCBiZWZvcmUgZm9yY2UgdGVybWluYXRpbmcgYW4gaW5zdGFuY2UgdGhhdCBpcyBkcmFpbmluZy5cbiAgICpcbiAgICogVGhpcyBjcmVhdGVzIGEgTGFtYmRhIGZ1bmN0aW9uIHRoYXQgaXMgdXNlZCBieSBhIGxpZmVjeWNsZSBob29rIGZvciB0aGVcbiAgICogQXV0b1NjYWxpbmdHcm91cCB0aGF0IHdpbGwgZGVsYXkgaW5zdGFuY2UgdGVybWluYXRpb24gdW50aWwgYWxsIEVDUyB0YXNrc1xuICAgKiBoYXZlIGRyYWluZWQgZnJvbSB0aGUgaW5zdGFuY2UuIFNldCB0byAwIHRvIGRpc2FibGUgdGFzayBkcmFpbmluZy5cbiAgICpcbiAgICogU2V0IHRvIDAgdG8gZGlzYWJsZSB0YXNrIGRyYWluaW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5taW51dGVzKDUpXG4gICAqL1xuICByZWFkb25seSB0YXNrRHJhaW5UaW1lPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgd2hldGhlciB0byBlbmFibGUgQXV0b21hdGVkIERyYWluaW5nIGZvciBTcG90IEluc3RhbmNlcyBydW5uaW5nIEFtYXpvbiBFQ1MgU2VydmljZXMuXG4gICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgW1VzaW5nIFNwb3QgSW5zdGFuY2VzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb250YWluZXItaW5zdGFuY2Utc3BvdC5odG1sKS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHNwb3RJbnN0YW5jZURyYWluaW5nPzogYm9vbGVhblxufVxuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciBhZGRpbmcgaW5zdGFuY2UgY2FwYWNpdHkgdG8gYW4gQXV0b1NjYWxpbmdHcm91cC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBZGRDYXBhY2l0eU9wdGlvbnMgZXh0ZW5kcyBBZGRBdXRvU2NhbGluZ0dyb3VwQ2FwYWNpdHlPcHRpb25zLCBhdXRvc2NhbGluZy5Db21tb25BdXRvU2NhbGluZ0dyb3VwUHJvcHMge1xuICAvKipcbiAgICogVGhlIEVDMiBpbnN0YW5jZSB0eXBlIHRvIHVzZSB3aGVuIGxhdW5jaGluZyBpbnN0YW5jZXMgaW50byB0aGUgQXV0b1NjYWxpbmdHcm91cC5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogVGhlIEVDUy1vcHRpbWl6ZWQgQU1JIHZhcmlhbnQgdG8gdXNlLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAqIFtBbWF6b24gRUNTLW9wdGltaXplZCBBTUlzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9lY3Mtb3B0aW1pemVkX0FNSS5odG1sKS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbWF6b24gTGludXggMlxuICAgKi9cbiAgcmVhZG9ubHkgbWFjaGluZUltYWdlPzogZWMyLklNYWNoaW5lSW1hZ2U7XG59XG5cbi8qKlxuICogVGhlIG9wdGlvbnMgZm9yIGNyZWF0aW5nIGFuIEFXUyBDbG91ZCBNYXAgbmFtZXNwYWNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkTWFwTmFtZXNwYWNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgbmFtZXNwYWNlLCBzdWNoIGFzIGV4YW1wbGUuY29tLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBDbG91ZE1hcCBOYW1lc3BhY2UgdG8gY3JlYXRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBQcml2YXRlRG5zXG4gICAqL1xuICByZWFkb25seSB0eXBlPzogY2xvdWRtYXAuTmFtZXNwYWNlVHlwZTtcblxuICAvKipcbiAgICogVGhlIFZQQyB0byBhc3NvY2lhdGUgdGhlIG5hbWVzcGFjZSB3aXRoLiBUaGlzIHByb3BlcnR5IGlzIHJlcXVpcmVkIGZvciBwcml2YXRlIEROUyBuYW1lc3BhY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBWUEMgb2YgdGhlIGNsdXN0ZXIgZm9yIFByaXZhdGUgRE5TIE5hbWVzcGFjZSwgb3RoZXJ3aXNlIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xufVxuXG4vKipcbiAqIFRoZSBFQ1Mtb3B0aW1pemVkIEFNSSB2YXJpYW50IHRvIHVzZS4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZVxuICogW0FtYXpvbiBFQ1Mtb3B0aW1pemVkIEFNSXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2Vjcy1vcHRpbWl6ZWRfQU1JLmh0bWwpLlxuICovXG5leHBvcnQgZW51bSBBbWlIYXJkd2FyZVR5cGUge1xuXG4gIC8qKlxuICAgKiBVc2UgdGhlIHN0YW5kYXJkIEFtYXpvbiBFQ1Mtb3B0aW1pemVkIEFNSS5cbiAgICovXG4gIFNUQU5EQVJEID0gJ1N0YW5kYXJkJyxcblxuICAvKipcbiAgICogVXNlIHRoZSBBbWF6b24gRUNTIEdQVS1vcHRpbWl6ZWQgQU1JLlxuICAgKi9cbiAgR1BVID0gJ0dQVScsXG5cbiAgLyoqXG4gICAqIFVzZSB0aGUgQW1hem9uIEVDUy1vcHRpbWl6ZWQgQW1hem9uIExpbnV4IDIgKGFybTY0KSBBTUkuXG4gICAqL1xuICBBUk0gPSAnQVJNNjQnLFxufVxuIl19