"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const task_definition_1 = require("./base/task-definition");
/**
 * A secret environment variable.
 */
class Secret {
    /**
     * Creates an environment variable value from a parameter stored in AWS
     * Systems Manager Parameter Store.
     */
    static fromSsmParameter(parameter) {
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     */
    static fromSecretsManager(secret) {
        return {
            arn: secret.secretArn,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
exports.Secret = Secret;
/**
 * A container definition is used in a task definition to describe the containers that are launched as part of a task.
 */
class ContainerDefinition extends cdk.Construct {
    /**
     * Constructs a new instance of the ContainerDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * The mount points for data volumes in your container.
         */
        this.mountPoints = new Array();
        /**
         * The list of port mappings for the container. Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         */
        this.portMappings = new Array();
        /**
         * The data volumes to mount from another container in the same task definition.
         */
        this.volumesFrom = new Array();
        /**
         * An array of ulimits to set in the container.
         */
        this.ulimits = new Array();
        /**
         * An array dependencies defined for container startup and shutdown.
         */
        this.containerDependencies = new Array();
        /**
         * The configured container links
         */
        this.links = new Array();
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error(`MemoryLimitMiB should not be less than MemoryReservationMiB.`);
            }
        }
        this.essential = props.essential !== undefined ? props.essential : true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = this.node.id;
        this.imageConfig = props.image.bind(this, this);
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        props.taskDefinition._linkContainer(this);
    }
    /**
     * This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     */
    addLink(container, alias) {
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error(`You must use network mode Bridge to add container links.`);
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * This method adds one or more mount points for data volumes to the container.
     */
    addMountPoints(...mountPoints) {
        this.mountPoints.push(...mountPoints);
    }
    /**
     * This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     */
    addScratch(scratch) {
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath
            },
            name: scratch.name
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * This method adds one or more port mappings to the container.
     */
    addPortMappings(...portMappings) {
        this.portMappings.push(...portMappings.map(pm => {
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC || this.taskDefinition.networkMode === task_definition_1.NetworkMode.HOST) {
                if (pm.containerPort !== pm.hostPort && pm.hostPort !== undefined) {
                    throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.taskDefinition.networkMode}`);
                }
            }
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
                if (pm.hostPort === undefined) {
                    pm = Object.assign(Object.assign({}, pm), { hostPort: 0 });
                }
            }
            return pm;
        }));
    }
    /**
     * This method adds one or more ulimits to the container.
     */
    addUlimits(...ulimits) {
        this.ulimits.push(...ulimits);
    }
    /**
     * This method adds one or more container dependencies to the container.
     */
    addContainerDependencies(...containerDependencies) {
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * This method adds one or more volumes to the container.
     */
    addVolumesFrom(...volumesFrom) {
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * This method adds the specified statement to the IAM task execution policy in the task definition.
     */
    addToExecutionPolicy(statement) {
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * Returns the host port for the requested container port if it exists
     *
     * @internal
     */
    _findPortMapping(containerPort, protocol) {
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * The port the container will listen on.
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * Render this container definition to a CloudFormation object
     *
     * @param taskDefinition [disable-awslint:ref-via-interface] (made optional to avoid breaking change)
     */
    renderContainerDefinition(taskDefinition) {
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.anyValue({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.anyValue({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.anyValue({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.anyValue({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.anyValue({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.props.environment && renderKV(this.props.environment, 'name', 'value'),
            secrets: this.props.secrets && Object.entries(this.props.secrets)
                .map(([k, v]) => {
                if (taskDefinition) {
                    v.grantRead(taskDefinition.obtainExecutionRole());
                }
                return {
                    name: k,
                    valueFrom: v.arn
                };
            }),
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.listValue({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (this.props.gpuCount !== undefined) ? renderResourceRequirements(this.props.gpuCount) : undefined,
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
function renderKV(env, keyName, valueName) {
    const ret = new Array();
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderHealthCheck(hc) {
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval != null ? hc.interval.toSeconds() : 30,
        retries: hc.retries !== undefined ? hc.retries : 3,
        startPeriod: hc.startPeriod && hc.startPeriod.toSeconds(),
        timeout: hc.timeout !== undefined ? hc.timeout.toSeconds() : 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error(`At least one argument must be supplied for health check command.`);
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== "CMD" && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount) {
    if (gpuCount === 0) {
        return undefined;
    }
    return [{
            type: 'GPU',
            value: gpuCount.toString(),
        }];
}
/**
 * Type of resource to set a limit on
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    /**
     * This condition emulates the behavior of links and volumes today.
     * It validates that a dependent container is started before permitting other containers to start.
     */
    ContainerDependencyCondition["START"] = "START";
    /**
     * This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.
     * This can be useful for nonessential containers that run a script and then exit.
     */
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    /**
     * This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.
     */
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    /**
     * This condition validates that the dependent container passes its Docker health check before permitting other containers to start.
     * This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
     */
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY
    };
}
/**
 * Network protocol
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP
     */
    Protocol["TCP"] = "tcp";
    /**
     * UDP
     */
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
//# sourceMappingURL=data:application/json;base64,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