import ec2 = require('@aws-cdk/aws-ec2');
import { Construct } from '@aws-cdk/core';
import { BaseService, BaseServiceOptions, IService, PropagatedTagSource } from '../base/base-service';
import { TaskDefinition } from '../base/task-definition';
import { PlacementConstraint, PlacementStrategy } from '../placement';
/**
 * The properties for defining a service using the EC2 launch type.
 */
export interface Ec2ServiceProps extends BaseServiceOptions {
    /**
     * The task definition to use for tasks in the service.
     *
     * [disable-awslint:ref-via-interface]
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * Specifies whether the task's elastic network interface receives a public IP address.
     * If true, each task will receive a public IP address.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default - Use subnet default.
     */
    readonly assignPublicIp?: boolean;
    /**
     * The subnets to associate with the service.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default - Private subnets.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The security groups to associate with the service. If you do not specify a security group, the default security group for the VPC is used.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @default - A new security group is created.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The placement constraints to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     *
     * @default - No constraints.
     */
    readonly placementConstraints?: PlacementConstraint[];
    /**
     * The placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     *
     * @default - No strategies.
     */
    readonly placementStrategies?: PlacementStrategy[];
    /**
     * Specifies whether the service will use the daemon scheduling strategy.
     * If true, the service scheduler deploys exactly one task on each container instance in your cluster.
     *
     * When you are using this strategy, do not specify a desired number of tasks orany task placement strategies.
     *
     * @default false
     */
    readonly daemon?: boolean;
    /**
     * Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.
     * Tags can only be propagated to the tasks within the service during service creation.
     *
     * @deprecated Use `propagateTags` instead.
     * @default PropagatedTagSource.NONE
     */
    readonly propagateTaskTagsFrom?: PropagatedTagSource;
}
/**
 * The interface for a service using the EC2 launch type on an ECS cluster.
 */
export interface IEc2Service extends IService {
}
/**
 * This creates a service using the EC2 launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
export declare class Ec2Service extends BaseService implements IEc2Service {
    /**
     * Imports from the specified service ARN.
     */
    static fromEc2ServiceArn(scope: Construct, id: string, ec2ServiceArn: string): IEc2Service;
    private readonly constraints;
    private readonly strategies;
    private readonly daemon;
    /**
     * Constructs a new instance of the Ec2Service class.
     */
    constructor(scope: Construct, id: string, props: Ec2ServiceProps);
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     */
    addPlacementStrategies(...strategies: PlacementStrategy[]): void;
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     */
    addPlacementConstraints(...constraints: PlacementConstraint[]): void;
    /**
     * Validates this Ec2Service.
     */
    protected validate(): string[];
}
/**
 * The built-in container instance attributes
 */
export declare class BuiltInAttributes {
    /**
     * The id of the instance.
     */
    static readonly INSTANCE_ID = "instanceId";
    /**
     * The AvailabilityZone where the instance is running in.
     */
    static readonly AVAILABILITY_ZONE = "attribute:ecs.availability-zone";
    /**
     * The AMI id the instance is using.
     */
    static readonly AMI_ID = "attribute:ecs.ami-id";
    /**
     * The EC2 instance type.
     */
    static readonly INSTANCE_TYPE = "attribute:ecs.instance-type";
    /**
     * The operating system of the instance.
     *
     * Either 'linux' or 'windows'.
     */
    static readonly OS_TYPE = "attribute:ecs.os-type";
}
