import ec2 = require('@aws-cdk/aws-ec2');
import cdk = require('@aws-cdk/core');
import { Construct } from '@aws-cdk/core';
import { BaseService, BaseServiceOptions, IService, PropagatedTagSource } from '../base/base-service';
import { TaskDefinition } from '../base/task-definition';
/**
 * The properties for defining a service using the Fargate launch type.
 */
export interface FargateServiceProps extends BaseServiceOptions {
    /**
     * The task definition to use for tasks in the service.
     *
     * [disable-awslint:ref-via-interface]
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * Specifies whether the task's elastic network interface receives a public IP address.
     *
     * If true, each task will receive a public IP address.
     *
     * @default - Use subnet default.
     */
    readonly assignPublicIp?: boolean;
    /**
     * The subnets to associate with the service.
     *
     * @default - Private subnets.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The security groups to associate with the service. If you do not specify a security group, the default security group for the VPC is used.
     *
     * @default - A new security group is created.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The platform version on which to run your service.
     *
     * If one is not specified, the LATEST platform version is used by default. For more information, see
     * [AWS Fargate Platform Versions](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html)
     * in the Amazon Elastic Container Service Developer Guide.
     *
     * @default Latest
     */
    readonly platformVersion?: FargatePlatformVersion;
    /**
     * Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.
     * Tags can only be propagated to the tasks within the service during service creation.
     *
     * @deprecated Use `propagateTags` instead.
     * @default PropagatedTagSource.NONE
     */
    readonly propagateTaskTagsFrom?: PropagatedTagSource;
}
/**
 * The interface for a service using the Fargate launch type on an ECS cluster.
 */
export interface IFargateService extends IService {
}
/**
 * This creates a service using the Fargate launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
export declare class FargateService extends BaseService implements IFargateService {
    /**
     * Import a task definition from the specified task definition ARN.
     */
    static fromFargateServiceArn(scope: Construct, id: string, fargateServiceArn: string): IFargateService;
    /**
     * Constructs a new instance of the FargateService class.
     */
    constructor(scope: cdk.Construct, id: string, props: FargateServiceProps);
}
/**
 * The platform version on which to run your service.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
 */
export declare enum FargatePlatformVersion {
    /**
     * The latest, recommended platform version
     */
    LATEST = "LATEST",
    /**
     * Version 1.3.0
     *
     * Supports secrets, task recycling.
     */
    VERSION1_3 = "1.3.0",
    /**
     * Version 1.2.0
     *
     * Supports private registries.
     */
    VERSION1_2 = "1.2.0",
    /**
     * Version 1.1.0
     *
     * Supports task metadata, health checks, service discovery.
     */
    VERSION1_1 = "1.1.0",
    /**
     * Initial release
     *
     * Based on Amazon Linux 2017.09.
     */
    VERSION1_0 = "1.0.0"
}
