"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const base_service_1 = require("../base/base-service");
/**
 * This creates a service using the Fargate launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
class FargateService extends base_service_1.BaseService {
    /**
     * Import a task definition from the specified task definition ARN.
     */
    static fromFargateServiceArn(scope, id, fargateServiceArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = fargateServiceArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Constructs a new instance of the FargateService class.
     */
    constructor(scope, id, props) {
        if (!props.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with Fargate');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, Object.assign(Object.assign({}, props), { desiredCount: props.desiredCount !== undefined ? props.desiredCount : 1, launchType: base_service_1.LaunchType.FARGATE, propagateTags: propagateTagsFromSource, enableECSManagedTags: props.enableECSManagedTags }), {
            cluster: props.cluster.clusterName,
            taskDefinition: props.taskDefinition.taskDefinitionArn,
            platformVersion: props.platformVersion,
        }, props.taskDefinition);
        this.configureAwsVpcNetworking(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, props.securityGroup);
        if (!props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
}
exports.FargateService = FargateService;
/**
 * The platform version on which to run your service.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
 */
var FargatePlatformVersion;
(function (FargatePlatformVersion) {
    /**
     * The latest, recommended platform version
     */
    FargatePlatformVersion["LATEST"] = "LATEST";
    /**
     * Version 1.3.0
     *
     * Supports secrets, task recycling.
     */
    FargatePlatformVersion["VERSION1_3"] = "1.3.0";
    /**
     * Version 1.2.0
     *
     * Supports private registries.
     */
    FargatePlatformVersion["VERSION1_2"] = "1.2.0";
    /**
     * Version 1.1.0
     *
     * Supports task metadata, health checks, service discovery.
     */
    FargatePlatformVersion["VERSION1_1"] = "1.1.0";
    /**
     * Initial release
     *
     * Based on Amazon Linux 2017.09.
     */
    FargatePlatformVersion["VERSION1_0"] = "1.0.0";
})(FargatePlatformVersion = exports.FargatePlatformVersion || (exports.FargatePlatformVersion = {}));
//# sourceMappingURL=data:application/json;base64,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