"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const task_definition_1 = require("../base/task-definition");
/**
 * The details of a task definition run on a Fargate cluster.
 *
 * @resource AWS::ECS::TaskDefinition
 */
class FargateTaskDefinition extends task_definition_1.TaskDefinition {
    // NOTE: Until the fix to https://github.com/Microsoft/TypeScript/issues/26969 gets released,
    // we need to explicitly write the type here, as type deduction for enums won't lead to
    // the import being generated in the .d.ts file.
    /**
     * Constructs a new instance of the FargateTaskDefinition class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, Object.assign(Object.assign({}, props), { cpu: props.cpu !== undefined ? core_1.Tokenization.stringifyNumber(props.cpu) : '256', memoryMiB: props.memoryLimitMiB !== undefined ? core_1.Tokenization.stringifyNumber(props.memoryLimitMiB) : '512', compatibility: task_definition_1.Compatibility.FARGATE, networkMode: task_definition_1.NetworkMode.AWS_VPC }));
        /**
         * The Docker networking mode to use for the containers in the task. Fargate tasks require the awsvpc network mode.
         */
        this.networkMode = task_definition_1.NetworkMode.AWS_VPC;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     */
    static fromFargateTaskDefinitionArn(scope, id, fargateTaskDefinitionArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = fargateTaskDefinitionArn;
                this.compatibility = task_definition_1.Compatibility.FARGATE;
                this.isEc2Compatible = false;
                this.isFargateCompatible = true;
            }
        }
        return new Import(scope, id);
    }
}
exports.FargateTaskDefinition = FargateTaskDefinition;
//# sourceMappingURL=data:application/json;base64,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