"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const container_image_1 = require("../container-image");
/**
 * Regex pattern to check if it is an ECR image URL.
 *
 * @experimental
 */
const ECR_IMAGE_REGEX = /(^[a-zA-Z0-9][a-zA-Z0-9-_]*).dkr.ecr.([a-zA-Z0-9][a-zA-Z0-9-_]*).amazonaws.com(.cn)?\/.*/;
/**
 * An image hosted in a public or private repository. For images hosted in Amazon ECR, see
 * [EcrImage](https://docs.aws.amazon.com/AmazonECR/latest/userguide/images.html).
 */
class RepositoryImage extends container_image_1.ContainerImage {
    /**
     * Constructs a new instance of the RepositoryImage class.
     */
    constructor(imageName, props = {}) {
        super();
        this.imageName = imageName;
        this.props = props;
    }
    bind(scope, containerDefinition) {
        // name could be a Token - in that case, skip validation altogether
        if (!core_1.Token.isUnresolved(this.imageName) && ECR_IMAGE_REGEX.test(this.imageName)) {
            scope.node.addWarning("Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
        }
        if (this.props.credentials) {
            this.props.credentials.grantRead(containerDefinition.taskDefinition.obtainExecutionRole());
        }
        return {
            imageName: this.imageName,
            repositoryCredentials: this.props.credentials && {
                credentialsParameter: this.props.credentials.secretArn
            }
        };
    }
}
exports.RepositoryImage = RepositoryImage;
//# sourceMappingURL=data:application/json;base64,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