import cdk = require('@aws-cdk/core');
import { CfnTaskDefinition } from './ecs.generated';
/**
 * The properties for defining Linux-specific options that are applied to the container.
 */
export interface LinuxParametersProps {
    /**
     * Specifies whether to run an init process inside the container that forwards signals and reaps processes.
     *
     * @default false
     */
    readonly initProcessEnabled?: boolean;
    /**
     * The value for the size (in MiB) of the /dev/shm volume.
     *
     * @default No shared memory.
     */
    readonly sharedMemorySize?: number;
}
/**
 * Linux-specific options that are applied to the container.
 */
export declare class LinuxParameters extends cdk.Construct {
    /**
     * Whether the init process is enabled
     */
    private readonly initProcessEnabled?;
    /**
     * The shared memory size. Not valid for Fargate launch type
     */
    private readonly sharedMemorySize?;
    /**
     * Capabilities to be added
     */
    private readonly capAdd;
    /**
     * Capabilities to be dropped
     */
    private readonly capDrop;
    /**
     * Device mounts
     */
    private readonly devices;
    /**
     * TmpFs mounts
     */
    private readonly tmpfs;
    /**
     * Constructs a new instance of the LinuxParameters class.
     */
    constructor(scope: cdk.Construct, id: string, props?: LinuxParametersProps);
    /**
     * Adds one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    addCapabilities(...cap: Capability[]): void;
    /**
     * Removes one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    dropCapabilities(...cap: Capability[]): void;
    /**
     * Adds one or more host devices to a container.
     */
    addDevices(...device: Device[]): void;
    /**
     * Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.
     *
     * Only works with EC2 launch type.
     */
    addTmpfs(...tmpfs: Tmpfs[]): void;
    /**
     * Renders the Linux parameters to a CloudFormation object.
     */
    renderLinuxParameters(): CfnTaskDefinition.LinuxParametersProperty;
}
/**
 * A container instance host device.
 */
export interface Device {
    /**
     * The path inside the container at which to expose the host device.
     *
     * @default Same path as the host
     */
    readonly containerPath?: string;
    /**
     * The path for the device on the host container instance.
     */
    readonly hostPath: string;
    /**
     * The explicit permissions to provide to the container for the device.
     * By default, the container has permissions for read, write, and mknod for the device.
     *
     * @default Readonly
     */
    readonly permissions?: DevicePermission[];
}
/**
 * The details of a tmpfs mount for a container.
 */
export interface Tmpfs {
    /**
     * The absolute file path where the tmpfs volume is to be mounted.
     */
    readonly containerPath: string;
    /**
     * The size (in MiB) of the tmpfs volume.
     */
    readonly size: number;
    /**
     * The list of tmpfs volume mount options. For more information, see
     * [TmpfsMountOptions](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Tmpfs.html).
     */
    readonly mountOptions?: TmpfsMountOption[];
}
/**
 * A Linux capability
 */
export declare enum Capability {
    ALL = "ALL",
    AUDIT_CONTROL = "AUDIT_CONTROL",
    AUDIT_WRITE = "AUDIT_WRITE",
    BLOCK_SUSPEND = "BLOCK_SUSPEND",
    CHOWN = "CHOWN",
    DAC_OVERRIDE = "DAC_OVERRIDE",
    DAC_READ_SEARCH = "DAC_READ_SEARCH",
    FOWNER = "FOWNER",
    FSETID = "FSETID",
    IPC_LOCK = "IPC_LOCK",
    IPC_OWNER = "IPC_OWNER",
    KILL = "KILL",
    LEASE = "LEASE",
    LINUX_IMMUTABLE = "LINUX_IMMUTABLE",
    MAC_ADMIN = "MAC_ADMIN",
    MAC_OVERRIDE = "MAC_OVERRIDE",
    MKNOD = "MKNOD",
    NET_ADMIN = "NET_ADMIN",
    NET_BIND_SERVICE = "NET_BIND_SERVICE",
    NET_BROADCAST = "NET_BROADCAST",
    NET_RAW = "NET_RAW",
    SETFCAP = "SETFCAP",
    SETGID = "SETGID",
    SETPCAP = "SETPCAP",
    SETUID = "SETUID",
    SYS_ADMIN = "SYS_ADMIN",
    SYS_BOOT = "SYS_BOOT",
    SYS_CHROOT = "SYS_CHROOT",
    SYS_MODULE = "SYS_MODULE",
    SYS_NICE = "SYS_NICE",
    SYS_PACCT = "SYS_PACCT",
    SYS_PTRACE = "SYS_PTRACE",
    SYS_RAWIO = "SYS_RAWIO",
    SYS_RESOURCE = "SYS_RESOURCE",
    SYS_TIME = "SYS_TIME",
    SYS_TTY_CONFIG = "SYS_TTY_CONFIG",
    SYSLOG = "SYSLOG",
    WAKE_ALARM = "WAKE_ALARM"
}
/**
 * Permissions for device access
 */
export declare enum DevicePermission {
    /**
     * Read
     */
    READ = "read",
    /**
     * Write
     */
    WRITE = "write",
    /**
     * Make a node
     */
    MKNOD = "mknod"
}
/**
 * The supported options for a tmpfs mount for a container.
 */
export declare enum TmpfsMountOption {
    DEFAULTS = "defaults",
    RO = "ro",
    RW = "rw",
    SUID = "suid",
    NOSUID = "nosuid",
    DEV = "dev",
    NODEV = "nodev",
    EXEC = "exec",
    NOEXEC = "noexec",
    SYNC = "sync",
    ASYNC = "async",
    DIRSYNC = "dirsync",
    REMOUNT = "remount",
    MAND = "mand",
    NOMAND = "nomand",
    ATIME = "atime",
    NOATIME = "noatime",
    DIRATIME = "diratime",
    NODIRATIME = "nodiratime",
    BIND = "bind",
    RBIND = "rbind",
    UNBINDABLE = "unbindable",
    RUNBINDABLE = "runbindable",
    PRIVATE = "private",
    RPRIVATE = "rprivate",
    SHARED = "shared",
    RSHARED = "rshared",
    SLAVE = "slave",
    RSLAVE = "rslave",
    RELATIME = "relatime",
    NORELATIME = "norelatime",
    STRICTATIME = "strictatime",
    NOSTRICTATIME = "nostrictatime",
    MODE = "mode",
    UID = "uid",
    GID = "gid",
    NR_INODES = "nr_inodes",
    NR_BLOCKS = "nr_blocks",
    MPOL = "mpol"
}
