"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
/**
 * Linux-specific options that are applied to the container.
 */
class LinuxParameters extends cdk.Construct {
    /**
     * Constructs a new instance of the LinuxParameters class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * Capabilities to be added
         */
        this.capAdd = new Array();
        /**
         * Capabilities to be dropped
         */
        this.capDrop = new Array();
        /**
         * Device mounts
         */
        this.devices = new Array();
        /**
         * TmpFs mounts
         */
        this.tmpfs = new Array();
        this.sharedMemorySize = props.sharedMemorySize;
        this.initProcessEnabled = props.initProcessEnabled;
    }
    /**
     * Adds one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    addCapabilities(...cap) {
        this.capAdd.push(...cap);
    }
    /**
     * Removes one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    dropCapabilities(...cap) {
        this.capDrop.push(...cap);
    }
    /**
     * Adds one or more host devices to a container.
     */
    addDevices(...device) {
        this.devices.push(...device);
    }
    /**
     * Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.
     *
     * Only works with EC2 launch type.
     */
    addTmpfs(...tmpfs) {
        this.tmpfs.push(...tmpfs);
    }
    /**
     * Renders the Linux parameters to a CloudFormation object.
     */
    renderLinuxParameters() {
        return {
            initProcessEnabled: this.initProcessEnabled,
            sharedMemorySize: this.sharedMemorySize,
            capabilities: {
                add: cdk.Lazy.listValue({ produce: () => this.capAdd }, { omitEmpty: true }),
                drop: cdk.Lazy.listValue({ produce: () => this.capDrop }, { omitEmpty: true }),
            },
            devices: cdk.Lazy.anyValue({ produce: () => this.devices.map(renderDevice) }, { omitEmptyArray: true }),
            tmpfs: cdk.Lazy.anyValue({ produce: () => this.tmpfs.map(renderTmpfs) }, { omitEmptyArray: true })
        };
    }
}
exports.LinuxParameters = LinuxParameters;
function renderDevice(device) {
    return {
        containerPath: device.containerPath,
        hostPath: device.hostPath,
        permissions: device.permissions
    };
}
function renderTmpfs(tmpfs) {
    return {
        containerPath: tmpfs.containerPath,
        size: tmpfs.size,
        mountOptions: tmpfs.mountOptions
    };
}
/**
 * A Linux capability
 */
var Capability;
(function (Capability) {
    Capability["ALL"] = "ALL";
    Capability["AUDIT_CONTROL"] = "AUDIT_CONTROL";
    Capability["AUDIT_WRITE"] = "AUDIT_WRITE";
    Capability["BLOCK_SUSPEND"] = "BLOCK_SUSPEND";
    Capability["CHOWN"] = "CHOWN";
    Capability["DAC_OVERRIDE"] = "DAC_OVERRIDE";
    Capability["DAC_READ_SEARCH"] = "DAC_READ_SEARCH";
    Capability["FOWNER"] = "FOWNER";
    Capability["FSETID"] = "FSETID";
    Capability["IPC_LOCK"] = "IPC_LOCK";
    Capability["IPC_OWNER"] = "IPC_OWNER";
    Capability["KILL"] = "KILL";
    Capability["LEASE"] = "LEASE";
    Capability["LINUX_IMMUTABLE"] = "LINUX_IMMUTABLE";
    Capability["MAC_ADMIN"] = "MAC_ADMIN";
    Capability["MAC_OVERRIDE"] = "MAC_OVERRIDE";
    Capability["MKNOD"] = "MKNOD";
    Capability["NET_ADMIN"] = "NET_ADMIN";
    Capability["NET_BIND_SERVICE"] = "NET_BIND_SERVICE";
    Capability["NET_BROADCAST"] = "NET_BROADCAST";
    Capability["NET_RAW"] = "NET_RAW";
    Capability["SETFCAP"] = "SETFCAP";
    Capability["SETGID"] = "SETGID";
    Capability["SETPCAP"] = "SETPCAP";
    Capability["SETUID"] = "SETUID";
    Capability["SYS_ADMIN"] = "SYS_ADMIN";
    Capability["SYS_BOOT"] = "SYS_BOOT";
    Capability["SYS_CHROOT"] = "SYS_CHROOT";
    Capability["SYS_MODULE"] = "SYS_MODULE";
    Capability["SYS_NICE"] = "SYS_NICE";
    Capability["SYS_PACCT"] = "SYS_PACCT";
    Capability["SYS_PTRACE"] = "SYS_PTRACE";
    Capability["SYS_RAWIO"] = "SYS_RAWIO";
    Capability["SYS_RESOURCE"] = "SYS_RESOURCE";
    Capability["SYS_TIME"] = "SYS_TIME";
    Capability["SYS_TTY_CONFIG"] = "SYS_TTY_CONFIG";
    Capability["SYSLOG"] = "SYSLOG";
    Capability["WAKE_ALARM"] = "WAKE_ALARM";
})(Capability = exports.Capability || (exports.Capability = {}));
/**
 * Permissions for device access
 */
var DevicePermission;
(function (DevicePermission) {
    /**
     * Read
     */
    DevicePermission["READ"] = "read";
    /**
     * Write
     */
    DevicePermission["WRITE"] = "write";
    /**
     * Make a node
     */
    DevicePermission["MKNOD"] = "mknod";
})(DevicePermission = exports.DevicePermission || (exports.DevicePermission = {}));
/**
 * The supported options for a tmpfs mount for a container.
 */
var TmpfsMountOption;
(function (TmpfsMountOption) {
    TmpfsMountOption["DEFAULTS"] = "defaults";
    TmpfsMountOption["RO"] = "ro";
    TmpfsMountOption["RW"] = "rw";
    TmpfsMountOption["SUID"] = "suid";
    TmpfsMountOption["NOSUID"] = "nosuid";
    TmpfsMountOption["DEV"] = "dev";
    TmpfsMountOption["NODEV"] = "nodev";
    TmpfsMountOption["EXEC"] = "exec";
    TmpfsMountOption["NOEXEC"] = "noexec";
    TmpfsMountOption["SYNC"] = "sync";
    TmpfsMountOption["ASYNC"] = "async";
    TmpfsMountOption["DIRSYNC"] = "dirsync";
    TmpfsMountOption["REMOUNT"] = "remount";
    TmpfsMountOption["MAND"] = "mand";
    TmpfsMountOption["NOMAND"] = "nomand";
    TmpfsMountOption["ATIME"] = "atime";
    TmpfsMountOption["NOATIME"] = "noatime";
    TmpfsMountOption["DIRATIME"] = "diratime";
    TmpfsMountOption["NODIRATIME"] = "nodiratime";
    TmpfsMountOption["BIND"] = "bind";
    TmpfsMountOption["RBIND"] = "rbind";
    TmpfsMountOption["UNBINDABLE"] = "unbindable";
    TmpfsMountOption["RUNBINDABLE"] = "runbindable";
    TmpfsMountOption["PRIVATE"] = "private";
    TmpfsMountOption["RPRIVATE"] = "rprivate";
    TmpfsMountOption["SHARED"] = "shared";
    TmpfsMountOption["RSHARED"] = "rshared";
    TmpfsMountOption["SLAVE"] = "slave";
    TmpfsMountOption["RSLAVE"] = "rslave";
    TmpfsMountOption["RELATIME"] = "relatime";
    TmpfsMountOption["NORELATIME"] = "norelatime";
    TmpfsMountOption["STRICTATIME"] = "strictatime";
    TmpfsMountOption["NOSTRICTATIME"] = "nostrictatime";
    TmpfsMountOption["MODE"] = "mode";
    TmpfsMountOption["UID"] = "uid";
    TmpfsMountOption["GID"] = "gid";
    TmpfsMountOption["NR_INODES"] = "nr_inodes";
    TmpfsMountOption["NR_BLOCKS"] = "nr_blocks";
    TmpfsMountOption["MPOL"] = "mpol";
})(TmpfsMountOption = exports.TmpfsMountOption || (exports.TmpfsMountOption = {}));
//# sourceMappingURL=data:application/json;base64,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