"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const proxy_configuration_1 = require("./proxy-configuration");
/**
 * The class for App Mesh proxy configurations.
 *
 * For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent and at least version
 * 1.26.0-1 of the ecs-init package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized
 * AMI version 20190301 or later, then they contain the required versions of the container agent and ecs-init.
 * For more information, see [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 *
 * For tasks using the Fargate launch type, the task or service requires platform version 1.3.0 or later.
 */
class AppMeshProxyConfiguration extends proxy_configuration_1.ProxyConfiguration {
    /**
     * Constructs a new instance of the AppMeshProxyConfiguration class.
     */
    constructor(props) {
        super();
        this.props = props;
        if (props.properties) {
            if (!props.properties.ignoredUID && !props.properties.ignoredGID) {
                throw new Error("At least one of ignoredUID or ignoredGID should be specified.");
            }
        }
    }
    /**
     * Called when the proxy configuration is configured on a task definition.
     */
    bind(_scope, _taskDefinition) {
        const configProps = this.props.properties;
        const configType = "APPMESH";
        return {
            containerName: this.props.containerName,
            proxyConfigurationProperties: renderProperties(configProps),
            type: configType
        };
    }
}
exports.AppMeshProxyConfiguration = AppMeshProxyConfiguration;
function renderProperties(props) {
    const ret = new Array();
    for (const [k, v] of Object.entries(props)) {
        const key = String(k);
        const value = String(v);
        if (value !== "undefined" && value !== "") {
            const capitalizedKey = key.charAt(0).toUpperCase() + key.slice(1);
            ret.push({ ["name"]: capitalizedKey, ["value"]: value });
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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