"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_1 = require("@aws-cdk/aws-ecr");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const path = require("path");
const ecs = require("../../lib");
module.exports = {
    "When creating an ECS TaskDefinition": {
        "with only required properties set, it correctly sets default properties"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                NetworkMode: ecs.NetworkMode.BRIDGE,
                RequiresCompatibilities: ["EC2"]
            }));
            // test error if no container defs?
            test.done();
        },
        "with all properties set"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                executionRole: new iam.Role(stack, 'ExecutionRole', {
                    path: '/',
                    assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal("ecs.amazonaws.com"), new iam.ServicePrincipal("ecs-tasks.amazonaws.com"))
                }),
                family: "ecs-tasks",
                networkMode: ecs.NetworkMode.AWS_VPC,
                placementConstraints: [ecs.PlacementConstraint.memberOf("attribute:ecs.instance-type =~ t2.*")],
                taskRole: new iam.Role(stack, 'TaskRole', {
                    assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                }),
                volumes: [{
                        host: {
                            sourcePath: "/tmp/cache",
                        },
                        name: "scratch"
                    }]
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                ExecutionRoleArn: {
                    "Fn::GetAtt": [
                        "ExecutionRole605A040B",
                        "Arn"
                    ]
                },
                Family: "ecs-tasks",
                NetworkMode: "awsvpc",
                PlacementConstraints: [
                    {
                        Expression: "attribute:ecs.instance-type =~ t2.*",
                        Type: "memberOf"
                    }
                ],
                RequiresCompatibilities: [
                    "EC2"
                ],
                TaskRoleArn: {
                    "Fn::GetAtt": [
                        "TaskRole30FC0FBB",
                        "Arn"
                    ]
                },
                Volumes: [
                    {
                        Host: {
                            SourcePath: "/tmp/cache"
                        },
                        Name: "scratch"
                    }
                ]
            }));
            test.done();
        },
        "correctly sets placement constraint"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addPlacementConstraint(ecs.PlacementConstraint.memberOf("attribute:ecs.instance-type =~ t2.*"));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                PlacementConstraints: [
                    {
                        Expression: "attribute:ecs.instance-type =~ t2.*",
                        Type: "memberOf"
                    }
                ],
            }));
            test.done();
        },
        "correctly sets network mode"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                NetworkMode: ecs.NetworkMode.AWS_VPC,
            }));
            test.done();
        },
        "correctly sets containers"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512 // add validation?
            });
            container.addPortMappings({
                containerPort: 3000
            });
            container.addUlimits({
                hardLimit: 128,
                name: ecs.UlimitName.RSS,
                softLimit: 128
            });
            container.addVolumesFrom({
                sourceContainer: "foo",
                readOnly: true
            });
            container.addToExecutionPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['ecs:*'],
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [{
                        Essential: true,
                        Memory: 512,
                        Image: "amazon/amazon-ecs-sample",
                        Name: "web",
                        PortMappings: [{
                                ContainerPort: 3000,
                                HostPort: 0,
                                Protocol: aws_ec2_1.Protocol.TCP
                            }],
                        Ulimits: [
                            {
                                HardLimit: 128,
                                Name: "rss",
                                SoftLimit: 128
                            }
                        ],
                        VolumesFrom: [
                            {
                                ReadOnly: true,
                                SourceContainer: "foo"
                            }
                        ]
                    }],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: "ecs:*",
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                },
            }));
            test.done();
        },
        "all container definition options defined"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const secret = new secretsmanager.Secret(stack, 'Secret');
            const parameter = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'Parameter', {
                parameterName: '/name',
                version: 1
            });
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 2048,
                cpu: 256,
                disableNetworking: true,
                command: ['CMD env'],
                dnsSearchDomains: ['0.0.0.0'],
                dnsServers: ['1.1.1.1'],
                dockerLabels: { LABEL: 'label' },
                dockerSecurityOptions: ['ECS_SELINUX_CAPABLE=true'],
                entryPoint: ["/app/node_modules/.bin/cdk"],
                environment: { TEST_ENVIRONMENT_VARIABLE: "test environment variable value" },
                essential: true,
                extraHosts: { EXTRAHOST: 'extra host' },
                healthCheck: {
                    command: ["curl localhost:8000"],
                    interval: cdk.Duration.seconds(20),
                    retries: 5,
                    startPeriod: cdk.Duration.seconds(10)
                },
                hostname: "webHost",
                linuxParameters: new ecs.LinuxParameters(stack, 'LinuxParameters', {
                    initProcessEnabled: true,
                    sharedMemorySize: 1024,
                }),
                logging: new ecs.AwsLogDriver({ streamPrefix: 'prefix' }),
                memoryReservationMiB: 1024,
                privileged: true,
                readonlyRootFilesystem: true,
                secrets: {
                    SECRET: ecs.Secret.fromSecretsManager(secret),
                    PARAMETER: ecs.Secret.fromSsmParameter(parameter),
                },
                user: "amazon",
                workingDirectory: "app/"
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [
                    {
                        Command: [
                            "CMD env"
                        ],
                        Cpu: 256,
                        DisableNetworking: true,
                        DnsSearchDomains: [
                            "0.0.0.0"
                        ],
                        DnsServers: [
                            "1.1.1.1"
                        ],
                        DockerLabels: {
                            LABEL: "label"
                        },
                        DockerSecurityOptions: [
                            "ECS_SELINUX_CAPABLE=true"
                        ],
                        EntryPoint: [
                            "/app/node_modules/.bin/cdk"
                        ],
                        Environment: [
                            {
                                Name: "TEST_ENVIRONMENT_VARIABLE",
                                Value: "test environment variable value"
                            }
                        ],
                        Essential: true,
                        ExtraHosts: [
                            {
                                Hostname: "EXTRAHOST",
                                IpAddress: "extra host"
                            }
                        ],
                        HealthCheck: {
                            Command: [
                                "CMD-SHELL",
                                "curl localhost:8000"
                            ],
                            Interval: 20,
                            Retries: 5,
                            StartPeriod: 10,
                            Timeout: 5
                        },
                        Hostname: "webHost",
                        Image: "amazon/amazon-ecs-sample",
                        LinuxParameters: {
                            Capabilities: {},
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                        LogConfiguration: {
                            LogDriver: "awslogs",
                            Options: {
                                "awslogs-group": {
                                    Ref: "Ec2TaskDefwebLogGroup7F786C6B"
                                },
                                "awslogs-stream-prefix": "prefix",
                                "awslogs-region": {
                                    Ref: "AWS::Region"
                                }
                            }
                        },
                        Memory: 2048,
                        MemoryReservation: 1024,
                        Name: "web",
                        Privileged: true,
                        ReadonlyRootFilesystem: true,
                        Secrets: [
                            {
                                Name: "SECRET",
                                ValueFrom: {
                                    Ref: "SecretA720EF05"
                                }
                            },
                            {
                                Name: "PARAMETER",
                                ValueFrom: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":ssm:",
                                            {
                                                Ref: "AWS::Region"
                                            },
                                            ":",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":parameter/name"
                                        ]
                                    ]
                                }
                            }
                        ],
                        User: "amazon",
                        WorkingDirectory: "app/"
                    }
                ],
            }));
            test.done();
        },
        "correctly sets containers from ECR repository using all props"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromEcrRepository(new aws_ecr_1.Repository(stack, "myECRImage", {
                    lifecycleRegistryId: '123456789101',
                    lifecycleRules: [{
                            rulePriority: 10,
                            tagPrefixList: ['abc'],
                            maxImageCount: 1
                        }],
                    removalPolicy: cdk.RemovalPolicy.DESTROY,
                    repositoryName: 'project-a/amazon-ecs-sample'
                })),
                memoryLimitMiB: 512
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                LifecyclePolicy: {
                    // tslint:disable-next-line:max-line-length
                    LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":10,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"abc\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":1},\"action\":{\"type\":\"expire\"}}]}",
                    RegistryId: "123456789101"
                },
                RepositoryName: "project-a/amazon-ecs-sample"
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [{
                        Essential: true,
                        Memory: 512,
                        Image: {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::Select": [
                                            4,
                                            {
                                                "Fn::Split": [
                                                    ":",
                                                    {
                                                        "Fn::GetAtt": [
                                                            "myECRImage7DEAE474",
                                                            "Arn"
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    ".dkr.ecr.",
                                    {
                                        "Fn::Select": [
                                            3,
                                            {
                                                "Fn::Split": [
                                                    ":",
                                                    {
                                                        "Fn::GetAtt": [
                                                            "myECRImage7DEAE474",
                                                            "Arn"
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    ".",
                                    {
                                        Ref: "AWS::URLSuffix"
                                    },
                                    "/",
                                    {
                                        Ref: "myECRImage7DEAE474"
                                    },
                                    ":latest"
                                ]
                            ]
                        },
                        Name: "web"
                    }],
            }));
            test.done();
        },
        "correctly sets containers from ECR repository using default props"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromEcrRepository(new aws_ecr_1.Repository(stack, "myECRImage")),
                memoryLimitMiB: 512
            });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ECR::Repository', {}));
            test.done();
        },
        "warns when setting containers from ECR repository using fromRegistry method"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY"),
                memoryLimitMiB: 512
            });
            // THEN
            test.deepEqual(container.node.metadata[0].data, "Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
            test.done();
        },
        "warns when setting containers from ECR repository by creating a RepositoryImage class"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const repo = new ecs.RepositoryImage("ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY");
            // WHEN
            const container = taskDefinition.addContainer("web", {
                image: repo,
                memoryLimitMiB: 512
            });
            // THEN
            test.deepEqual(container.node.metadata[0].data, "Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
            test.done();
        },
        "correctly sets containers from asset using default props"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromAsset(path.join(__dirname, '..', 'demo-image')),
                memoryLimitMiB: 512
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [
                    {
                        Essential: true,
                        Image: {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ".dkr.ecr.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".",
                                    {
                                        Ref: "AWS::URLSuffix"
                                    },
                                    "/",
                                    {
                                        "Fn::Select": [
                                            0,
                                            {
                                                "Fn::Split": [
                                                    "@sha256:",
                                                    {
                                                        Ref: "AssetParameters1a17a141505ac69144931fe263d130f4612251caa4bbbdaf68a44ed0f405439cImageName1ADCADB3"
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    "@sha256:",
                                    {
                                        "Fn::Select": [
                                            1,
                                            {
                                                "Fn::Split": [
                                                    "@sha256:",
                                                    {
                                                        Ref: "AssetParameters1a17a141505ac69144931fe263d130f4612251caa4bbbdaf68a44ed0f405439cImageName1ADCADB3"
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            ]
                        },
                        Memory: 512,
                        Name: "web"
                    }
                ],
            }));
            test.done();
        },
        "correctly sets containers from asset using all props"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromAsset(path.join(__dirname, '..', 'demo-image'), {
                    buildArgs: { HTTP_PROXY: 'http://10.20.30.2:1234' }
                }),
                memoryLimitMiB: 512
            });
            test.done();
        },
        "correctly sets scratch space"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            container.addScratch({
                containerPath: "./cache",
                readOnly: true,
                sourcePath: "/tmp/cache",
                name: "scratch"
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [{
                        MountPoints: [
                            {
                                ContainerPath: "./cache",
                                ReadOnly: true,
                                SourceVolume: "scratch"
                            }
                        ]
                    }],
                Volumes: [{
                        Host: {
                            SourcePath: "/tmp/cache"
                        },
                        Name: "scratch"
                    }]
            }));
            test.done();
        },
        "correctly sets container dependenices"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const dependency1 = taskDefinition.addContainer('dependency1', {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            const dependency2 = taskDefinition.addContainer('dependency2', {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            container.addContainerDependencies({
                container: dependency1
            }, {
                container: dependency2,
                condition: ecs.ContainerDependencyCondition.SUCCESS
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [{
                        Name: "dependency1"
                    },
                    {
                        Name: "dependency2"
                    },
                    {
                        Name: "web",
                        DependsOn: [{
                                Condition: "HEALTHY",
                                ContainerName: "dependency1"
                            },
                            {
                                Condition: "SUCCESS",
                                ContainerName: "dependency2"
                            }]
                    }]
            }));
            test.done();
        },
        "correctly sets links"(test) {
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.BRIDGE
            });
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            const linkedContainer1 = taskDefinition.addContainer("linked1", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            const linkedContainer2 = taskDefinition.addContainer("linked2", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            container.addLink(linkedContainer1, 'linked');
            container.addLink(linkedContainer2);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                ContainerDefinitions: [
                    {
                        Links: [
                            'linked1:linked',
                            'linked2'
                        ],
                        Name: "web"
                    },
                    {
                        Name: 'linked1'
                    },
                    {
                        Name: 'linked2'
                    }
                ]
            }));
            test.done();
        },
        "correctly set policy statement to the task IAM role"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
                actions: ['test:SpecialName'],
                resources: ['*']
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: "test:SpecialName",
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                },
            }));
            test.done();
        },
        "correctly sets volumes from"(test) {
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {});
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            container.addVolumesFrom({
                sourceContainer: "SourceContainer",
                readOnly: true
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                ContainerDefinitions: [{
                        VolumesFrom: [
                            {
                                SourceContainer: "SourceContainer",
                                ReadOnly: true,
                            }
                        ]
                    }]
            }));
            test.done();
        },
        "correctly set policy statement to the task execution IAM role"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                actions: ['test:SpecialName'],
                resources: ['*']
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: "test:SpecialName",
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                },
            }));
            test.done();
        },
        "correctly sets volumes"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const volume = {
                host: {
                    sourcePath: "/tmp/cache",
                },
                name: "scratch"
            };
            // Adding volumes via props is a bit clunky
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                volumes: [volume]
            });
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            // this needs to be a better API -- should auto-add volumes
            container.addMountPoints({
                containerPath: "./cache",
                readOnly: true,
                sourceVolume: "scratch",
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                ContainerDefinitions: [{
                        MountPoints: [
                            {
                                ContainerPath: "./cache",
                                ReadOnly: true,
                                SourceVolume: "scratch"
                            }
                        ]
                    }],
                Volumes: [{
                        Host: {
                            SourcePath: "/tmp/cache"
                        },
                        Name: "scratch"
                    }]
            }));
            test.done();
        },
        "correctly sets placement constraints"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                placementConstraints: [
                    ecs.PlacementConstraint.memberOf("attribute:ecs.instance-type =~ t2.*"),
                ]
            });
            taskDefinition.addContainer("web", {
                memoryLimitMiB: 1024,
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::TaskDefinition", {
                PlacementConstraints: [
                    {
                        Expression: "attribute:ecs.instance-type =~ t2.*",
                        Type: "memberOf"
                    }
                ]
            }));
            test.done();
        },
        "correctly sets taskRole"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                taskRole: new iam.Role(stack, 'TaskRole', {
                    assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                })
            });
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                TaskRoleArn: stack.resolve(taskDefinition.taskRole.roleArn)
            }));
            test.done();
        },
        "automatically sets taskRole by default"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                TaskRoleArn: stack.resolve(taskDefinition.taskRole.roleArn)
            }));
            test.done();
        },
        "correctly sets dockerVolumeConfiguration"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const volume = {
                name: "scratch",
                dockerVolumeConfiguration: {
                    driver: "local",
                    scope: ecs.Scope.TASK,
                    driverOpts: {
                        key1: "value"
                    }
                }
            };
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                volumes: [volume]
            });
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Family: "Ec2TaskDef",
                Volumes: [{
                        Name: "scratch",
                        DockerVolumeConfiguration: {
                            Driver: "local",
                            Scope: 'task',
                            DriverOpts: {
                                key1: "value"
                            }
                        }
                    }]
            }));
            test.done();
        }
    },
    'throws when setting proxyConfiguration without networkMode AWS_VPC'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const proxyConfiguration = ecs.ProxyConfigurations.appMeshProxyConfiguration({
            containerName: 'envoy',
            properties: {
                ignoredUID: 1337,
                proxyIngressPort: 15000,
                proxyEgressPort: 15001,
                appPorts: [9080, 9081],
                egressIgnoredIPs: ["169.254.170.2", "169.254.169.254"]
            }
        });
        // THEN
        test.throws(() => {
            new ecs.Ec2TaskDefinition(stack, 'TaskDef', { networkMode: ecs.NetworkMode.BRIDGE, proxyConfiguration });
        }, /ProxyConfiguration can only be used with AwsVpc network mode, got: bridge/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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