"use strict";
const assert_1 = require("@aws-cdk/assert");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const ecs = require("../../lib");
const lib_1 = require("../../lib");
const base_service_1 = require("../../lib/base/base-service");
module.exports = {
    "When creating a Fargate Service": {
        "with only required properties set, it correctly sets default properties"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
            });
            new ecs.FargateService(stack, "FargateService", {
                cluster,
                taskDefinition,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::Service", {
                TaskDefinition: {
                    Ref: "FargateTaskDefC6FB60B4"
                },
                Cluster: {
                    Ref: "EcsCluster97242B84"
                },
                DeploymentConfiguration: {
                    MaximumPercent: 200,
                    MinimumHealthyPercent: 50
                },
                DesiredCount: 1,
                LaunchType: base_service_1.LaunchType.FARGATE,
                EnableECSManagedTags: false,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: "DISABLED",
                        SecurityGroups: [
                            {
                                "Fn::GetAtt": [
                                    "FargateServiceSecurityGroup0A0E79CB",
                                    "GroupId"
                                ]
                            }
                        ],
                        Subnets: [
                            {
                                Ref: "MyVpcPrivateSubnet1Subnet5057CF7E"
                            },
                            {
                                Ref: "MyVpcPrivateSubnet2Subnet0040C983"
                            },
                        ]
                    }
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroup", {
                GroupDescription: "FargateService/SecurityGroup",
                SecurityGroupEgress: [
                    {
                        CidrIp: "0.0.0.0/0",
                        Description: "Allow all outbound traffic by default",
                        IpProtocol: "-1"
                    }
                ],
                VpcId: {
                    Ref: "MyVpcF9F0CA6F"
                }
            }));
            test.done();
        },
        "with custom cloudmap namespace"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
                memoryLimitMiB: 512
            });
            container.addPortMappings({ containerPort: 8000 });
            const cloudMapNamespace = new cloudmap.PrivateDnsNamespace(stack, 'TestCloudMapNamespace', {
                name: "scorekeep.com",
                vpc,
            });
            new ecs.FargateService(stack, "FargateService", {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: "myApp",
                    failureThreshold: 20,
                    cloudMapNamespace,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: "A"
                        }
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'TestCloudMapNamespace1FB9B446',
                            'Id'
                        ]
                    },
                    RoutingPolicy: 'MULTIVALUE'
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 20
                },
                Name: "myApp",
                NamespaceId: {
                    'Fn::GetAtt': [
                        'TestCloudMapNamespace1FB9B446',
                        'Id'
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::PrivateDnsNamespace', {
                Name: "scorekeep.com",
                Vpc: {
                    Ref: "MyVpcF9F0CA6F"
                }
            }));
            test.done();
        },
        "with all properties set"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE
            });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
            });
            const svc = new ecs.FargateService(stack, "FargateService", {
                cluster,
                taskDefinition,
                desiredCount: 2,
                assignPublicIp: true,
                cloudMapOptions: {
                    name: "myapp",
                    dnsRecordType: cloudmap.DnsRecordType.A,
                    dnsTtl: cdk.Duration.seconds(50),
                    failureThreshold: 20
                },
                healthCheckGracePeriod: cdk.Duration.seconds(60),
                maxHealthyPercent: 150,
                minHealthyPercent: 55,
                securityGroup: new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                    allowAllOutbound: true,
                    description: 'Example',
                    securityGroupName: 'Bob',
                    vpc,
                }),
                serviceName: "bonjour",
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC }
            });
            // THEN
            test.ok(svc.cloudMapService !== undefined);
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::Service", {
                TaskDefinition: {
                    Ref: "FargateTaskDefC6FB60B4"
                },
                Cluster: {
                    Ref: "EcsCluster97242B84"
                },
                DeploymentConfiguration: {
                    MaximumPercent: 150,
                    MinimumHealthyPercent: 55
                },
                DesiredCount: 2,
                HealthCheckGracePeriodSeconds: 60,
                LaunchType: base_service_1.LaunchType.FARGATE,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: "ENABLED",
                        SecurityGroups: [
                            {
                                "Fn::GetAtt": [
                                    "SecurityGroup1F554B36F",
                                    "GroupId"
                                ]
                            }
                        ],
                        Subnets: [
                            {
                                Ref: "MyVpcPublicSubnet1SubnetF6608456"
                            },
                            {
                                Ref: "MyVpcPublicSubnet2Subnet492B6BFB"
                            }
                        ]
                    }
                },
                ServiceName: "bonjour",
                ServiceRegistries: [
                    {
                        RegistryArn: {
                            "Fn::GetAtt": [
                                "FargateServiceCloudmapService9544B753",
                                "Arn"
                            ]
                        }
                    }
                ]
            }));
            test.done();
        },
        "throws when task definition is not Fargate compatible"(test) {
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.TaskDefinition(stack, 'Ec2TaskDef', {
                compatibility: ecs.Compatibility.EC2,
            });
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, "FargateService", {
                    cluster,
                    taskDefinition,
                });
            }, /Supplied TaskDefinition is not configured for compatibility with Fargate/);
            test.done();
        },
        "errors when no container specified on task definition"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, "FargateService", {
                    cluster,
                    taskDefinition,
                });
            });
            test.done();
        },
        "allows specifying assignPublicIP as enabled"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
            });
            new ecs.FargateService(stack, "FargateService", {
                cluster,
                taskDefinition,
                assignPublicIp: true
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::Service", {
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: "ENABLED",
                    }
                }
            }));
            test.done();
        },
        "allows specifying 0 for minimumHealthyPercent"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer("web", {
                image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample"),
            });
            new ecs.FargateService(stack, "FargateService", {
                cluster,
                taskDefinition,
                minHealthyPercent: 0,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::Service", {
                DeploymentConfiguration: {
                    MinimumHealthyPercent: 0,
                }
            }));
            test.done();
        },
    },
    "When setting up a health check": {
        'grace period is respected'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('MainContainer', {
                image: lib_1.ContainerImage.fromRegistry('hello'),
            });
            // WHEN
            new ecs.FargateService(stack, 'Svc', {
                cluster,
                taskDefinition,
                healthCheckGracePeriod: cdk.Duration.seconds(10)
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                HealthCheckGracePeriodSeconds: 10
            }));
            test.done();
        },
    },
    "When adding an app load balancer": {
        'allows auto scaling by ALB request per target'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: lib_1.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.FargateService(stack, 'Service', { cluster, taskDefinition });
            const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
            const listener = lb.addListener("listener", { port: 80 });
            const targetGroup = listener.addTargets("target", {
                port: 80,
                targets: [service]
            });
            // WHEN
            const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
            capacity.scaleOnRequestCount("ScaleOnRequests", {
                requestsPerTarget: 1000,
                targetGroup
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
                MaxCapacity: 10,
                MinCapacity: 1,
                ResourceId: {
                    "Fn::Join": [
                        "",
                        [
                            "service/",
                            {
                                Ref: "EcsCluster97242B84"
                            },
                            "/",
                            {
                                "Fn::GetAtt": [
                                    "ServiceD69D759B",
                                    "Name"
                                ]
                            }
                        ]
                    ]
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
                TargetTrackingScalingPolicyConfiguration: {
                    PredefinedMetricSpecification: {
                        PredefinedMetricType: "ALBRequestCountPerTarget",
                        ResourceLabel: {
                            "Fn::Join": ["", [
                                    { "Fn::Select": [1, { "Fn::Split": ["/", { Ref: "lblistener657ADDEC" }] }] }, "/",
                                    { "Fn::Select": [2, { "Fn::Split": ["/", { Ref: "lblistener657ADDEC" }] }] }, "/",
                                    { "Fn::Select": [3, { "Fn::Split": ["/", { Ref: "lblistener657ADDEC" }] }] }, "/",
                                    { "Fn::GetAtt": ["lblistenertargetGroupC7489D1E", "TargetGroupFullName"] }
                                ]]
                        }
                    },
                    TargetValue: 1000
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                // if any load balancer is configured and healthCheckGracePeriodSeconds is not
                // set, then it should default to 60 seconds.
                HealthCheckGracePeriodSeconds: 60
            }));
            test.done();
        },
        'allows auto scaling by ALB with new service arn format'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: lib_1.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition
            });
            const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
            const listener = lb.addListener("listener", { port: 80 });
            const targetGroup = listener.addTargets("target", {
                port: 80,
                targets: [service]
            });
            // WHEN
            const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
            capacity.scaleOnRequestCount("ScaleOnRequests", {
                requestsPerTarget: 1000,
                targetGroup
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
                MaxCapacity: 10,
                MinCapacity: 1,
                ResourceId: {
                    "Fn::Join": [
                        "",
                        [
                            "service/",
                            {
                                Ref: "EcsCluster97242B84"
                            },
                            "/",
                            {
                                "Fn::GetAtt": [
                                    "ServiceD69D759B",
                                    "Name"
                                ]
                            }
                        ]
                    ]
                },
            }));
            test.done();
        },
        'allows specify any existing container name and port in a service': {
            'with default setting'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: lib_1.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                const listener = lb.addListener("listener", { port: 80 });
                listener.addTargets("target", {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: "MainContainer"
                        })]
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                    LoadBalancers: [
                        {
                            ContainerName: "MainContainer",
                            ContainerPort: 8000,
                            TargetGroupArn: {
                                Ref: "lblistenertargetGroupC7489D1E"
                            }
                        }
                    ],
                }));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                    Description: "Load balancer to target",
                    FromPort: 8000,
                    ToPort: 8000,
                }));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                    Description: "Load balancer to target",
                    FromPort: 8000,
                    ToPort: 8000
                }));
                test.done();
            },
            'with TCP protocol'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: lib_1.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001, protocol: ecs.Protocol.TCP });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                const listener = lb.addListener("listener", { port: 80 });
                // THEN
                listener.addTargets("target", {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: "MainContainer",
                            containerPort: 8001,
                            protocol: ecs.Protocol.TCP
                        })]
                });
                test.done();
            },
            'with UDP protocol'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: lib_1.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001, protocol: ecs.Protocol.UDP });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                const listener = lb.addListener("listener", { port: 80 });
                // THEN
                listener.addTargets("target", {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: "MainContainer",
                            containerPort: 8001,
                            protocol: ecs.Protocol.UDP
                        })]
                });
                test.done();
            },
            'throws when protocol does not match'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: lib_1.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001, protocol: ecs.Protocol.UDP });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                const listener = lb.addListener("listener", { port: 80 });
                // THEN
                test.throws(() => {
                    listener.addTargets("target", {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: "MainContainer",
                                containerPort: 8001,
                                protocol: ecs.Protocol.TCP
                            })]
                    });
                }, /Container 'FargateTaskDef\/MainContainer' has no mapping for port 8001 and protocol tcp. Did you call "container.addPortMapping()"?/);
                test.done();
            },
            'throws when port does not match'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: lib_1.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                const listener = lb.addListener("listener", { port: 80 });
                // THEN
                test.throws(() => {
                    listener.addTargets("target", {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: "MainContainer",
                                containerPort: 8002,
                            })]
                    });
                }, /Container 'FargateTaskDef\/MainContainer' has no mapping for port 8002 and protocol tcp. Did you call "container.addPortMapping()"?/);
                test.done();
            },
            'throws when container does not exist'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: lib_1.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                const listener = lb.addListener("listener", { port: 80 });
                // THEN
                test.throws(() => {
                    listener.addTargets("target", {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: "SideContainer",
                                containerPort: 8001,
                            })]
                    });
                }, /No container named 'SideContainer'. Did you call "addContainer()"?/);
                test.done();
            }
        },
        'allows load balancing to any container and port of service': {
            'with application load balancers': {
                'with default target group port and protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: lib_1.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                    const listener = lb.addListener("listener", { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: "MainContainer",
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: "lblistenertarget1Group1A1A5C9E"
                                }
                            }
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 80,
                        Protocol: "HTTP",
                    }));
                    test.done();
                },
                'with default target group port and HTTP protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: lib_1.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                    const listener = lb.addListener("listener", { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener, {
                            protocol: elbv2.ApplicationProtocol.HTTP
                        }),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: "MainContainer",
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: "lblistenertarget1Group1A1A5C9E"
                                }
                            }
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 80,
                        Protocol: "HTTP",
                    }));
                    test.done();
                },
                'with default target group port and HTTPS protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: lib_1.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                    const listener = lb.addListener("listener", { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener, {
                            protocol: elbv2.ApplicationProtocol.HTTPS
                        }),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: "MainContainer",
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: "lblistenertarget1Group1A1A5C9E"
                                }
                            }
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 443,
                        Protocol: "HTTPS",
                    }));
                    test.done();
                },
                'with any target group port and protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: lib_1.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, "lb", { vpc });
                    const listener = lb.addListener("listener", { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener, {
                            port: 83,
                            protocol: elbv2.ApplicationProtocol.HTTP
                        }),
                        newTargetGroupId: 'target1'
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: "MainContainer",
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: "lblistenertarget1Group1A1A5C9E"
                                }
                            }
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 83,
                        Protocol: "HTTP",
                    }));
                    test.done();
                },
            },
            'with network load balancers': {
                'with default target group port'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: lib_1.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition
                    });
                    // WHEN
                    const lb = new elbv2.NetworkLoadBalancer(stack, "lb", { vpc });
                    const listener = lb.addListener("listener", { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.networkListener(listener),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: "MainContainer",
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: "lblistenertarget1Group1A1A5C9E"
                                }
                            }
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 80,
                        Protocol: "TCP",
                    }));
                    test.done();
                },
                'with any target group port'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: lib_1.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition
                    });
                    // WHEN
                    const lb = new elbv2.NetworkLoadBalancer(stack, "lb", { vpc });
                    const listener = lb.addListener("listener", { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.networkListener(listener, {
                            port: 81
                        }),
                        newTargetGroupId: 'target1'
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: "MainContainer",
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: "lblistenertarget1Group1A1A5C9E"
                                }
                            }
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 81,
                        Protocol: "TCP",
                    }));
                    test.done();
                },
            }
        }
    },
    'allows scaling on a specified scheduled time'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: lib_1.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnSchedule("ScaleOnSchedule", {
            schedule: appscaling.Schedule.cron({ hour: '8', minute: '0' }),
            minCapacity: 10,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ScheduledActions: [
                {
                    ScalableTargetAction: {
                        MinCapacity: 10
                    },
                    Schedule: "cron(0 8 * * ? *)",
                    ScheduledActionName: "ScaleOnSchedule"
                }
            ]
        }));
        test.done();
    },
    'allows scaling on a specified metric value'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: lib_1.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnMetric("ScaleOnMetric", {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            scalingSteps: [
                { upper: 0, change: -1 },
                { lower: 100, change: +1 },
                { lower: 500, change: +5 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "StepScaling",
            ScalingTargetId: {
                Ref: "ServiceTaskCountTarget23E25614"
            },
            StepScalingPolicyConfiguration: {
                AdjustmentType: "ChangeInCapacity",
                MetricAggregationType: "Average",
                StepAdjustments: [
                    {
                        MetricIntervalUpperBound: 0,
                        ScalingAdjustment: -1
                    }
                ]
            }
        }));
        test.done();
    },
    'allows scaling on a target CPU utilization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: lib_1.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnCpuUtilization("ScaleOnCpu", {
            targetUtilizationPercent: 30
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "TargetTrackingScaling",
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: "ECSServiceAverageCPUUtilization" },
                TargetValue: 30
            }
        }));
        test.done();
    },
    'allows scaling on memory utilization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: lib_1.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnMemoryUtilization("ScaleOnMemory", {
            targetUtilizationPercent: 30
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "TargetTrackingScaling",
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: "ECSServiceAverageMemoryUtilization" },
                TargetValue: 30
            }
        }));
        test.done();
    },
    'allows scaling on custom CloudWatch metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: lib_1.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleToTrackCustomMetric("ScaleOnCustomMetric", {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            targetValue: 5
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: "TargetTrackingScaling",
            TargetTrackingScalingPolicyConfiguration: {
                CustomizedMetricSpecification: {
                    MetricName: "Metric",
                    Namespace: "Test",
                    Statistic: "Average"
                },
                TargetValue: 5
            }
        }));
        test.done();
    },
    'When enabling service discovery': {
        'throws if namespace has not been added to cluster'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: lib_1.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512
            });
            container.addPortMappings({ containerPort: 8000 });
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                    cloudMapOptions: {
                        name: 'myApp',
                    }
                });
            }, /Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster./);
            test.done();
        },
        'creates cloud map service for Private DNS namespace'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: lib_1.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE
            });
            new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp'
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: "A"
                        }
                    ],
                    NamespaceId: {
                        "Fn::GetAtt": [
                            "EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F",
                            "Id"
                        ]
                    },
                    RoutingPolicy: "MULTIVALUE"
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1
                },
                Name: "myApp",
                NamespaceId: {
                    'Fn::GetAtt': [
                        "EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F",
                        "Id"
                    ]
                }
            }));
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with SRV records'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: lib_1.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE
            });
            new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                    dnsRecordType: cloudmap.DnsRecordType.SRV
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: "SRV"
                        }
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id'
                        ]
                    },
                    RoutingPolicy: 'MULTIVALUE'
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1
                },
                Name: "myApp",
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id'
                    ]
                }
            }));
            test.done();
        },
    },
    'Metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        taskDefinition.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('hello')
        });
        // WHEN
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // THEN
        test.deepEqual(stack.resolve(service.metricCpuUtilization()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
                ServiceName: { 'Fn::GetAtt': ['ServiceD69D759B', 'Name'] }
            },
            namespace: 'AWS/ECS',
            metricName: 'CPUUtilization',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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