"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const ecs = require("../../lib");
module.exports = {
    "When creating an Fargate TaskDefinition": {
        "with only required properties set, it correctly sets default properties"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Family: "FargateTaskDef",
                NetworkMode: ecs.NetworkMode.AWS_VPC,
                RequiresCompatibilities: ["FARGATE"],
                Cpu: "256",
                Memory: "512",
            }));
            test.done();
        },
        "support lazy cpu and memory values"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.FargateTaskDefinition(stack, 'FargateTaskDef', {
                cpu: cdk.Lazy.numberValue({ produce: () => 128 }),
                memoryLimitMiB: cdk.Lazy.numberValue({ produce: () => 1024 })
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Cpu: "128",
                Memory: "1024"
            }));
            test.done();
        },
        "with all properties set"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef', {
                cpu: 128,
                executionRole: new iam.Role(stack, 'ExecutionRole', {
                    path: '/',
                    assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal("ecs.amazonaws.com"), new iam.ServicePrincipal("ecs-tasks.amazonaws.com"))
                }),
                family: "myApp",
                memoryLimitMiB: 1024,
                taskRole: new iam.Role(stack, 'TaskRole', {
                    assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                })
            });
            taskDefinition.addVolume({
                host: {
                    sourcePath: "/tmp/cache",
                },
                name: "scratch"
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
                Cpu: "128",
                ExecutionRoleArn: {
                    "Fn::GetAtt": [
                        "ExecutionRole605A040B",
                        "Arn"
                    ]
                },
                Family: "myApp",
                Memory: "1024",
                NetworkMode: "awsvpc",
                RequiresCompatibilities: [
                    ecs.LaunchType.FARGATE
                ],
                TaskRoleArn: {
                    "Fn::GetAtt": [
                        "TaskRole30FC0FBB",
                        "Arn"
                    ]
                },
                Volumes: [
                    {
                        Host: {
                            SourcePath: "/tmp/cache"
                        },
                        Name: "scratch"
                    }
                ]
            }));
            test.done();
        },
        'throws when adding placement constraint'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            test.throws(() => {
                taskDefinition.addPlacementConstraint(ecs.PlacementConstraint.memberOf("attribute:ecs.instance-type =~ t2.*"));
            }, /Cannot set placement constraints on tasks that run on Fargate/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5mYXJnYXRlLXRhc2stZGVmaW5pdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuZmFyZ2F0ZS10YXNrLWRlZmluaXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUEyRDtBQUMzRCx3Q0FBeUM7QUFDekMscUNBQXNDO0FBRXRDLGlDQUFrQztBQUVsQyxpQkFBUztJQUNQLHlDQUF5QyxFQUFFO1FBQ3pDLHlFQUF5RSxDQUFDLElBQVU7WUFDbEYsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBRXZELE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO2dCQUM1RCxNQUFNLEVBQUUsZ0JBQWdCO2dCQUN4QixXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2dCQUNwQyx1QkFBdUIsRUFBRSxDQUFDLFNBQVMsQ0FBQztnQkFDcEMsR0FBRyxFQUFFLEtBQUs7Z0JBQ1YsTUFBTSxFQUFFLEtBQUs7YUFDZCxDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxvQ0FBb0MsQ0FBQyxJQUFVO1lBQzdDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7Z0JBQ3JELEdBQUcsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFDLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxHQUFHLEVBQUMsQ0FBQztnQkFDL0MsY0FBYyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUMsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksRUFBQyxDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO2dCQUM1RCxHQUFHLEVBQUUsS0FBSztnQkFDVixNQUFNLEVBQUUsTUFBTTthQUNmLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHlCQUF5QixDQUFDLElBQVU7WUFDbEMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtnQkFDNUUsR0FBRyxFQUFFLEdBQUc7Z0JBQ1IsYUFBYSxFQUFFLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO29CQUNsRCxJQUFJLEVBQUUsR0FBRztvQkFDVCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQ25DLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDLEVBQzdDLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDLENBQ3BEO2lCQUNGLENBQUM7Z0JBQ0YsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsY0FBYyxFQUFFLElBQUk7Z0JBQ3BCLFFBQVEsRUFBRSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtvQkFDeEMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO2lCQUMvRCxDQUFDO2FBQ0gsQ0FBQyxDQUFDO1lBRUgsY0FBYyxDQUFDLFNBQVMsQ0FBQztnQkFDdkIsSUFBSSxFQUFFO29CQUNKLFVBQVUsRUFBRSxZQUFZO2lCQUN6QjtnQkFDRCxJQUFJLEVBQUUsU0FBUzthQUNoQixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDNUQsR0FBRyxFQUFFLEtBQUs7Z0JBQ1YsZ0JBQWdCLEVBQUU7b0JBQ2hCLFlBQVksRUFBRTt3QkFDWix1QkFBdUI7d0JBQ3ZCLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsTUFBTSxFQUFFLE1BQU07Z0JBQ2QsV0FBVyxFQUFFLFFBQVE7Z0JBQ3JCLHVCQUF1QixFQUFFO29CQUN2QixHQUFHLENBQUMsVUFBVSxDQUFDLE9BQU87aUJBQ3ZCO2dCQUNELFdBQVcsRUFBRTtvQkFDWCxZQUFZLEVBQUU7d0JBQ1osa0JBQWtCO3dCQUNsQixLQUFLO3FCQUNOO2lCQUNGO2dCQUNELE9BQU8sRUFBRTtvQkFDUDt3QkFDRSxJQUFJLEVBQUU7NEJBQ0osVUFBVSxFQUFFLFlBQVk7eUJBQ3pCO3dCQUNELElBQUksRUFBRSxTQUFTO3FCQUNoQjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHlDQUF5QyxDQUFDLElBQVU7WUFDbEQsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBRTlFLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixjQUFjLENBQUMsc0JBQXNCLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDLENBQUM7WUFDakgsQ0FBQyxFQUFFLCtEQUErRCxDQUFDLENBQUM7WUFFcEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgaWFtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWlhbScpO1xuaW1wb3J0IGNkayA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2NvcmUnKTtcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgZWNzID0gcmVxdWlyZSgnLi4vLi4vbGliJyk7XG5cbmV4cG9ydCA9IHtcbiAgXCJXaGVuIGNyZWF0aW5nIGFuIEZhcmdhdGUgVGFza0RlZmluaXRpb25cIjoge1xuICAgIFwid2l0aCBvbmx5IHJlcXVpcmVkIHByb3BlcnRpZXMgc2V0LCBpdCBjb3JyZWN0bHkgc2V0cyBkZWZhdWx0IHByb3BlcnRpZXNcIih0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ0ZhcmdhdGVUYXNrRGVmJyk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUNTOjpUYXNrRGVmaW5pdGlvblwiLCB7XG4gICAgICAgIEZhbWlseTogXCJGYXJnYXRlVGFza0RlZlwiLFxuICAgICAgICBOZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkFXU19WUEMsXG4gICAgICAgIFJlcXVpcmVzQ29tcGF0aWJpbGl0aWVzOiBbXCJGQVJHQVRFXCJdLFxuICAgICAgICBDcHU6IFwiMjU2XCIsXG4gICAgICAgIE1lbW9yeTogXCI1MTJcIixcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgIFwic3VwcG9ydCBsYXp5IGNwdSBhbmQgbWVtb3J5IHZhbHVlc1wiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICdGYXJnYXRlVGFza0RlZicsIHtcbiAgICAgICAgY3B1OiBjZGsuTGF6eS5udW1iZXJWYWx1ZSh7cHJvZHVjZTogKCkgPT4gMTI4fSksXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiBjZGsuTGF6eS5udW1iZXJWYWx1ZSh7cHJvZHVjZTogKCkgPT4gMTAyNH0pXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uXCIsIHtcbiAgICAgICAgQ3B1OiBcIjEyOFwiLFxuICAgICAgICBNZW1vcnk6IFwiMTAyNFwiXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICBcIndpdGggYWxsIHByb3BlcnRpZXMgc2V0XCIodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkZhcmdhdGVUYXNrRGVmaW5pdGlvbihzdGFjaywgJ0ZhcmdhdGVUYXNrRGVmJywge1xuICAgICAgICBjcHU6IDEyOCxcbiAgICAgICAgZXhlY3V0aW9uUm9sZTogbmV3IGlhbS5Sb2xlKHN0YWNrLCAnRXhlY3V0aW9uUm9sZScsIHtcbiAgICAgICAgICBwYXRoOiAnLycsXG4gICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLkNvbXBvc2l0ZVByaW5jaXBhbChcbiAgICAgICAgICAgIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbChcImVjcy5hbWF6b25hd3MuY29tXCIpLFxuICAgICAgICAgICAgbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKFwiZWNzLXRhc2tzLmFtYXpvbmF3cy5jb21cIilcbiAgICAgICAgICApXG4gICAgICAgIH0pLFxuICAgICAgICBmYW1pbHk6IFwibXlBcHBcIixcbiAgICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICAgIHRhc2tSb2xlOiBuZXcgaWFtLlJvbGUoc3RhY2ssICdUYXNrUm9sZScsIHtcbiAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWNzLXRhc2tzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgfSlcbiAgICAgIH0pO1xuXG4gICAgICB0YXNrRGVmaW5pdGlvbi5hZGRWb2x1bWUoe1xuICAgICAgICBob3N0OiB7XG4gICAgICAgICAgc291cmNlUGF0aDogXCIvdG1wL2NhY2hlXCIsXG4gICAgICAgIH0sXG4gICAgICAgIG5hbWU6IFwic2NyYXRjaFwiXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uXCIsIHtcbiAgICAgICAgQ3B1OiBcIjEyOFwiLFxuICAgICAgICBFeGVjdXRpb25Sb2xlQXJuOiB7XG4gICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgIFwiRXhlY3V0aW9uUm9sZTYwNUEwNDBCXCIsXG4gICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBGYW1pbHk6IFwibXlBcHBcIixcbiAgICAgICAgTWVtb3J5OiBcIjEwMjRcIixcbiAgICAgICAgTmV0d29ya01vZGU6IFwiYXdzdnBjXCIsXG4gICAgICAgIFJlcXVpcmVzQ29tcGF0aWJpbGl0aWVzOiBbXG4gICAgICAgICAgZWNzLkxhdW5jaFR5cGUuRkFSR0FURVxuICAgICAgICBdLFxuICAgICAgICBUYXNrUm9sZUFybjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIlRhc2tSb2xlMzBGQzBGQkJcIixcbiAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIFZvbHVtZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBIb3N0OiB7XG4gICAgICAgICAgICAgIFNvdXJjZVBhdGg6IFwiL3RtcC9jYWNoZVwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgTmFtZTogXCJzY3JhdGNoXCJcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd0aHJvd3Mgd2hlbiBhZGRpbmcgcGxhY2VtZW50IGNvbnN0cmFpbnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICdGYXJnYXRlVGFza0RlZicpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIHRhc2tEZWZpbml0aW9uLmFkZFBsYWNlbWVudENvbnN0cmFpbnQoZWNzLlBsYWNlbWVudENvbnN0cmFpbnQubWVtYmVyT2YoXCJhdHRyaWJ1dGU6ZWNzLmluc3RhbmNlLXR5cGUgPX4gdDIuKlwiKSk7XG4gICAgICB9LCAvQ2Fubm90IHNldCBwbGFjZW1lbnQgY29uc3RyYWludHMgb24gdGFza3MgdGhhdCBydW4gb24gRmFyZ2F0ZS8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH1cbn07XG4iXX0=