"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
module.exports = {
    "correctly sets all appMeshProxyConfiguration"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
            networkMode: ecs.NetworkMode.AWS_VPC,
            proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                containerName: "envoy",
                properties: {
                    ignoredUID: 1337,
                    ignoredGID: 1338,
                    appPorts: [80, 81],
                    proxyIngressPort: 80,
                    proxyEgressPort: 81,
                    egressIgnoredPorts: [8081],
                    egressIgnoredIPs: ["169.254.170.2", "169.254.169.254"],
                }
            })
        });
        taskDefinition.addContainer("web", {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")
        });
        taskDefinition.addContainer("envoy", {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry("envoyproxy/envoy")
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
            ProxyConfiguration: {
                ContainerName: "envoy",
                ProxyConfigurationProperties: [
                    {
                        Name: "IgnoredUID",
                        Value: "1337"
                    },
                    {
                        Name: "IgnoredGID",
                        Value: "1338"
                    },
                    {
                        Name: "AppPorts",
                        Value: "80,81"
                    },
                    {
                        Name: "ProxyIngressPort",
                        Value: "80"
                    },
                    {
                        Name: "ProxyEgressPort",
                        Value: "81"
                    },
                    {
                        Name: "EgressIgnoredPorts",
                        Value: "8081"
                    },
                    {
                        Name: "EgressIgnoredIPs",
                        Value: "169.254.170.2,169.254.169.254"
                    }
                ],
                Type: "APPMESH"
            }
        }));
        test.done();
    },
    "correctly sets appMeshProxyConfiguration with default properties set"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
            networkMode: ecs.NetworkMode.AWS_VPC,
            proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                containerName: "envoy",
                properties: {
                    ignoredUID: 1337,
                    appPorts: [80, 81],
                    proxyIngressPort: 80,
                    proxyEgressPort: 81
                }
            })
        });
        taskDefinition.addContainer("web", {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")
        });
        taskDefinition.addContainer("envoy", {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry("envoyproxy/envoy")
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
            ProxyConfiguration: {
                ContainerName: "envoy",
                ProxyConfigurationProperties: [
                    {
                        Name: "IgnoredUID",
                        Value: "1337"
                    },
                    {
                        Name: "AppPorts",
                        Value: "80,81"
                    },
                    {
                        Name: "ProxyIngressPort",
                        Value: "80"
                    },
                    {
                        Name: "ProxyEgressPort",
                        Value: "81"
                    }
                ],
                Type: "APPMESH"
            }
        }));
        test.done();
    },
    "correctly sets appMeshProxyConfiguration with empty egressIgnoredPorts and egressIgnoredIPs"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
            networkMode: ecs.NetworkMode.AWS_VPC,
            proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                containerName: "envoy",
                properties: {
                    ignoredUID: 1337,
                    appPorts: [80, 81],
                    proxyIngressPort: 80,
                    proxyEgressPort: 81,
                    egressIgnoredIPs: [],
                    egressIgnoredPorts: []
                }
            })
        });
        taskDefinition.addContainer("web", {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry("amazon/amazon-ecs-sample")
        });
        taskDefinition.addContainer("envoy", {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry("envoyproxy/envoy")
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike("AWS::ECS::TaskDefinition", {
            ProxyConfiguration: {
                ContainerName: "envoy",
                ProxyConfigurationProperties: [
                    {
                        Name: "IgnoredUID",
                        Value: "1337"
                    },
                    {
                        Name: "AppPorts",
                        Value: "80,81"
                    },
                    {
                        Name: "ProxyIngressPort",
                        Value: "80"
                    },
                    {
                        Name: "ProxyEgressPort",
                        Value: "81"
                    }
                ],
                Type: "APPMESH"
            }
        }));
        test.done();
    },
    "throws when neither of IgnoredUID and IgnoredGID is set"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => {
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
                proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                    containerName: "envoy",
                    properties: {
                        appPorts: [80, 81],
                        proxyIngressPort: 80,
                        proxyEgressPort: 81
                    }
                })
            });
        }, /At least one of ignoredUID or ignoredGID should be specified./);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5hcHAtbWVzaC1wcm94eS1jb25maWd1cmF0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5hcHAtbWVzaC1wcm94eS1jb25maWd1cmF0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBMkQ7QUFDM0QscUNBQXNDO0FBRXRDLDhCQUErQjtBQUUvQixpQkFBUztJQUNQLDhDQUE4QyxDQUFDLElBQVU7UUFDdkQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE9BQU87WUFDcEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLHlCQUF5QixDQUFDO2dCQUNwRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsVUFBVSxFQUFFO29CQUNWLFVBQVUsRUFBRSxJQUFJO29CQUNoQixVQUFVLEVBQUUsSUFBSTtvQkFDaEIsUUFBUSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQztvQkFDbEIsZ0JBQWdCLEVBQUUsRUFBRTtvQkFDcEIsZUFBZSxFQUFFLEVBQUU7b0JBQ25CLGtCQUFrQixFQUFFLENBQUMsSUFBSSxDQUFDO29CQUMxQixnQkFBZ0IsRUFBRSxDQUFDLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQztpQkFDdkQ7YUFDRixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUU7WUFDakMsY0FBYyxFQUFFLElBQUk7WUFDcEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLDBCQUEwQixDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO1lBQ25DLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQztTQUMzRCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLDRCQUE0QixFQUFFO29CQUM1Qjt3QkFDRSxJQUFJLEVBQUUsWUFBWTt3QkFDbEIsS0FBSyxFQUFFLE1BQU07cUJBQ2Q7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLFlBQVk7d0JBQ2xCLEtBQUssRUFBRSxNQUFNO3FCQUNkO29CQUNEO3dCQUNFLElBQUksRUFBRSxVQUFVO3dCQUNoQixLQUFLLEVBQUUsT0FBTztxQkFDZjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsa0JBQWtCO3dCQUN4QixLQUFLLEVBQUUsSUFBSTtxQkFDWjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsaUJBQWlCO3dCQUN2QixLQUFLLEVBQUUsSUFBSTtxQkFDWjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsb0JBQW9CO3dCQUMxQixLQUFLLEVBQUUsTUFBTTtxQkFDZDtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsa0JBQWtCO3dCQUN4QixLQUFLLEVBQUUsK0JBQStCO3FCQUN2QztpQkFDRjtnQkFDRCxJQUFJLEVBQUUsU0FBUzthQUNoQjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNFQUFzRSxDQUFDLElBQVU7UUFDL0UsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE9BQU87WUFDcEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLHlCQUF5QixDQUFDO2dCQUNwRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsVUFBVSxFQUFFO29CQUNWLFVBQVUsRUFBRSxJQUFJO29CQUNoQixRQUFRLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDO29CQUNsQixnQkFBZ0IsRUFBRSxFQUFFO29CQUNwQixlQUFlLEVBQUUsRUFBRTtpQkFDcEI7YUFDRixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUU7WUFDakMsY0FBYyxFQUFFLElBQUk7WUFDcEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLDBCQUEwQixDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO1lBQ25DLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQztTQUMzRCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLDRCQUE0QixFQUFFO29CQUM1Qjt3QkFDRSxJQUFJLEVBQUUsWUFBWTt3QkFDbEIsS0FBSyxFQUFFLE1BQU07cUJBQ2Q7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLFVBQVU7d0JBQ2hCLEtBQUssRUFBRSxPQUFPO3FCQUNmO29CQUNEO3dCQUNFLElBQUksRUFBRSxrQkFBa0I7d0JBQ3hCLEtBQUssRUFBRSxJQUFJO3FCQUNaO29CQUNEO3dCQUNFLElBQUksRUFBRSxpQkFBaUI7d0JBQ3ZCLEtBQUssRUFBRSxJQUFJO3FCQUNaO2lCQUNGO2dCQUNELElBQUksRUFBRSxTQUFTO2FBQ2hCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkZBQTZGLENBQUMsSUFBVTtRQUN0RyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsT0FBTztZQUNwQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMseUJBQXlCLENBQUM7Z0JBQ3BFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixVQUFVLEVBQUU7b0JBQ1YsVUFBVSxFQUFFLElBQUk7b0JBQ2hCLFFBQVEsRUFBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUM7b0JBQ2xCLGdCQUFnQixFQUFFLEVBQUU7b0JBQ3BCLGVBQWUsRUFBRSxFQUFFO29CQUNuQixnQkFBZ0IsRUFBRSxFQUFFO29CQUNwQixrQkFBa0IsRUFBRSxFQUFFO2lCQUN2QjthQUNGLENBQUM7U0FDSCxDQUFDLENBQUM7UUFDSCxjQUFjLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRTtZQUNqQyxjQUFjLEVBQUUsSUFBSTtZQUNwQixLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsMEJBQTBCLENBQUM7U0FDbkUsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUU7WUFDbkMsY0FBYyxFQUFFLElBQUk7WUFDcEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLGtCQUFrQixDQUFDO1NBQzNELENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO1lBQzVELGtCQUFrQixFQUFFO2dCQUNsQixhQUFhLEVBQUUsT0FBTztnQkFDdEIsNEJBQTRCLEVBQUU7b0JBQzVCO3dCQUNFLElBQUksRUFBRSxZQUFZO3dCQUNsQixLQUFLLEVBQUUsTUFBTTtxQkFDZDtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsVUFBVTt3QkFDaEIsS0FBSyxFQUFFLE9BQU87cUJBQ2Y7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLGtCQUFrQjt3QkFDeEIsS0FBSyxFQUFFLElBQUk7cUJBQ1o7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLGlCQUFpQjt3QkFDdkIsS0FBSyxFQUFFLElBQUk7cUJBQ1o7aUJBQ0Y7Z0JBQ0QsSUFBSSxFQUFFLFNBQVM7YUFDaEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5REFBeUQsQ0FBQyxJQUFVO1FBQ2xFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO2dCQUM3QyxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2dCQUNwQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMseUJBQXlCLENBQUM7b0JBQ3BFLGFBQWEsRUFBRSxPQUFPO29CQUN0QixVQUFVLEVBQUU7d0JBQ1YsUUFBUSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQzt3QkFDbEIsZ0JBQWdCLEVBQUUsRUFBRTt3QkFDcEIsZUFBZSxFQUFFLEVBQUU7cUJBQ3BCO2lCQUNGLENBQUM7YUFDSCxDQUFDLENBQUM7UUFDTCxDQUFDLEVBQUUsK0RBQStELENBQUMsQ0FBQztRQUVwRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY29yZScpO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCBlY3MgPSByZXF1aXJlKCcuLi9saWInKTtcblxuZXhwb3J0ID0ge1xuICBcImNvcnJlY3RseSBzZXRzIGFsbCBhcHBNZXNoUHJveHlDb25maWd1cmF0aW9uXCIodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnRWMyVGFza0RlZicsIHtcbiAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgIHByb3h5Q29uZmlndXJhdGlvbjogZWNzLlByb3h5Q29uZmlndXJhdGlvbnMuYXBwTWVzaFByb3h5Q29uZmlndXJhdGlvbih7XG4gICAgICAgIGNvbnRhaW5lck5hbWU6IFwiZW52b3lcIixcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGlnbm9yZWRVSUQ6IDEzMzcsXG4gICAgICAgICAgaWdub3JlZEdJRDogMTMzOCxcbiAgICAgICAgICBhcHBQb3J0czogWzgwLCA4MV0sXG4gICAgICAgICAgcHJveHlJbmdyZXNzUG9ydDogODAsXG4gICAgICAgICAgcHJveHlFZ3Jlc3NQb3J0OiA4MSxcbiAgICAgICAgICBlZ3Jlc3NJZ25vcmVkUG9ydHM6IFs4MDgxXSxcbiAgICAgICAgICBlZ3Jlc3NJZ25vcmVkSVBzOiBbXCIxNjkuMjU0LjE3MC4yXCIsIFwiMTY5LjI1NC4xNjkuMjU0XCJdLFxuICAgICAgICB9XG4gICAgICB9KVxuICAgIH0pO1xuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcihcIndlYlwiLCB7XG4gICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KFwiYW1hem9uL2FtYXpvbi1lY3Mtc2FtcGxlXCIpXG4gICAgfSk7XG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKFwiZW52b3lcIiwge1xuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeShcImVudm95cHJveHkvZW52b3lcIilcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDUzo6VGFza0RlZmluaXRpb25cIiwge1xuICAgICAgUHJveHlDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIENvbnRhaW5lck5hbWU6IFwiZW52b3lcIixcbiAgICAgICAgUHJveHlDb25maWd1cmF0aW9uUHJvcGVydGllczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6IFwiSWdub3JlZFVJRFwiLFxuICAgICAgICAgICAgVmFsdWU6IFwiMTMzN1wiXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIklnbm9yZWRHSURcIixcbiAgICAgICAgICAgIFZhbHVlOiBcIjEzMzhcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogXCJBcHBQb3J0c1wiLFxuICAgICAgICAgICAgVmFsdWU6IFwiODAsODFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogXCJQcm94eUluZ3Jlc3NQb3J0XCIsXG4gICAgICAgICAgICBWYWx1ZTogXCI4MFwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIlByb3h5RWdyZXNzUG9ydFwiLFxuICAgICAgICAgICAgVmFsdWU6IFwiODFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogXCJFZ3Jlc3NJZ25vcmVkUG9ydHNcIixcbiAgICAgICAgICAgIFZhbHVlOiBcIjgwODFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogXCJFZ3Jlc3NJZ25vcmVkSVBzXCIsXG4gICAgICAgICAgICBWYWx1ZTogXCIxNjkuMjU0LjE3MC4yLDE2OS4yNTQuMTY5LjI1NFwiXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBUeXBlOiBcIkFQUE1FU0hcIlxuICAgICAgfVxuICAgIH0pKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICBcImNvcnJlY3RseSBzZXRzIGFwcE1lc2hQcm94eUNvbmZpZ3VyYXRpb24gd2l0aCBkZWZhdWx0IHByb3BlcnRpZXMgc2V0XCIodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnRWMyVGFza0RlZicsIHtcbiAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgIHByb3h5Q29uZmlndXJhdGlvbjogZWNzLlByb3h5Q29uZmlndXJhdGlvbnMuYXBwTWVzaFByb3h5Q29uZmlndXJhdGlvbih7XG4gICAgICAgIGNvbnRhaW5lck5hbWU6IFwiZW52b3lcIixcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGlnbm9yZWRVSUQ6IDEzMzcsXG4gICAgICAgICAgYXBwUG9ydHM6IFs4MCwgODFdLFxuICAgICAgICAgIHByb3h5SW5ncmVzc1BvcnQ6IDgwLFxuICAgICAgICAgIHByb3h5RWdyZXNzUG9ydDogODFcbiAgICAgICAgfVxuICAgICAgfSlcbiAgICB9KTtcbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoXCJ3ZWJcIiwge1xuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeShcImFtYXpvbi9hbWF6b24tZWNzLXNhbXBsZVwiKVxuICAgIH0pO1xuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcihcImVudm95XCIsIHtcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoXCJlbnZveXByb3h5L2Vudm95XCIpXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uXCIsIHtcbiAgICAgIFByb3h5Q29uZmlndXJhdGlvbjoge1xuICAgICAgICBDb250YWluZXJOYW1lOiBcImVudm95XCIsXG4gICAgICAgIFByb3h5Q29uZmlndXJhdGlvblByb3BlcnRpZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIklnbm9yZWRVSURcIixcbiAgICAgICAgICAgIFZhbHVlOiBcIjEzMzdcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogXCJBcHBQb3J0c1wiLFxuICAgICAgICAgICAgVmFsdWU6IFwiODAsODFcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogXCJQcm94eUluZ3Jlc3NQb3J0XCIsXG4gICAgICAgICAgICBWYWx1ZTogXCI4MFwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIlByb3h5RWdyZXNzUG9ydFwiLFxuICAgICAgICAgICAgVmFsdWU6IFwiODFcIlxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVHlwZTogXCJBUFBNRVNIXCJcbiAgICAgIH1cbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgXCJjb3JyZWN0bHkgc2V0cyBhcHBNZXNoUHJveHlDb25maWd1cmF0aW9uIHdpdGggZW1wdHkgZWdyZXNzSWdub3JlZFBvcnRzIGFuZCBlZ3Jlc3NJZ25vcmVkSVBzXCIodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnRWMyVGFza0RlZicsIHtcbiAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgIHByb3h5Q29uZmlndXJhdGlvbjogZWNzLlByb3h5Q29uZmlndXJhdGlvbnMuYXBwTWVzaFByb3h5Q29uZmlndXJhdGlvbih7XG4gICAgICAgIGNvbnRhaW5lck5hbWU6IFwiZW52b3lcIixcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGlnbm9yZWRVSUQ6IDEzMzcsXG4gICAgICAgICAgYXBwUG9ydHM6IFs4MCwgODFdLFxuICAgICAgICAgIHByb3h5SW5ncmVzc1BvcnQ6IDgwLFxuICAgICAgICAgIHByb3h5RWdyZXNzUG9ydDogODEsXG4gICAgICAgICAgZWdyZXNzSWdub3JlZElQczogW10sXG4gICAgICAgICAgZWdyZXNzSWdub3JlZFBvcnRzOiBbXVxuICAgICAgICB9XG4gICAgICB9KVxuICAgIH0pO1xuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcihcIndlYlwiLCB7XG4gICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KFwiYW1hem9uL2FtYXpvbi1lY3Mtc2FtcGxlXCIpXG4gICAgfSk7XG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKFwiZW52b3lcIiwge1xuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeShcImVudm95cHJveHkvZW52b3lcIilcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDUzo6VGFza0RlZmluaXRpb25cIiwge1xuICAgICAgUHJveHlDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIENvbnRhaW5lck5hbWU6IFwiZW52b3lcIixcbiAgICAgICAgUHJveHlDb25maWd1cmF0aW9uUHJvcGVydGllczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6IFwiSWdub3JlZFVJRFwiLFxuICAgICAgICAgICAgVmFsdWU6IFwiMTMzN1wiXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIkFwcFBvcnRzXCIsXG4gICAgICAgICAgICBWYWx1ZTogXCI4MCw4MVwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIlByb3h5SW5ncmVzc1BvcnRcIixcbiAgICAgICAgICAgIFZhbHVlOiBcIjgwXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6IFwiUHJveHlFZ3Jlc3NQb3J0XCIsXG4gICAgICAgICAgICBWYWx1ZTogXCI4MVwiXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBUeXBlOiBcIkFQUE1FU0hcIlxuICAgICAgfVxuICAgIH0pKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICBcInRocm93cyB3aGVuIG5laXRoZXIgb2YgSWdub3JlZFVJRCBhbmQgSWdub3JlZEdJRCBpcyBzZXRcIih0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnRWMyVGFza0RlZicsIHtcbiAgICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5BV1NfVlBDLFxuICAgICAgICBwcm94eUNvbmZpZ3VyYXRpb246IGVjcy5Qcm94eUNvbmZpZ3VyYXRpb25zLmFwcE1lc2hQcm94eUNvbmZpZ3VyYXRpb24oe1xuICAgICAgICAgIGNvbnRhaW5lck5hbWU6IFwiZW52b3lcIixcbiAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICBhcHBQb3J0czogWzgwLCA4MV0sXG4gICAgICAgICAgICBwcm94eUluZ3Jlc3NQb3J0OiA4MCxcbiAgICAgICAgICAgIHByb3h5RWdyZXNzUG9ydDogODFcbiAgICAgICAgICB9XG4gICAgICAgIH0pXG4gICAgICB9KTtcbiAgICB9LCAvQXQgbGVhc3Qgb25lIG9mIGlnbm9yZWRVSUQgb3IgaWdub3JlZEdJRCBzaG91bGQgYmUgc3BlY2lmaWVkLi8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH1cbn07Il19