"use strict";
const assert_1 = require("@aws-cdk/assert");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
module.exports = {
    "When creating a Task Definition": {
        "add a container using default props"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            new ecs.ContainerDefinition(stack, "Container", {
                image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                taskDefinition,
                memoryLimitMiB: 2048,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                        Image: "/aws/aws-example-app",
                        Memory: 2048,
                        Name: "Container"
                    }
                ]
            }));
            test.done();
        },
        "add a container using all props"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const secret = new secretsmanager.Secret(stack, 'Secret');
            new ecs.ContainerDefinition(stack, "Container", {
                image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                taskDefinition,
                memoryLimitMiB: 1024,
                memoryReservationMiB: 512,
                command: ["CMD-SHELL"],
                cpu: 128,
                disableNetworking: true,
                dnsSearchDomains: ['example.com'],
                dnsServers: ['host.com'],
                dockerLabels: {
                    key: 'fooLabel',
                    value: 'barLabel'
                },
                dockerSecurityOptions: ['ECS_SELINUX_CAPABLE=true'],
                entryPoint: ["top", "-b"],
                environment: {
                    key: "foo",
                    value: "bar"
                },
                essential: true,
                extraHosts: {
                    name: 'dev-db.hostname.pvt'
                },
                gpuCount: 256,
                hostname: "host.example.com",
                privileged: true,
                readonlyRootFilesystem: true,
                startTimeout: cdk.Duration.millis(2000),
                stopTimeout: cdk.Duration.millis(5000),
                user: "rootUser",
                workingDirectory: "a/b/c",
                healthCheck: {
                    command: ["curl localhost:8000"]
                },
                linuxParameters: new ecs.LinuxParameters(stack, 'LinuxParameters'),
                logging: new ecs.AwsLogDriver({ streamPrefix: 'prefix' }),
                secrets: {
                    SECRET: ecs.Secret.fromSecretsManager(secret),
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Command: [
                            "CMD-SHELL"
                        ],
                        Cpu: 128,
                        DisableNetworking: true,
                        DnsSearchDomains: [
                            "example.com"
                        ],
                        DnsServers: [
                            "host.com"
                        ],
                        DockerLabels: {
                            key: "fooLabel",
                            value: "barLabel"
                        },
                        DockerSecurityOptions: [
                            "ECS_SELINUX_CAPABLE=true"
                        ],
                        EntryPoint: [
                            "top",
                            "-b"
                        ],
                        Environment: [
                            {
                                Name: "key",
                                Value: "foo"
                            },
                            {
                                Name: "value",
                                Value: "bar"
                            }
                        ],
                        Essential: true,
                        ExtraHosts: [
                            {
                                Hostname: "name",
                                IpAddress: "dev-db.hostname.pvt"
                            }
                        ],
                        HealthCheck: {
                            Command: [
                                "CMD-SHELL",
                                "curl localhost:8000"
                            ],
                            Interval: 30,
                            Retries: 3,
                            Timeout: 5
                        },
                        Hostname: "host.example.com",
                        Image: "/aws/aws-example-app",
                        LinuxParameters: {
                            Capabilities: {}
                        },
                        LogConfiguration: {
                            LogDriver: "awslogs",
                            Options: {
                                "awslogs-group": {
                                    Ref: "ContainerLogGroupE6FD74A4"
                                },
                                "awslogs-stream-prefix": "prefix",
                                "awslogs-region": {
                                    Ref: "AWS::Region"
                                }
                            }
                        },
                        Memory: 1024,
                        MemoryReservation: 512,
                        Name: "Container",
                        Privileged: true,
                        ReadonlyRootFilesystem: true,
                        ResourceRequirements: [
                            {
                                Type: "GPU",
                                Value: "256"
                            }
                        ],
                        Secrets: [
                            {
                                Name: "SECRET",
                                ValueFrom: {
                                    Ref: "SecretA720EF05"
                                }
                            }
                        ],
                        StartTimeout: 2,
                        StopTimeout: 5,
                        User: "rootUser",
                        WorkingDirectory: "a/b/c"
                    }
                ]
            }));
            test.done();
        },
        "throws when MemoryLimit is less than MemoryReservationLimit"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // THEN
            test.throws(() => {
                new ecs.ContainerDefinition(stack, "Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    taskDefinition,
                    memoryLimitMiB: 512,
                    memoryReservationMiB: 1024,
                });
            }, /MemoryLimitMiB should not be less than MemoryReservationMiB./);
            test.done();
        },
        "With network mode AwsVpc": {
            "throws when Host port is different from container port"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // THEN
                test.throws(() => {
                    container.addPortMappings({
                        containerPort: 8080,
                        hostPort: 8081
                    });
                });
                test.done();
            },
            "Host port is the same as container port"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8080
                });
                // THEN no exception raised
                test.done();
            },
            "Host port can be empty "(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                // THEN no exception raised
                test.done();
            },
        },
        "With network mode Host ": {
            "throws when Host port is different from container port"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // THEN
                test.throws(() => {
                    container.addPortMappings({
                        containerPort: 8080,
                        hostPort: 8081
                    });
                });
                test.done();
            },
            "when host port is the same as container port"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8080
                });
                // THEN no exception raised
                test.done();
            },
            "Host port can be empty "(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                // THEN no exception raised
                test.done();
            },
            "errors when adding links"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                const logger = taskDefinition.addContainer("LoggingContainer", {
                    image: ecs.ContainerImage.fromRegistry("myLogger"),
                    memoryLimitMiB: 1024,
                });
                // THEN
                test.throws(() => {
                    container.addLink(logger);
                });
                test.done();
            },
        },
        "With network mode Bridge": {
            "when Host port is empty "(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                });
                // THEN no exception raises
                test.done();
            },
            "when Host port is not empty "(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8084
                });
                // THEN no exception raises
                test.done();
            },
            "allows adding links"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                const logger = taskDefinition.addContainer("LoggingContainer", {
                    image: ecs.ContainerImage.fromRegistry("myLogger"),
                    memoryLimitMiB: 1024,
                });
                // THEN
                container.addLink(logger);
                test.done();
            },
        },
        "With network mode NAT": {
            "produces undefined CF networkMode property"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                new ecs.TaskDefinition(stack, 'TD', {
                    compatibility: ecs.Compatibility.EC2,
                    networkMode: ecs.NetworkMode.NAT
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', (props, inspection) => {
                    if (props.NetworkMode === undefined) {
                        return true;
                    }
                    inspection.failureReason = 'CF template should not have NetworkMode defined for a task definition that relies on NAT network mode.';
                    return false;
                }));
                test.done();
            }
        }
    },
    "Container Port": {
        "should return the first container port in PortMappings"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            const container = taskDefinition.addContainer("Container", {
                image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                memoryLimitMiB: 2048,
            });
            // WHEN
            container.addPortMappings({
                containerPort: 8080,
            });
            container.addPortMappings({
                containerPort: 8081,
            });
            const actual = container.containerPort;
            // THEN
            const expected = 8080;
            test.equal(actual, expected, "containerPort should return the first container port in PortMappings");
            test.done();
        },
        "throws when calling containerPort with no PortMappings"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            const container = taskDefinition.addContainer("MyContainer", {
                image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                memoryLimitMiB: 2048
            });
            // THEN
            test.throws(() => {
                const actual = container.containerPort;
                const expected = 8080;
                test.equal(actual, expected);
            }, /Container MyContainer hasn't defined any ports. Call addPortMappings()./);
            test.done();
        },
    },
    "Ingress Port": {
        "With network mode AwsVpc": {
            "Ingress port should be the same as container port"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 8080;
                test.equal(actual, expected, "Ingress port should be the same as container port");
                test.done();
            },
            "throws when calling ingressPort with no PortMappings"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer("MyContainer", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048
                });
                // THEN
                test.throws(() => {
                    const actual = container.ingressPort;
                    const expected = 8080;
                    test.equal(actual, expected);
                }, /Container MyContainer hasn't defined any ports. Call addPortMappings()./);
                test.done();
            },
        },
        "With network mode Host ": {
            "Ingress port should be the same as container port"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 8080;
                test.equal(actual, expected);
                test.done();
            }
        },
        "With network mode Bridge": {
            "Ingress port should be the same as host port if supplied"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8081,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 8081;
                test.equal(actual, expected);
                test.done();
            },
            "Ingress port should be 0 if not supplied"(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer("Container", {
                    image: ecs.ContainerImage.fromRegistry("/aws/aws-example-app"),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8081,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 0;
                test.equal(actual, expected);
                test.done();
            }
        },
    },
    'can add environment variables to the container definition'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            environment: {
                TEST_ENVIRONMENT_VARIABLE: "test environment variable value"
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [{
                            Name: "TEST_ENVIRONMENT_VARIABLE",
                            Value: "test environment variable value"
                        }]
                }
            ]
        }));
        test.done();
    },
    'Given GPU count parameter': {
        'will add resource requirements to container definition'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                gpuCount: 4,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        ResourceRequirements: [
                            {
                                Type: "GPU",
                                Value: "4"
                            }
                        ]
                    }
                ]
            }));
            test.done();
        },
    },
    'can add secret environment variables to the container definition'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const parameter = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'Parameter', {
            parameterName: '/name',
            version: 1
        });
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            secrets: {
                SECRET: ecs.Secret.fromSecretsManager(secret),
                PARAMETER: ecs.Secret.fromSsmParameter(parameter),
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Secrets: [
                        {
                            Name: "SECRET",
                            ValueFrom: {
                                Ref: "SecretA720EF05"
                            }
                        },
                        {
                            Name: "PARAMETER",
                            ValueFrom: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":ssm:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":parameter/name"
                                    ]
                                ]
                            }
                        },
                    ]
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'secretsmanager:GetSecretValue',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'SecretA720EF05'
                        }
                    },
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory'
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition'
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region'
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId'
                                    },
                                    ':parameter/name'
                                ]
                            ]
                        }
                    }
                ],
                Version: '2012-10-17'
            }
        }));
        test.done();
    },
    'can add AWS logging to container definition'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            logging: new ecs.AwsLogDriver({ streamPrefix: 'prefix' })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: "awslogs",
                        Options: {
                            "awslogs-group": { Ref: "TaskDefcontLogGroup4E10DCBF" },
                            "awslogs-stream-prefix": "prefix",
                            "awslogs-region": { Ref: "AWS::Region" }
                        }
                    },
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: ["logs:CreateLogStream", "logs:PutLogEvents"],
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["TaskDefcontLogGroup4E10DCBF", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
    'can set Health Check with defaults'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = "curl localhost:8000";
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: [hcCommand]
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ["CMD-SHELL", hcCommand],
                        Interval: 30,
                        Retries: 3,
                        Timeout: 5
                    },
                }
            ]
        }));
        test.done();
    },
    'throws when setting Health Check with no commands'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: []
            }
        });
        // THEN
        test.throws(() => {
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        HealthCheck: {
                            Command: [],
                            Interval: 30,
                            Retries: 3,
                            Timeout: 5
                        },
                    }
                ]
            }));
        }, /At least one argument must be supplied for health check command./);
        test.done();
    },
    'can specify Health Check values in shell form'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = "curl localhost:8000";
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: [hcCommand],
                interval: cdk.Duration.seconds(20),
                retries: 5,
                startPeriod: cdk.Duration.seconds(10)
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ["CMD-SHELL", hcCommand],
                        Interval: 20,
                        Retries: 5,
                        Timeout: 5,
                        StartPeriod: 10
                    },
                }
            ]
        }));
        test.done();
    },
    'can specify Health Check values in array form starting with CMD-SHELL'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = "curl localhost:8000";
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: ["CMD-SHELL", hcCommand],
                interval: cdk.Duration.seconds(20),
                retries: 5,
                startPeriod: cdk.Duration.seconds(10)
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ["CMD-SHELL", hcCommand],
                        Interval: 20,
                        Retries: 5,
                        Timeout: 5,
                        StartPeriod: 10
                    },
                }
            ]
        }));
        test.done();
    },
    'can specify Health Check values in array form starting with CMD'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = "curl localhost:8000";
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: ["CMD", hcCommand],
                interval: cdk.Duration.seconds(20),
                retries: 5,
                startPeriod: cdk.Duration.seconds(10)
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ["CMD", hcCommand],
                        Interval: 20,
                        Retries: 5,
                        Timeout: 5,
                        StartPeriod: 10
                    },
                }
            ]
        }));
        test.done();
    },
    'can specify private registry credentials'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const mySecretArn = 'arn:aws:secretsmanager:region:1234567890:secret:MyRepoSecret-6f8hj3';
        const repoCreds = secretsmanager.Secret.fromSecretArn(stack, 'MyRepoSecret', mySecretArn);
        // WHEN
        taskDefinition.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('user-x/my-app', {
                credentials: repoCreds
            }),
            memoryLimitMiB: 2048,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Image: 'user-x/my-app',
                    RepositoryCredentials: {
                        CredentialsParameter: mySecretArn
                    },
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "secretsmanager:GetSecretValue",
                        Effect: "Allow",
                        Resource: mySecretArn
                    }
                ]
            }
        }));
        test.done();
    },
    '_linkContainer works properly': {
        'when the props passed in is an essential container'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // WHEN
            const container = taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                essential: true
            });
            // THEN
            test.equal(taskDefinition.defaultContainer, container);
            test.done();
        },
        'when the props passed in is not an essential container'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                essential: false
            });
            // THEN
            test.equal(taskDefinition.defaultContainer, undefined);
            test.done();
        }
    },
    'Can specify linux parameters': {
        'with only required properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters');
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Capabilities: {},
                        }
                    }
                ]
            }));
            test.done();
        },
        'before calling addContainer'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            linuxParameters.addCapabilities(ecs.Capability.ALL);
            linuxParameters.dropCapabilities(ecs.Capability.KILL);
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Capabilities: {
                                Add: ["ALL"],
                                Drop: ["KILL"]
                            },
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    }
                ]
            }));
            test.done();
        },
        'after calling addContainer'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            linuxParameters.addCapabilities(ecs.Capability.ALL);
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // Mutate linuxParameter after added to a container
            linuxParameters.dropCapabilities(ecs.Capability.SETUID);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Capabilities: {
                                Add: ["ALL"],
                                Drop: ["SETUID"]
                            },
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    }
                ]
            }));
            test.done();
        },
        "with one or more host devices"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            // WHEN
            linuxParameters.addDevices({
                hostPath: "a/b/c",
            });
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Devices: [
                                {
                                    HostPath: "a/b/c"
                                }
                            ],
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    }
                ]
            }));
            test.done();
        },
        "with the tmpfs mount for a container"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            // WHEN
            linuxParameters.addTmpfs({
                containerPath: "a/b/c",
                size: 1024
            });
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Tmpfs: [
                                {
                                    ContainerPath: "a/b/c",
                                    Size: 1024
                                }
                            ],
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    }
                ]
            }));
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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