"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
module.exports = {
    "When creating an ECS Cluster": {
        "with no properties set, it correctly sets default properties"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            cluster.addCapacity('DefaultAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro')
            });
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::Cluster"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", {
                CidrBlock: '10.0.0.0/16',
                EnableDnsHostnames: true,
                EnableDnsSupport: true,
                InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
                Tags: [
                    {
                        Key: "Name",
                        Value: "EcsCluster/Vpc"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
                ImageId: {
                    Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
                },
                InstanceType: "t2.micro",
                IamInstanceProfile: {
                    Ref: "EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3"
                },
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231",
                            "GroupId"
                        ]
                    }
                ],
                UserData: {
                    "Fn::Base64": {
                        "Fn::Join": [
                            "",
                            [
                                "#!/bin/bash\necho ECS_CLUSTER=",
                                {
                                    Ref: "EcsCluster97242B84"
                                },
                                // tslint:disable-next-line:max-line-length
                                " >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config"
                            ]
                        ]
                    }
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::AutoScalingGroup", {
                MaxSize: "1",
                MinSize: "1",
                DesiredCapacity: "1",
                LaunchConfigurationName: {
                    Ref: "EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1"
                },
                Tags: [
                    {
                        Key: "Name",
                        PropagateAtLaunch: true,
                        Value: "EcsCluster/DefaultAutoScalingGroup"
                    }
                ],
                VPCZoneIdentifier: [
                    {
                        Ref: "EcsClusterVpcPrivateSubnet1SubnetFAB0E487"
                    },
                    {
                        Ref: "EcsClusterVpcPrivateSubnet2SubnetC2B7B1BA"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroup", {
                GroupDescription: "EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup",
                SecurityGroupEgress: [
                    {
                        CidrIp: "0.0.0.0/0",
                        Description: "Allow all outbound traffic by default",
                        IpProtocol: "-1"
                    }
                ],
                Tags: [
                    {
                        Key: "Name",
                        Value: "EcsCluster/DefaultAutoScalingGroup"
                    }
                ],
                VpcId: {
                    Ref: "EcsClusterVpc779914AB"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Role", {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: { "Fn::Join": ["", ["ec2.", { Ref: "AWS::URLSuffix" }]] }
                            }
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Policy", {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "ecs:CreateCluster",
                                "ecs:DeregisterContainerInstance",
                                "ecs:DiscoverPollEndpoint",
                                "ecs:Poll",
                                "ecs:RegisterContainerInstance",
                                "ecs:StartTelemetrySession",
                                "ecs:Submit*",
                                "ecr:GetAuthorizationToken",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
        "with only vpc set, it correctly sets default properties"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', {
                vpc,
            });
            cluster.addCapacity('DefaultAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro')
            });
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::Cluster"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", {
                CidrBlock: '10.0.0.0/16',
                EnableDnsHostnames: true,
                EnableDnsSupport: true,
                InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
                Tags: [
                    {
                        Key: "Name",
                        Value: "MyVpc"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
                ImageId: {
                    Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
                },
                InstanceType: "t2.micro",
                IamInstanceProfile: {
                    Ref: "EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3"
                },
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231",
                            "GroupId"
                        ]
                    }
                ],
                UserData: {
                    "Fn::Base64": {
                        "Fn::Join": [
                            "",
                            [
                                "#!/bin/bash\necho ECS_CLUSTER=",
                                {
                                    Ref: "EcsCluster97242B84"
                                },
                                // tslint:disable-next-line:max-line-length
                                " >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config"
                            ]
                        ]
                    }
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::AutoScalingGroup", {
                MaxSize: "1",
                MinSize: "1",
                DesiredCapacity: "1",
                LaunchConfigurationName: {
                    Ref: "EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1"
                },
                Tags: [
                    {
                        Key: "Name",
                        PropagateAtLaunch: true,
                        Value: "EcsCluster/DefaultAutoScalingGroup"
                    }
                ],
                VPCZoneIdentifier: [
                    {
                        Ref: "MyVpcPrivateSubnet1Subnet5057CF7E"
                    },
                    {
                        Ref: "MyVpcPrivateSubnet2Subnet0040C983"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroup", {
                GroupDescription: "EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup",
                SecurityGroupEgress: [
                    {
                        CidrIp: "0.0.0.0/0",
                        Description: "Allow all outbound traffic by default",
                        IpProtocol: "-1"
                    }
                ],
                Tags: [
                    {
                        Key: "Name",
                        Value: "EcsCluster/DefaultAutoScalingGroup"
                    }
                ],
                VpcId: {
                    Ref: "MyVpcF9F0CA6F"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Role", {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: { "Fn::Join": ["", ["ec2.", { Ref: "AWS::URLSuffix" }]] }
                            }
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Policy", {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "ecs:CreateCluster",
                                "ecs:DeregisterContainerInstance",
                                "ecs:DiscoverPollEndpoint",
                                "ecs:Poll",
                                "ecs:RegisterContainerInstance",
                                "ecs:StartTelemetrySession",
                                "ecs:Submit*",
                                "ecr:GetAuthorizationToken",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
        "multiple clusters with default capacity"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            // WHEN
            for (let i = 0; i < 2; i++) {
                const cluster = new ecs.Cluster(stack, `EcsCluster${i}`, { vpc, });
                cluster.addCapacity('MyCapacity', {
                    instanceType: new ec2.InstanceType('m3.medium'),
                });
            }
            test.done();
        },
        'lifecycle hook is automatically added'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', {
                vpc,
            });
            // WHEN
            cluster.addCapacity('DefaultAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro')
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LifecycleHook', {
                AutoScalingGroupName: { Ref: "EcsClusterDefaultAutoScalingGroupASGC1A785DB" },
                LifecycleTransition: "autoscaling:EC2_INSTANCE_TERMINATING",
                DefaultResult: "CONTINUE",
                HeartbeatTimeout: 300,
                NotificationTargetARN: { Ref: "EcsClusterDefaultAutoScalingGroupLifecycleHookDrainHookTopicACD2D4A4" },
                RoleARN: { "Fn::GetAtt": ["EcsClusterDefaultAutoScalingGroupLifecycleHookDrainHookRoleA38EC83B", "Arn"] }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Timeout: 310,
                Environment: {
                    Variables: {
                        CLUSTER: {
                            Ref: "EcsCluster97242B84"
                        }
                    }
                },
                Handler: "index.lambda_handler"
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "ec2:DescribeInstances",
                                "ec2:DescribeInstanceAttribute",
                                "ec2:DescribeInstanceStatus",
                                "ec2:DescribeHosts"
                            ],
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: "autoscaling:CompleteLifecycleAction",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":autoscaling:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":autoScalingGroup:*:autoScalingGroupName/",
                                        {
                                            Ref: "EcsClusterDefaultAutoScalingGroupASGC1A785DB"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "ecs:DescribeContainerInstances",
                                "ecs:DescribeTasks"
                            ],
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: [
                                "ecs:ListContainerInstances",
                                "ecs:SubmitContainerStateChange",
                                "ecs:SubmitTaskStateChange"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    "EcsCluster97242B84",
                                    "Arn"
                                ]
                            }
                        },
                        {
                            Action: [
                                "ecs:UpdateContainerInstancesState",
                                "ecs:ListTasks"
                            ],
                            Condition: {
                                ArnEquals: {
                                    "ecs:cluster": {
                                        "Fn::GetAtt": [
                                            "EcsCluster97242B84",
                                            "Arn"
                                        ]
                                    }
                                }
                            },
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "EcsClusterDefaultAutoScalingGroupDrainECSHookFunctionServiceRoleDefaultPolicyA45BF396",
                Roles: [
                    {
                        Ref: "EcsClusterDefaultAutoScalingGroupDrainECSHookFunctionServiceRole94543EDA"
                    }
                ]
            }));
            test.done();
        },
        "with capacity and cloudmap namespace properties set"(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            new ecs.Cluster(stack, 'EcsCluster', {
                vpc,
                capacity: {
                    instanceType: new ec2.InstanceType('t2.micro')
                },
                defaultCloudMapNamespace: {
                    name: "foo.com"
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ServiceDiscovery::PrivateDnsNamespace", {
                Name: 'foo.com',
                Vpc: {
                    Ref: 'MyVpcF9F0CA6F'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::Cluster"));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", {
                CidrBlock: '10.0.0.0/16',
                EnableDnsHostnames: true,
                EnableDnsSupport: true,
                InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
                Tags: [
                    {
                        Key: "Name",
                        Value: "MyVpc"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
                ImageId: {
                    Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
                },
                InstanceType: "t2.micro",
                IamInstanceProfile: {
                    Ref: "EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3"
                },
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231",
                            "GroupId"
                        ]
                    }
                ],
                UserData: {
                    "Fn::Base64": {
                        "Fn::Join": [
                            "",
                            [
                                "#!/bin/bash\necho ECS_CLUSTER=",
                                {
                                    Ref: "EcsCluster97242B84"
                                },
                                // tslint:disable-next-line:max-line-length
                                " >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config"
                            ]
                        ]
                    }
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::AutoScalingGroup", {
                MaxSize: "1",
                MinSize: "1",
                DesiredCapacity: "1",
                LaunchConfigurationName: {
                    Ref: "EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1"
                },
                Tags: [
                    {
                        Key: "Name",
                        PropagateAtLaunch: true,
                        Value: "EcsCluster/DefaultAutoScalingGroup"
                    }
                ],
                VPCZoneIdentifier: [
                    {
                        Ref: "MyVpcPrivateSubnet1Subnet5057CF7E"
                    },
                    {
                        Ref: "MyVpcPrivateSubnet2Subnet0040C983"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroup", {
                GroupDescription: "EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup",
                SecurityGroupEgress: [
                    {
                        CidrIp: "0.0.0.0/0",
                        Description: "Allow all outbound traffic by default",
                        IpProtocol: "-1"
                    }
                ],
                Tags: [
                    {
                        Key: "Name",
                        Value: "EcsCluster/DefaultAutoScalingGroup"
                    }
                ],
                VpcId: {
                    Ref: "MyVpcF9F0CA6F"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Role", {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: { "Fn::Join": ["", ["ec2.", { Ref: "AWS::URLSuffix" }]] }
                            }
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Policy", {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "ecs:CreateCluster",
                                "ecs:DeregisterContainerInstance",
                                "ecs:DiscoverPollEndpoint",
                                "ecs:Poll",
                                "ecs:RegisterContainerInstance",
                                "ecs:StartTelemetrySession",
                                "ecs:Submit*",
                                "ecr:GetAuthorizationToken",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
    },
    "allows specifying instance type"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new aws_ec2_1.InstanceType("m3.large")
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            InstanceType: "m3.large"
        }));
        test.done();
    },
    "allows specifying cluster size"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            desiredCapacity: 3
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::AutoScalingGroup", {
            MaxSize: "3"
        }));
        test.done();
    },
    /*
     * TODO:v2.0.0 BEGINNING OF OBSOLETE BLOCK
     */
    "allows specifying special HW AMI Type"(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('GpuAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: new ecs.EcsOptimizedAmi({
                hardwareType: ecs.AmiHardwareType.GPU
            }),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            ImageId: {
                Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
            }
        }));
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: "AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>",
                Default: "/aws/service/ecs/optimized-ami/amazon-linux-2/gpu/recommended/image_id"
            }
        });
        test.done();
    },
    "errors if amazon linux given with special HW type"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.throws(() => {
            cluster.addCapacity('GpuAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
                machineImage: new ecs.EcsOptimizedAmi({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX,
                    hardwareType: ecs.AmiHardwareType.GPU,
                }),
            });
        }, /Amazon Linux does not support special hardware type/);
        test.done();
    },
    "allows specifying windows image"(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('WindowsAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: new ecs.EcsOptimizedAmi({
                windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
            }),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiwindowsserver2019englishfullrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: "AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>",
                Default: "/aws/service/ecs/optimized-ami/windows_server/2019/english/full/recommended/image_id"
            }
        });
        test.done();
    },
    "errors if windows given with special HW type"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.throws(() => {
            cluster.addCapacity('WindowsGpuAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
                machineImage: new ecs.EcsOptimizedAmi({
                    windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
                    hardwareType: ecs.AmiHardwareType.GPU,
                }),
            });
        }, /Windows Server does not support special hardware type/);
        test.done();
    },
    "errors if windowsVersion and linux generation are set"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.throws(() => {
            cluster.addCapacity('WindowsScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
                machineImage: new ecs.EcsOptimizedAmi({
                    windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX
                }),
            });
        }, /"windowsVersion" and Linux image "generation" cannot be both set/);
        test.done();
    },
    "allows returning the correct image for windows for EcsOptimizedAmi"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const ami = new ecs.EcsOptimizedAmi({
            windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
        });
        test.equal(ami.getImage(stack).osType, ec2.OperatingSystemType.WINDOWS);
        test.done();
    },
    "allows returning the correct image for linux for EcsOptimizedAmi"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const ami = new ecs.EcsOptimizedAmi({
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX
        });
        test.equal(ami.getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    "allows returning the correct image for linux 2 for EcsOptimizedAmi"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const ami = new ecs.EcsOptimizedAmi({
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2
        });
        test.equal(ami.getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    "allows returning the correct image for linux for EcsOptimizedImage"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.equal(ecs.EcsOptimizedImage.amazonLinux().getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    "allows returning the correct image for linux 2 for EcsOptimizedImage"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.equal(ecs.EcsOptimizedImage.amazonLinux2().getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    "allows returning the correct image for windows for EcsOptimizedImage"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.equal(ecs.EcsOptimizedImage.windows(ecs.WindowsOptimizedVersion.SERVER_2019).getImage(stack).osType, ec2.OperatingSystemType.WINDOWS);
        test.done();
    },
    /*
     * TODO:v2.0.0 END OF OBSOLETE BLOCK
     */
    "allows specifying special HW AMI Type v2"(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('GpuAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.GPU)
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            ImageId: {
                Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
            }
        }));
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: "AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>",
                Default: "/aws/service/ecs/optimized-ami/amazon-linux-2/gpu/recommended/image_id"
            }
        });
        test.done();
    },
    "allows specifying Amazon Linux v1 AMI"(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('GpuAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: ecs.EcsOptimizedImage.amazonLinux()
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            ImageId: {
                Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinuxrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
            }
        }));
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiamazonlinuxrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: "AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>",
                Default: "/aws/service/ecs/optimized-ami/amazon-linux/recommended/image_id"
            }
        });
        test.done();
    },
    "allows specifying windows image v2"(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('WindowsAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: ecs.EcsOptimizedImage.windows(ecs.WindowsOptimizedVersion.SERVER_2019),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiwindowsserver2019englishfullrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: "AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>",
                Default: "/aws/service/ecs/optimized-ami/windows_server/2019/english/full/recommended/image_id"
            }
        });
        test.done();
    },
    "allows specifying spot fleet"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            spotPrice: "0.31"
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            SpotPrice: "0.31"
        }));
        test.done();
    },
    "allows specifying drain time"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            taskDrainTime: cdk.Duration.minutes(1)
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LifecycleHook", {
            HeartbeatTimeout: 60
        }));
        test.done();
    },
    "allows specifying automated spot draining"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('c5.xlarge'),
            spotPrice: '0.0735',
            spotInstanceDraining: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            UserData: {
                "Fn::Base64": {
                    "Fn::Join": [
                        "",
                        [
                            "#!/bin/bash\necho ECS_CLUSTER=",
                            {
                                Ref: "EcsCluster97242B84"
                            },
                            " >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config\necho ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config"
                        ]
                    ]
                }
            }
        }));
        test.done();
    },
    "allows containers access to instance metadata service"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            canContainersAccessInstanceRole: true
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            UserData: {
                "Fn::Base64": {
                    "Fn::Join": [
                        "",
                        [
                            "#!/bin/bash\necho ECS_CLUSTER=",
                            {
                                Ref: "EcsCluster97242B84"
                            },
                            " >> /etc/ecs/ecs.config"
                        ]
                    ]
                }
            }
        }));
        test.done();
    },
    "allows adding default service discovery namespace"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
        });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: "foo.com"
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::ServiceDiscovery::PrivateDnsNamespace", {
            Name: 'foo.com',
            Vpc: {
                Ref: 'MyVpcF9F0CA6F'
            }
        }));
        test.done();
    },
    "allows adding public service discovery namespace"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
        });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: "foo.com",
            type: cloudmap.NamespaceType.DNS_PUBLIC
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource("AWS::ServiceDiscovery::PublicDnsNamespace", {
            Name: 'foo.com',
        }));
        test.equal(cluster.defaultCloudMapNamespace.type, cloudmap.NamespaceType.DNS_PUBLIC);
        test.done();
    },
    "throws if default service discovery namespace added more than once"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
        });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: "foo.com"
        });
        // THEN
        test.throws(() => {
            cluster.addDefaultCloudMapNamespace({
                name: "foo.com"
            });
        }, /Can only add default namespace once./);
        test.done();
    },
    'export/import of a cluster with a namespace'(test) {
        // GIVEN
        const stack1 = new cdk.Stack();
        const vpc1 = new ec2.Vpc(stack1, 'Vpc');
        const cluster1 = new ecs.Cluster(stack1, 'Cluster', { vpc: vpc1 });
        cluster1.addDefaultCloudMapNamespace({
            name: 'hello.com',
        });
        const stack2 = new cdk.Stack();
        // WHEN
        const cluster2 = ecs.Cluster.fromClusterAttributes(stack2, 'Cluster', {
            vpc: vpc1,
            securityGroups: cluster1.connections.securityGroups,
            defaultCloudMapNamespace: cloudmap.PrivateDnsNamespace.fromPrivateDnsNamespaceAttributes(stack2, 'ns', {
                namespaceId: 'import-namespace-id',
                namespaceArn: 'import-namespace-arn',
                namespaceName: 'import-namespace-name',
            }),
            clusterName: 'cluster-name',
        });
        // THEN
        test.equal(cluster2.defaultCloudMapNamespace.type, cloudmap.NamespaceType.DNS_PRIVATE);
        test.deepEqual(stack2.resolve(cluster2.defaultCloudMapNamespace.namespaceId), 'import-namespace-id');
        // Can retrieve subnets from VPC - will throw 'There are no 'Private' subnets in this VPC. Use a different VPC subnet selection.' if broken.
        cluster2.vpc.selectSubnets();
        test.done();
    },
    'imported cluster with imported security groups honors allowAllOutbound'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const importedSg1 = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-1', { allowAllOutbound: false });
        const importedSg2 = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-2');
        const cluster = ecs.Cluster.fromClusterAttributes(stack, 'Cluster', {
            clusterName: 'cluster-name',
            securityGroups: [importedSg1, importedSg2],
            vpc,
        });
        // WHEN
        cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-1'
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroupEgress', 1));
        test.done();
    },
    "Metric"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.deepEqual(stack.resolve(cluster.metricCpuReservation()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
            },
            namespace: 'AWS/ECS',
            metricName: 'CPUReservation',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.deepEqual(stack.resolve(cluster.metricMemoryReservation()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
            },
            namespace: 'AWS/ECS',
            metricName: 'MemoryReservation',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.deepEqual(stack.resolve(cluster.metric("myMetric")), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
            },
            namespace: 'AWS/ECS',
            metricName: 'myMetric',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    },
    "ASG with a public VPC without NAT Gateways"(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyPublicVpc', {
            natGateways: 0,
            subnetConfiguration: [
                { cidrMask: 24, name: "ingress", subnetType: ec2.SubnetType.PUBLIC }
            ]
        });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // WHEN
        cluster.addCapacity("DefaultAutoScalingGroup", {
            instanceType: new ec2.InstanceType('t2.micro'),
            associatePublicIpAddress: true,
            vpcSubnets: {
                onePerAz: true,
                subnetType: ec2.SubnetType.PUBLIC
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource("AWS::ECS::Cluster"));
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::VPC", {
            CidrBlock: '10.0.0.0/16',
            EnableDnsHostnames: true,
            EnableDnsSupport: true,
            InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
            Tags: [
                {
                    Key: "Name",
                    Value: "MyPublicVpc"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::LaunchConfiguration", {
            ImageId: {
                Ref: "SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter"
            },
            InstanceType: "t2.micro",
            AssociatePublicIpAddress: true,
            IamInstanceProfile: {
                Ref: "EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3"
            },
            SecurityGroups: [
                {
                    "Fn::GetAtt": [
                        "EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231",
                        "GroupId"
                    ]
                }
            ],
            UserData: {
                "Fn::Base64": {
                    "Fn::Join": [
                        "",
                        [
                            "#!/bin/bash\necho ECS_CLUSTER=",
                            {
                                Ref: "EcsCluster97242B84"
                            },
                            // tslint:disable-next-line:max-line-length
                            " >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config"
                        ]
                    ]
                }
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource("AWS::AutoScaling::AutoScalingGroup", {
            MaxSize: "1",
            MinSize: "1",
            DesiredCapacity: "1",
            LaunchConfigurationName: {
                Ref: "EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1"
            },
            Tags: [
                {
                    Key: "Name",
                    PropagateAtLaunch: true,
                    Value: "EcsCluster/DefaultAutoScalingGroup"
                }
            ],
            VPCZoneIdentifier: [
                {
                    Ref: "MyPublicVpcingressSubnet1Subnet9191044C"
                },
                {
                    Ref: "MyPublicVpcingressSubnet2SubnetD2F2E034"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroup", {
            GroupDescription: "EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup",
            SecurityGroupEgress: [
                {
                    CidrIp: "0.0.0.0/0",
                    Description: "Allow all outbound traffic by default",
                    IpProtocol: "-1"
                }
            ],
            Tags: [
                {
                    Key: "Name",
                    Value: "EcsCluster/DefaultAutoScalingGroup"
                }
            ],
            VpcId: {
                Ref: "MyPublicVpcA2BF6CDA"
            }
        }));
        // THEN
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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