# ***************************************************************
# Copyright (c) 2021 Jittor. All Rights Reserved. 
# Maintainers: 
#     Wenyang Zhou <576825820@qq.com>
#     Dun Liang <randonlang@gmail.com>. 
# 
# This file is subject to the terms and conditions defined in
# file 'LICENSE.txt', which is part of this source code package.
# ***************************************************************
# This model is generated by pytorch converter.
import jittor as jt
from jittor import nn

__all__ = ['GoogLeNet', 'googlenet']

def googlenet(pretrained=False, **kwargs):
    model = GoogLeNet(**kwargs)
    if pretrained: model.load("jittorhub://googlenet.pkl")
    return model

class GoogLeNet(nn.Module):
    """ GoogLeNet model architecture.

    Args:

    * num_classes: Number of classes. Default: 1000.
    * aux_logits: If True, add an auxiliary branch that can improve training. Default: True
    * init_weights: Defualt: True.
    * blocks: List of three blocks, [conv_block, inception_block, inception_aux_block]. If None, will use [BasicConv2d, Inception, InceptionAux] instead. Default: None.
    """

    def __init__(self, num_classes=1000, aux_logits=True, init_weights=True, blocks=None):
        super(GoogLeNet, self).__init__()
        if (blocks is None):
            blocks = [BasicConv2d, Inception, InceptionAux]
        assert (len(blocks) == 3)
        conv_block = blocks[0]
        inception_block = blocks[1]
        inception_aux_block = blocks[2]
        self.aux_logits = aux_logits
        self.conv1 = conv_block(3, 64, kernel_size=7, stride=2, padding=3)
        self.maxpool1 = nn.Pool(3, stride=2, ceil_mode=True, op='maximum')
        self.conv2 = conv_block(64, 64, kernel_size=1)
        self.conv3 = conv_block(64, 192, kernel_size=3, padding=1)
        self.maxpool2 = nn.Pool(3, stride=2, ceil_mode=True, op='maximum')
        self.inception3a = inception_block(192, 64, 96, 128, 16, 32, 32)
        self.inception3b = inception_block(256, 128, 128, 192, 32, 96, 64)
        self.maxpool3 = nn.Pool(3, stride=2, ceil_mode=True, op='maximum')
        self.inception4a = inception_block(480, 192, 96, 208, 16, 48, 64)
        self.inception4b = inception_block(512, 160, 112, 224, 24, 64, 64)
        self.inception4c = inception_block(512, 128, 128, 256, 24, 64, 64)
        self.inception4d = inception_block(512, 112, 144, 288, 32, 64, 64)
        self.inception4e = inception_block(528, 256, 160, 320, 32, 128, 128)
        self.maxpool4 = nn.Pool(2, stride=2, ceil_mode=True, op='maximum')
        self.inception5a = inception_block(832, 256, 160, 320, 32, 128, 128)
        self.inception5b = inception_block(832, 384, 192, 384, 48, 128, 128)
        if aux_logits:
            self.aux1 = inception_aux_block(512, num_classes)
            self.aux2 = inception_aux_block(528, num_classes)
        else:
            self.aux1 = None
            self.aux2 = None
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.dropout = nn.Dropout(0.2)
        self.fc = nn.Linear(1024, num_classes)

    def _forward(self, x):
        x = self.conv1(x)
        x = self.maxpool1(x)
        x = self.conv2(x)
        x = self.conv3(x)
        x = self.maxpool2(x)
        x = self.inception3a(x)
        x = self.inception3b(x)
        x = self.maxpool3(x)
        x = self.inception4a(x)
        if (self.aux1 is not None):
            aux1 = self.aux1(x)
        x = self.inception4b(x)
        x = self.inception4c(x)
        x = self.inception4d(x)
        if (self.aux2 is not None):
            aux2 = self.aux2(x)
        x = self.inception4e(x)
        x = self.maxpool4(x)
        x = self.inception5a(x)
        x = self.inception5b(x)
        x = self.avgpool(x)

        x = jt.reshape(x, (x.shape[0], (- 1)))
        x = self.dropout(x)
        x = self.fc(x)
        return (x, aux2, aux1)

    def eager_outputs(self, x, aux2, aux1):
        return x

    def execute(self, x):
        (x, aux1, aux2) = self._forward(x)
        aux_defined = (self.aux_logits)
        return self.eager_outputs(x, aux2, aux1)

class Inception(nn.Module):

    def __init__(self, in_channels, ch1x1, ch3x3red, ch3x3, ch5x5red, ch5x5, pool_proj, conv_block=None):
        super(Inception, self).__init__()
        if (conv_block is None):
            conv_block = BasicConv2d
        self.branch1 = conv_block(in_channels, ch1x1, kernel_size=1)
        self.branch2 = nn.Sequential(conv_block(in_channels, ch3x3red, kernel_size=1), conv_block(ch3x3red, ch3x3, kernel_size=3, padding=1))
        self.branch3 = nn.Sequential(conv_block(in_channels, ch5x5red, kernel_size=1), conv_block(ch5x5red, ch5x5, kernel_size=3, padding=1))
        self.branch4 = nn.Sequential(nn.Pool(kernel_size=3, stride=1, padding=1, ceil_mode=True, op='maximum'), conv_block(in_channels, pool_proj, kernel_size=1))

    def _forward(self, x):
        branch1 = self.branch1(x)
        branch2 = self.branch2(x)
        branch3 = self.branch3(x)
        branch4 = self.branch4(x)
        outputs = [branch1, branch2, branch3, branch4]
        return outputs

    def execute(self, x):
        outputs = self._forward(x)
        return jt.contrib.concat(outputs, dim=1)

class InceptionAux(nn.Module):

    def __init__(self, in_channels, num_classes, conv_block=None):
        super(InceptionAux, self).__init__()
        if (conv_block is None):
            conv_block = BasicConv2d
        self.conv = conv_block(in_channels, 128, kernel_size=1)
        self.fc1 = nn.Linear(2048, 1024)
        self.fc2 = nn.Linear(1024, num_classes)

    def execute(self, x):
        x = nn.AdaptiveAvgPool2d(4)(x)
        x = self.conv(x)
        x = jt.reshape(x, (x.shape[0], (- 1)))
        x = nn.relu(self.fc1(x))
        x = nn.Dropout(0.7)(x)
        x = self.fc2(x)
        return x

class BasicConv2d(nn.Module):

    def __init__(self, in_channels, out_channels, **kwargs):
        super(BasicConv2d, self).__init__()
        self.conv = nn.Conv(in_channels, out_channels, bias=False, **kwargs)
        self.bn = nn.BatchNorm(out_channels, eps=0.001)

    def execute(self, x):
        x = self.conv(x)
        x = self.bn(x)
        return nn.relu(x)
