# CDK Feature Flags

[CDK Feature Flags](https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html) are a mechanism that allows the CDK to evolve and change the behavior of certain classes and methods, without causing disruption to existing deployed infrastructure.

Feature flags are [context values](https://docs.aws.amazon.com/cdk/v2/guide/context.html) and can be configured using any of the context management methods, at any level of the construct tree. Commonly, they are specified in the `cdk.json` file.
`cdk init` will create new CDK projects with a `cdk.json` file containing all recommended feature flags enabled.

## Current list of feature flags

Flags come in three types:

- **Default change**: The default behavior of an API has been changed in order to improve its ergonomics. The old behavior can still be achieved, but requires source changes.
- **Fix/deprecation**: The old behavior was incorrect or not recommended for new users. The only way to keep it is to not set this flag.
- **Config**: Configurable behavior that we recommend you turn on.

<!-- BEGIN table -->

| Flag | Summary | Since | Type |
| ----- | ----- | ----- | ----- |
| [@aws-cdk/core:newStyleStackSynthesis](#aws-cdkcorenewstylestacksynthesis) | Switch to new stack synthesis method which enables CI/CD | 2.0.0 | (fix) |
| [@aws-cdk/core:stackRelativeExports](#aws-cdkcorestackrelativeexports) | Name exports based on the construct paths relative to the stack, rather than the global construct path | 2.0.0 | (fix) |
| [@aws-cdk/aws-rds:lowercaseDbIdentifier](#aws-cdkaws-rdslowercasedbidentifier) | Force lowercasing of RDS Cluster names in CDK | 2.0.0 | (fix) |
| [@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId](#aws-cdkaws-apigatewayusageplankeyorderinsensitiveid) | Allow adding/removing multiple UsagePlanKeys independently | 2.0.0 | (fix) |
| [@aws-cdk/aws-lambda:recognizeVersionProps](#aws-cdkaws-lambdarecognizeversionprops) | Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the  `fn.currentVersion`. | 2.0.0 | (fix) |
| [@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2\_2021](#aws-cdkaws-cloudfrontdefaultsecuritypolicytlsv12_2021) | Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default. | 2.0.0 | (fix) |
| [@aws-cdk/core:target-partitions](#aws-cdkcoretarget-partitions) | What regions to include in lookup tables of environment agnostic stacks | 2.4.0 | (config) |
| [@aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver](#aws-cdk-containersecs-service-extensionsenabledefaultlogdriver) | ECS extensions will automatically add an `awslogs` driver if no logging is specified | 2.8.0 | (default) |
| [@aws-cdk/aws-ec2:uniqueImdsv2TemplateName](#aws-cdkaws-ec2uniqueimdsv2templatename) | Enable this feature flag to have Launch Templates generated by the `InstanceRequireImdsv2Aspect` use unique names. | 2.8.0 | (fix) |
| [@aws-cdk/aws-iam:minimizePolicies](#aws-cdkaws-iamminimizepolicies) | Minimize IAM policies by combining Statements | 2.18.0 | (config) |
| [@aws-cdk/core:checkSecretUsage](#aws-cdkcorechecksecretusage) | Enable this flag to make it impossible to accidentally use SecretValues in unsafe locations | 2.21.0 | (config) |
| [@aws-cdk/aws-lambda:recognizeLayerVersion](#aws-cdkaws-lambdarecognizelayerversion) | Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the `fn.currentVersion`. | 2.27.0 | (fix) |
| [@aws-cdk/core:validateSnapshotRemovalPolicy](#aws-cdkcorevalidatesnapshotremovalpolicy) | Error on snapshot removal policies on resources that do not support it. | 2.28.0 | (default) |
| [@aws-cdk/aws-codepipeline:crossAccountKeyAliasStackSafeResourceName](#aws-cdkaws-codepipelinecrossaccountkeyaliasstacksaferesourcename) | Generate key aliases that include the stack name | 2.29.0 | (fix) |
| [@aws-cdk/aws-s3:createDefaultLoggingPolicy](#aws-cdkaws-s3createdefaultloggingpolicy) | Enable this feature flag to create an S3 bucket policy by default in cases where an AWS service would automatically create the Policy if one does not exist. | 2.31.0 | (fix) |
| [@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption](#aws-cdkaws-sns-subscriptionsrestrictsqsdescryption) | Restrict KMS key policy for encrypted Queues a bit more | 2.32.0 | (fix) |
| [@aws-cdk/aws-ecs:arnFormatIncludesClusterName](#aws-cdkaws-ecsarnformatincludesclustername) | ARN format used by ECS. In the new ARN format, the cluster name is part of the resource ID. | 2.35.0 | (fix) |
| [@aws-cdk/aws-apigateway:disableCloudWatchRole](#aws-cdkaws-apigatewaydisablecloudwatchrole) | Make default CloudWatch Role behavior safe for multiple API Gateways in one environment | 2.38.0 | (fix) |
| [@aws-cdk/core:enablePartitionLiterals](#aws-cdkcoreenablepartitionliterals) | Make ARNs concrete if AWS partition is known | 2.38.0 | (fix) |
| [@aws-cdk/aws-ecs:disableExplicitDeploymentControllerForCircuitBreaker](#aws-cdkaws-ecsdisableexplicitdeploymentcontrollerforcircuitbreaker) | Avoid setting the "ECS" deployment controller when adding a circuit breaker | 2.51.0 | (fix) |
| [@aws-cdk/aws-events:eventsTargetQueueSameAccount](#aws-cdkaws-eventseventstargetqueuesameaccount) | Event Rules may only push to encrypted SQS queues in the same account | 2.51.0 | (fix) |
| [@aws-cdk/aws-iam:standardizedServicePrincipals](#aws-cdkaws-iamstandardizedserviceprincipals) | Use standardized (global) service principals everywhere | 2.51.0 | (fix) |
| [@aws-cdk/aws-iam:importedRoleStackSafeDefaultPolicyName](#aws-cdkaws-iamimportedrolestacksafedefaultpolicyname) | Enable this feature to by default create default policy names for imported roles that depend on the stack the role is in. | 2.60.0 | (fix) |
| [@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy](#aws-cdkaws-s3serveraccesslogsusebucketpolicy) | Use S3 Bucket Policy instead of ACLs for Server Access Logging | 2.60.0 | (fix) |
| [@aws-cdk/customresources:installLatestAwsSdkDefault](#aws-cdkcustomresourcesinstalllatestawssdkdefault) | Whether to install the latest SDK by default in AwsCustomResource | 2.60.0 | (default) |
| [@aws-cdk/aws-route53-patters:useCertificate](#aws-cdkaws-route53-pattersusecertificate) | Use the official `Certificate` resource instead of `DnsValidatedCertificate` | 2.61.0 | (default) |
| [@aws-cdk/aws-codedeploy:removeAlarmsFromDeploymentGroup](#aws-cdkaws-codedeployremovealarmsfromdeploymentgroup) | Remove CloudWatch alarms from deployment group | 2.65.0 | (fix) |
| [@aws-cdk/aws-rds:databaseProxyUniqueResourceName](#aws-cdkaws-rdsdatabaseproxyuniqueresourcename) | Use unique resource name for Database Proxy | 2.65.0 | (fix) |
| [@aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId](#aws-cdkaws-apigatewayauthorizerchangedeploymentlogicalid) | Include authorizer configuration in the calculation of the API deployment logical ID. | 2.66.0 | (fix) |
| [@aws-cdk/aws-ec2:launchTemplateDefaultUserData](#aws-cdkaws-ec2launchtemplatedefaultuserdata) | Define user data for a launch template by default when a machine image is provided. | 2.67.0 | (fix) |
| [@aws-cdk/aws-secretsmanager:useAttachedSecretResourcePolicyForSecretTargetAttachments](#aws-cdkaws-secretsmanageruseattachedsecretresourcepolicyforsecrettargetattachments) | SecretTargetAttachments uses the ResourcePolicy of the attached Secret. | 2.67.0 | (fix) |
| [@aws-cdk/aws-redshift:columnId](#aws-cdkaws-redshiftcolumnid) | Whether to use an ID to track Redshift column changes | 2.68.0 | (fix) |
| [@aws-cdk/aws-stepfunctions-tasks:enableEmrServicePolicyV2](#aws-cdkaws-stepfunctions-tasksenableemrservicepolicyv2) | Enable AmazonEMRServicePolicy_v2 managed policies | 2.72.0 | (fix) |
| [@aws-cdk/aws-apigateway:requestValidatorUniqueId](#aws-cdkaws-apigatewayrequestvalidatoruniqueid) | Generate a unique id for each RequestValidator added to a method | 2.78.0 | (fix) |
| [@aws-cdk/aws-ec2:restrictDefaultSecurityGroup](#aws-cdkaws-ec2restrictdefaultsecuritygroup) | Restrict access to the VPC default security group | 2.78.0 | (default) |
| [@aws-cdk/aws-kms:aliasNameRef](#aws-cdkaws-kmsaliasnameref) | KMS Alias name and keyArn will have implicit reference to KMS Key | 2.83.0 | (fix) |
| [@aws-cdk/core:includePrefixInUniqueNameGeneration](#aws-cdkcoreincludeprefixinuniquenamegeneration) | Include the stack prefix in the stack name generation process | 2.84.0 | (fix) |
| [@aws-cdk/aws-autoscaling:generateLaunchTemplateInsteadOfLaunchConfig](#aws-cdkaws-autoscalinggeneratelaunchtemplateinsteadoflaunchconfig) | Generate a launch template when creating an AutoScalingGroup | 2.88.0 | (fix) |
| [@aws-cdk/aws-opensearchservice:enableOpensearchMultiAzWithStandby](#aws-cdkaws-opensearchserviceenableopensearchmultiazwithstandby) | Enables support for Multi-AZ with Standby deployment for opensearch domains | 2.88.0 | (default) |
| [@aws-cdk/aws-efs:denyAnonymousAccess](#aws-cdkaws-efsdenyanonymousaccess) | EFS denies anonymous clients accesses | 2.93.0 | (default) |
| [@aws-cdk/aws-efs:mountTargetOrderInsensitiveLogicalId](#aws-cdkaws-efsmounttargetorderinsensitivelogicalid) | When enabled, mount targets will have a stable logicalId that is linked to the associated subnet. | 2.93.0 | (fix) |
| [@aws-cdk/aws-lambda-nodejs:useLatestRuntimeVersion](#aws-cdkaws-lambda-nodejsuselatestruntimeversion) | Enables aws-lambda-nodejs.Function to use the latest available NodeJs runtime as the default | 2.93.0 | (default) |
| [@aws-cdk/aws-appsync:useArnForSourceApiAssociationIdentifier](#aws-cdkaws-appsyncusearnforsourceapiassociationidentifier) | When enabled, will always use the arn for identifiers for CfnSourceApiAssociation in the GraphqlApi construct rather than id. | 2.97.0 | (fix) |
| [@aws-cdk/aws-rds:auroraClusterChangeScopeOfInstanceParameterGroupWithEachParameters](#aws-cdkaws-rdsauroraclusterchangescopeofinstanceparametergroupwitheachparameters) | When enabled, a scope of InstanceParameterGroup for AuroraClusterInstance with each parameters will change. | 2.97.0 | (fix) |
| [@aws-cdk/aws-rds:preventRenderingDeprecatedCredentials](#aws-cdkaws-rdspreventrenderingdeprecatedcredentials) | When enabled, creating an RDS database cluster from a snapshot will only render credentials for snapshot credentials. | 2.98.0 | (fix) |
| [@aws-cdk/aws-codepipeline-actions:useNewDefaultBranchForCodeCommitSource](#aws-cdkaws-codepipeline-actionsusenewdefaultbranchforcodecommitsource) | When enabled, the CodeCommit source action is using the default branch name 'main'. | 2.103.1 | (fix) |
| [@aws-cdk/aws-cloudwatch-actions:changeLambdaPermissionLogicalIdForLambdaAction](#aws-cdkaws-cloudwatch-actionschangelambdapermissionlogicalidforlambdaaction) | When enabled, the logical ID of a Lambda permission for a Lambda action includes an alarm ID. | 2.124.0 | (fix) |
| [@aws-cdk/aws-codepipeline:crossAccountKeysDefaultValueToFalse](#aws-cdkaws-codepipelinecrossaccountkeysdefaultvaluetofalse) | Enables Pipeline to set the default value for crossAccountKeys to false. | 2.127.0 | (default) |
| [@aws-cdk/aws-codepipeline:defaultPipelineTypeToV2](#aws-cdkaws-codepipelinedefaultpipelinetypetov2) | Enables Pipeline to set the default pipeline type to V2. | 2.133.0 | (default) |
| [@aws-cdk/aws-kms:reduceCrossAccountRegionPolicyScope](#aws-cdkaws-kmsreducecrossaccountregionpolicyscope) | When enabled, IAM Policy created from KMS key grant will reduce the resource scope to this key only. | 2.134.0 | (fix) |

<!-- END table -->

## Currently recommended cdk.json

The following json shows the current recommended set of flags, as `cdk init` would generate it for new projects.

<!-- BEGIN json -->
```json
{
  "context": {
    "@aws-cdk/aws-lambda:recognizeLayerVersion": true,
    "@aws-cdk/core:checkSecretUsage": true,
    "@aws-cdk/core:target-partitions": [
      "aws",
      "aws-cn"
    ],
    "@aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver": true,
    "@aws-cdk/aws-ec2:uniqueImdsv2TemplateName": true,
    "@aws-cdk/aws-ecs:arnFormatIncludesClusterName": true,
    "@aws-cdk/aws-iam:minimizePolicies": true,
    "@aws-cdk/core:validateSnapshotRemovalPolicy": true,
    "@aws-cdk/aws-codepipeline:crossAccountKeyAliasStackSafeResourceName": true,
    "@aws-cdk/aws-s3:createDefaultLoggingPolicy": true,
    "@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption": true,
    "@aws-cdk/aws-apigateway:disableCloudWatchRole": true,
    "@aws-cdk/core:enablePartitionLiterals": true,
    "@aws-cdk/aws-events:eventsTargetQueueSameAccount": true,
    "@aws-cdk/aws-iam:standardizedServicePrincipals": true,
    "@aws-cdk/aws-ecs:disableExplicitDeploymentControllerForCircuitBreaker": true,
    "@aws-cdk/aws-iam:importedRoleStackSafeDefaultPolicyName": true,
    "@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy": true,
    "@aws-cdk/aws-route53-patters:useCertificate": true,
    "@aws-cdk/customresources:installLatestAwsSdkDefault": false,
    "@aws-cdk/aws-rds:databaseProxyUniqueResourceName": true,
    "@aws-cdk/aws-codedeploy:removeAlarmsFromDeploymentGroup": true,
    "@aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId": true,
    "@aws-cdk/aws-ec2:launchTemplateDefaultUserData": true,
    "@aws-cdk/aws-secretsmanager:useAttachedSecretResourcePolicyForSecretTargetAttachments": true,
    "@aws-cdk/aws-redshift:columnId": true,
    "@aws-cdk/aws-stepfunctions-tasks:enableEmrServicePolicyV2": true,
    "@aws-cdk/aws-ec2:restrictDefaultSecurityGroup": true,
    "@aws-cdk/aws-apigateway:requestValidatorUniqueId": true,
    "@aws-cdk/aws-kms:aliasNameRef": true,
    "@aws-cdk/aws-autoscaling:generateLaunchTemplateInsteadOfLaunchConfig": true,
    "@aws-cdk/core:includePrefixInUniqueNameGeneration": true,
    "@aws-cdk/aws-efs:denyAnonymousAccess": true,
    "@aws-cdk/aws-opensearchservice:enableOpensearchMultiAzWithStandby": true,
    "@aws-cdk/aws-lambda-nodejs:useLatestRuntimeVersion": true,
    "@aws-cdk/aws-efs:mountTargetOrderInsensitiveLogicalId": true,
    "@aws-cdk/aws-rds:auroraClusterChangeScopeOfInstanceParameterGroupWithEachParameters": true,
    "@aws-cdk/aws-appsync:useArnForSourceApiAssociationIdentifier": true,
    "@aws-cdk/aws-rds:preventRenderingDeprecatedCredentials": true,
    "@aws-cdk/aws-codepipeline-actions:useNewDefaultBranchForCodeCommitSource": true,
    "@aws-cdk/aws-cloudwatch-actions:changeLambdaPermissionLogicalIdForLambdaAction": true,
    "@aws-cdk/aws-codepipeline:crossAccountKeysDefaultValueToFalse": true,
    "@aws-cdk/aws-codepipeline:defaultPipelineTypeToV2": true,
    "@aws-cdk/aws-kms:reduceCrossAccountRegionPolicyScope": true
  }
}
```
<!-- END json -->

## Flags removed in v2

These **default change** flags have been removed in v2. These used to be configurable in v1, but in v2 their
behavior has become the default. Remove these from your `cdk.json` file. If the old behavior is important
for your infrastructure, see the flag's description on how to achieve it.

<!-- BEGIN removed -->

| Flag | Summary | Type | Since |
| ----- | ----- | ----- | ----- |
| [@aws-cdk/core:enableStackNameDuplicates](#aws-cdkcoreenablestacknameduplicates) | Allow multiple stacks with the same name | (default) | 1.16.0 |
| [aws-cdk:enableDiffNoFail](#aws-cdkenablediffnofail) | Make `cdk diff` not fail when there are differences | (default) | 1.19.0 |
| [@aws-cdk/aws-ecr-assets:dockerIgnoreSupport](#aws-cdkaws-ecr-assetsdockerignoresupport) | DockerImageAsset properly supports `.dockerignore` files by default | (default) | 1.73.0 |
| [@aws-cdk/aws-secretsmanager:parseOwnedSecretName](#aws-cdkaws-secretsmanagerparseownedsecretname) | Fix the referencing of SecretsManager names from ARNs | (default) | 1.77.0 |
| [@aws-cdk/aws-kms:defaultKeyPolicies](#aws-cdkaws-kmsdefaultkeypolicies) | Tighten default KMS key policies | (default) | 1.78.0 |
| [@aws-cdk/aws-s3:grantWriteWithoutAcl](#aws-cdkaws-s3grantwritewithoutacl) | Remove `PutObjectAcl` from Bucket.grantWrite | (default) | 1.85.0 |
| [@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount](#aws-cdkaws-ecs-patternsremovedefaultdesiredcount) | Do not specify a default DesiredCount for ECS services | (default) | 1.92.0 |
| [@aws-cdk/aws-efs:defaultEncryptionAtRest](#aws-cdkaws-efsdefaultencryptionatrest) | Enable this feature flag to have elastic file systems encrypted at rest by default. | (default) | 1.98.0 |

<!-- END removed -->

## Flags with a different default in v2

These **fix/deprecation** flags are still configurable in v2, but their default has changed compared to v1. If you
are migrating a v1 CDK project to v2, explicitly set any of these flags which does not currently appear in your
`cdk.json` to `false`, to avoid unexpected infrastructure changes.

<!-- BEGIN diff -->

| Flag | Summary | Type | Since | v1 default | v2 default |
| ----- | ----- | ----- | ----- | ----- | ----- |
| [@aws-cdk/core:newStyleStackSynthesis](#aws-cdkcorenewstylestacksynthesis) | Switch to new stack synthesis method which enables CI/CD | (fix) | 1.39.0 | `false` | `true` |
| [@aws-cdk/core:stackRelativeExports](#aws-cdkcorestackrelativeexports) | Name exports based on the construct paths relative to the stack, rather than the global construct path | (fix) | 1.58.0 | `false` | `true` |
| [@aws-cdk/aws-rds:lowercaseDbIdentifier](#aws-cdkaws-rdslowercasedbidentifier) | Force lowercasing of RDS Cluster names in CDK | (fix) | 1.97.0 | `false` | `true` |
| [@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId](#aws-cdkaws-apigatewayusageplankeyorderinsensitiveid) | Allow adding/removing multiple UsagePlanKeys independently | (fix) | 1.98.0 | `false` | `true` |
| [@aws-cdk/aws-lambda:recognizeVersionProps](#aws-cdkaws-lambdarecognizeversionprops) | Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the  `fn.currentVersion`. | (fix) | 1.106.0 | `false` | `true` |
| [@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2\_2021](#aws-cdkaws-cloudfrontdefaultsecuritypolicytlsv12_2021) | Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default. | (fix) | 1.117.0 | `false` | `true` |

<!-- END diff -->

Here is an example of a `cdk.json` file that restores v1 behavior for these flags:

<!-- BEGIN migratejson -->
```json
{
  "context": {
    "@aws-cdk/core:newStyleStackSynthesis": false,
    "@aws-cdk/core:stackRelativeExports": false,
    "@aws-cdk/aws-rds:lowercaseDbIdentifier": false,
    "@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId": false,
    "@aws-cdk/aws-lambda:recognizeVersionProps": false,
    "@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021": false
  }
}
```
<!-- END migratejson -->
## Feature flag details

Here are more details about each of the flags:

<!-- BEGIN details -->
### @aws-cdk/core:enableStackNameDuplicates

*Allow multiple stacks with the same name* (default)

If this is set, multiple stacks can use the same stack name (e.g. deployed to
different environments). This means that the name of the synthesized template
file will be based on the construct path and not on the defined `stackName`
of the stack.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.16.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Pass stack identifiers to the CLI instead of stack names.


### aws-cdk:enableDiffNoFail

*Make `cdk diff` not fail when there are differences* (default)

Determines what status code `cdk diff` should return when the specified stack
differs from the deployed stack or the local CloudFormation template:

* `aws-cdk:enableDiffNoFail=true` => status code == 0
* `aws-cdk:enableDiffNoFail=false` => status code == 1

You can override this behavior with the --fail flag:

* `--fail` => status code == 1
* `--no-fail` => status code == 0


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.19.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Specify `--fail` to the CLI.


### @aws-cdk/aws-ecr-assets:dockerIgnoreSupport

*DockerImageAsset properly supports `.dockerignore` files by default* (default)

If this flag is not set, the default behavior for `DockerImageAsset` is to use
glob semantics for `.dockerignore` files. If this flag is set, the default behavior
is standard Docker ignore semantics.

This is a feature flag as the old behavior was technically incorrect but
users may have come to depend on it.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.73.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Update your `.dockerignore` file to match standard Docker ignore rules, if necessary.


### @aws-cdk/aws-secretsmanager:parseOwnedSecretName

*Fix the referencing of SecretsManager names from ARNs* (default)

Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
rather than the default full resource name, which includes the SecretsManager suffix.

If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.77.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Use `parseArn(secret.secretName).resourceName` to emulate the incorrect old parsing.


### @aws-cdk/aws-kms:defaultKeyPolicies

*Tighten default KMS key policies* (default)

KMS Keys start with a default key policy that grants the account access to administer the key,
mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
policy by specifying their own.

If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
resembles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
true, the policy matches what happens when this feature flag is set.

Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
to the key's default policy (rather than replacing it).


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.78.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Pass `trustAccountIdentities: false` to `Key` construct to restore the old behavior.


### @aws-cdk/aws-s3:grantWriteWithoutAcl

*Remove `PutObjectAcl` from Bucket.grantWrite* (default)

Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
as the former includes 's3:PutObjectAcl',
which could be used to grant read/write object access to IAM principals in other accounts.
Use a feature flag to make sure existing customers who might be relying
on the overly-broad permissions are not broken.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.85.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Call `bucket.grantPutAcl()` in addition to `bucket.grantWrite()` to grant ACL permissions.


### @aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount

*Do not specify a default DesiredCount for ECS services* (default)

ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
QueueProcessingServiceBase currently determine a default value for the desired count of
a CfnService if a desiredCount is not provided. The result of this is that on every
deployment, the service count is reset to the fixed value, even if it was autoscaled.

If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
desiredCount of 1, if one is not provided. If true, a default will not be defined for
CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.92.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** You can pass `desiredCount: 1` explicitly, but you should never need this.


### @aws-cdk/aws-efs:defaultEncryptionAtRest

*Enable this feature flag to have elastic file systems encrypted at rest by default.* (default)

Encryption can also be configured explicitly using the `encrypted` property.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.98.0 | `false` | `true` |
| (default in v2) | `true` |  |

**Compatibility with old behavior:** Pass the `encrypted: false` property to the `FileSystem` construct to disable encryption.


### @aws-cdk/core:newStyleStackSynthesis

*Switch to new stack synthesis method which enables CI/CD* (fix)

If this flag is specified, all `Stack`s will use the `DefaultStackSynthesizer` by
default. If it is not set, they will use the `LegacyStackSynthesizer`.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.39.0 | `false` | `true` |
| 2.0.0 | `true` | `true` |


### @aws-cdk/core:stackRelativeExports

*Name exports based on the construct paths relative to the stack, rather than the global construct path* (fix)

Combined with the stack name this relative construct path is good enough to
ensure uniqueness, and makes the export names robust against refactoring
the location of the stack in the construct tree (specifically, moving the Stack
into a Stage).


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.58.0 | `false` | `true` |
| 2.0.0 | `true` | `true` |


### @aws-cdk/aws-rds:lowercaseDbIdentifier

*Force lowercasing of RDS Cluster names in CDK* (fix)

Cluster names must be lowercase, and the service will lowercase the name when the cluster
is created. However, CDK did not use to know about this, and would use the user-provided name
referencing the cluster, which would fail if it happened to be mixed-case.

With this flag, lowercase the name in CDK so we can reference it properly.

Must be behind a permanent flag because changing a name from mixed case to lowercase between deployments
would lead CloudFormation to think the name was changed and would trigger a cluster replacement
(losing data!).


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.97.0 | `false` | `true` |
| 2.0.0 | `true` | `true` |


### @aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId

*Allow adding/removing multiple UsagePlanKeys independently* (fix)

The UsagePlanKey resource connects an ApiKey with a UsagePlan. API Gateway does not allow more than one UsagePlanKey
for any given UsagePlan and ApiKey combination. For this reason, CloudFormation cannot replace this resource without
either the UsagePlan or ApiKey changing.

The feature addition to support multiple UsagePlanKey resources - 142bd0e2 - recognized this and attempted to keep
existing UsagePlanKey logical ids unchanged.
However, this intentionally caused the logical id of the UsagePlanKey to be sensitive to order. That is, when
the 'first' UsagePlanKey resource is removed, the logical id of the 'second' assumes what was originally the 'first',
which again is disallowed.

In effect, there is no way to get out of this mess in a backwards compatible way, while supporting existing stacks.
This flag changes the logical id layout of UsagePlanKey to not be sensitive to order.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.98.0 | `false` | `true` |
| 2.0.0 | `true` | `true` |


### @aws-cdk/aws-lambda:recognizeVersionProps

*Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the  `fn.currentVersion`.* (fix)

The previous calculation incorrectly considered properties of the `AWS::Lambda::Function` resource that did
not constitute creating a new Version.

See 'currentVersion' section in the aws-lambda module's README for more details.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.106.0 | `false` | `true` |
| 2.0.0 | `true` | `true` |


### @aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021

*Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default.* (fix)

The security policy can also be configured explicitly using the `minimumProtocolVersion` property.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.117.0 | `false` | `true` |
| 2.0.0 | `true` | `true` |


### @aws-cdk/core:target-partitions

*What regions to include in lookup tables of environment agnostic stacks* (config)

Has no effect on stacks that have a defined region, but will limit the amount
of unnecessary regions included in stacks without a known region.

The type of this value should be a list of strings.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.137.0 | `false` | `["aws","aws-cn"]` |
| 2.4.0 | `false` | `["aws","aws-cn"]` |


### @aws-cdk-containers/ecs-service-extensions:enableDefaultLogDriver

*ECS extensions will automatically add an `awslogs` driver if no logging is specified* (default)

Enable this feature flag to configure default logging behavior for the ECS Service Extensions. This will enable the
`awslogs` log driver for the application container of the service to send the container logs to CloudWatch Logs.

This is a feature flag as the new behavior provides a better default experience for the users.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.140.0 | `false` | `true` |
| 2.8.0 | `false` | `true` |

**Compatibility with old behavior:** Specify a log driver explicitly.


### @aws-cdk/aws-ec2:uniqueImdsv2TemplateName

*Enable this feature flag to have Launch Templates generated by the `InstanceRequireImdsv2Aspect` use unique names.* (fix)

Previously, the generated Launch Template names were only unique within a stack because they were based only on the
`Instance` construct ID. If another stack that has an `Instance` with the same construct ID is deployed in the same
account and region, the deployments would always fail as the generated Launch Template names were the same.

The new implementation addresses this issue by generating the Launch Template name with the `Names.uniqueId` method.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.140.0 | `false` | `true` |
| 2.8.0 | `false` | `true` |


### @aws-cdk/aws-iam:minimizePolicies

*Minimize IAM policies by combining Statements* (config)

Minimize IAM policies by combining Principals, Actions and Resources of two
Statements in the policies, as long as it doesn't change the meaning of the
policy.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.150.0 | `false` | `true` |
| 2.18.0 | `false` | `true` |


### @aws-cdk/core:checkSecretUsage

*Enable this flag to make it impossible to accidentally use SecretValues in unsafe locations* (config)

With this flag enabled, `SecretValue` instances can only be passed to
constructs that accept `SecretValue`s; otherwise, `unsafeUnwrap()` must be
called to use it as a regular string.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.153.0 | `false` | `true` |
| 2.21.0 | `false` | `true` |


### @aws-cdk/aws-lambda:recognizeLayerVersion

*Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the `fn.currentVersion`.* (fix)

This flag correct incorporates Lambda Layer properties into the Lambda Function Version.

See 'currentVersion' section in the aws-lambda module's README for more details.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| 1.159.0 | `false` | `true` |
| 2.27.0 | `false` | `true` |


### @aws-cdk/core:validateSnapshotRemovalPolicy

*Error on snapshot removal policies on resources that do not support it.* (default)

Makes sure we do not allow snapshot removal policy on resources that do not support it.
If supplied on an unsupported resource, CloudFormation ignores the policy altogether.
This flag will reduce confusion and unexpected loss of data when erroneously supplying
the snapshot removal policy.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.28.0 | `false` | `true` |

**Compatibility with old behavior:** The old behavior was incorrect. Update your source to not specify SNAPSHOT policies on resources that do not support it.


### @aws-cdk/aws-codepipeline:crossAccountKeyAliasStackSafeResourceName

*Generate key aliases that include the stack name* (fix)

Enable this feature flag to have CodePipeline generate a unique cross account key alias name using the stack name.

Previously, when creating multiple pipelines with similar naming conventions and when crossAccountKeys is true,
the KMS key alias name created for these pipelines may be the same due to how the uniqueId is generated.

This new implementation creates a stack safe resource name for the alias using the stack name instead of the stack ID.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.29.0 | `false` | `true` |


### @aws-cdk/aws-s3:createDefaultLoggingPolicy

*Enable this feature flag to create an S3 bucket policy by default in cases where an AWS service would automatically create the Policy if one does not exist.* (fix)

For example, in order to send VPC flow logs to an S3 bucket, there is a specific Bucket Policy
that needs to be attached to the bucket. If you create the bucket without a policy and then add the
bucket as the flow log destination, the service will automatically create the bucket policy with the
necessary permissions. If you were to then try and add your own bucket policy CloudFormation will throw
and error indicating that a bucket policy already exists.

In cases where we know what the required policy is we can go ahead and create the policy so we can
remain in control of it.

@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html#AWS-logs-infrastructure-S3


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.31.0 | `false` | `true` |


### @aws-cdk/aws-sns-subscriptions:restrictSqsDescryption

*Restrict KMS key policy for encrypted Queues a bit more* (fix)

Enable this feature flag to restrict the decryption of a SQS queue, which is subscribed to a SNS topic, to
only the topic which it is subscribed to and not the whole SNS service of an account.

Previously the decryption was only restricted to the SNS service principal. To make the SQS subscription more
secure, it is a good practice to restrict the decryption further and only allow the connected SNS topic to decryption
the subscribed queue.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.32.0 | `false` | `true` |


### @aws-cdk/aws-ecs:arnFormatIncludesClusterName

*ARN format used by ECS. In the new ARN format, the cluster name is part of the resource ID.* (fix)

If this flag is not set, the old ARN format (without cluster name) for ECS is used.
If this flag is set, the new ARN format (with cluster name) for ECS is used.

This is a feature flag as the old format is still valid for existing ECS clusters.

See https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-account-settings.html#ecs-resource-ids


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.35.0 | `false` | `true` |


### @aws-cdk/aws-apigateway:disableCloudWatchRole

*Make default CloudWatch Role behavior safe for multiple API Gateways in one environment* (fix)

Enable this feature flag to change the default behavior for aws-apigateway.RestApi and aws-apigateway.SpecRestApi
to _not_ create a CloudWatch role and Account. There is only a single ApiGateway account per AWS
environment which means that each time you create a RestApi in your account the ApiGateway account
is overwritten. If at some point the newest RestApi is deleted, the ApiGateway Account and CloudWatch
role will also be deleted, breaking any existing ApiGateways that were depending on them.

When this flag is enabled you should either create the ApiGateway account and CloudWatch role
separately _or_ only enable the cloudWatchRole on a single RestApi.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.38.0 | `false` | `true` |


### @aws-cdk/core:enablePartitionLiterals

*Make ARNs concrete if AWS partition is known* (fix)

Enable this feature flag to get partition names as string literals in Stacks with known regions defined in
their environment, such as "aws" or "aws-cn".  Previously the CloudFormation intrinsic function
"Ref: AWS::Partition" was used.  For example:

```yaml
Principal:
  AWS:
    Fn::Join:
      - ""
      - - "arn:"
        - Ref: AWS::Partition
        - :iam::123456789876:root
```

becomes:

```yaml
Principal:
  AWS: "arn:aws:iam::123456789876:root"
```

The intrinsic function will still be used in Stacks where no region is defined or the region's partition
is unknown.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.38.0 | `false` | `true` |


### @aws-cdk/aws-ecs:disableExplicitDeploymentControllerForCircuitBreaker

*Avoid setting the "ECS" deployment controller when adding a circuit breaker* (fix)

Enable this feature flag to avoid setting the "ECS" deployment controller when adding a circuit breaker to an
ECS Service, as this will trigger a full replacement which fails to deploy when using set service names.
This does not change any behaviour as the default deployment controller when it is not defined is ECS.

This is a feature flag as the new behavior provides a better default experience for the users.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.51.0 | `false` | `true` |


### @aws-cdk/aws-events:eventsTargetQueueSameAccount

*Event Rules may only push to encrypted SQS queues in the same account* (fix)

This flag applies to SQS Queues that are used as the target of event Rules. When enabled, only principals
from the same account as the Rule can send messages. If a queue is unencrypted, this restriction will
always apply, regardless of the value of this flag.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.51.0 | `false` | `true` |


### @aws-cdk/aws-iam:standardizedServicePrincipals

*Use standardized (global) service principals everywhere* (fix)

We used to maintain a database of exceptions to Service Principal names in various regions. This database
is no longer necessary: all service principals names have been standardized to their global form (`SERVICE.amazonaws.com`).

This flag disables use of that exceptions database and always uses the global service principal.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.51.0 | `false` | `true` |


### @aws-cdk/aws-iam:importedRoleStackSafeDefaultPolicyName

*Enable this feature to by default create default policy names for imported roles that depend on the stack the role is in.* (fix)

Without this, importing the same role in multiple places could lead to the permissions given for one version of the imported role
to overwrite permissions given to the role at a different place where it was imported. This was due to all imported instances
of a role using the same default policy name.

This new implementation creates default policy names based on the constructs node path in their stack.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.60.0 | `false` | `true` |


### @aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy

*Use S3 Bucket Policy instead of ACLs for Server Access Logging* (fix)

Enable this feature flag to use S3 Bucket Policy for granting permission fo Server Access Logging
rather than using the canned `LogDeliveryWrite` ACL. ACLs do not work when Object Ownership is
enabled on the bucket.

This flag uses a Bucket Policy statement to allow Server Access Log delivery, following best
practices for S3.

@see https://docs.aws.amazon.com/AmazonS3/latest/userguide/enable-server-access-logging.html


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.60.0 | `false` | `true` |


### @aws-cdk/customresources:installLatestAwsSdkDefault

*Whether to install the latest SDK by default in AwsCustomResource* (default)

This was originally introduced and enabled by default to not be limited by the SDK version
that's installed on AWS Lambda. However, it creates issues for Lambdas bound to VPCs that
do not have internet access, or in environments where 'npmjs.com' is not available.

The recommended setting is to disable the default installation behavior, and pass the
flag on a resource-by-resource basis to enable it if necessary.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.60.0 | `false` | `false` |

**Compatibility with old behavior:** Set installLatestAwsSdk: true on all resources that need it.


### @aws-cdk/aws-route53-patters:useCertificate

*Use the official `Certificate` resource instead of `DnsValidatedCertificate`* (default)

Enable this feature flag to use the official CloudFormation supported `Certificate` resource instead
of the deprecated `DnsValidatedCertificate` construct. If this flag is enabled and you are creating
the stack in a region other than us-east-1 then you must also set `crossRegionReferences=true` on the
stack.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.61.0 | `false` | `true` |

**Compatibility with old behavior:** Define a `DnsValidatedCertificate` explicitly and pass in the `certificate` property


### @aws-cdk/aws-codedeploy:removeAlarmsFromDeploymentGroup

*Remove CloudWatch alarms from deployment group* (fix)

Enable this flag to be able to remove all CloudWatch alarms from a deployment group by removing
the alarms from the construct. If this flag is not set, removing all alarms from the construct
will still leave the alarms configured for the deployment group.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.65.0 | `false` | `true` |


### @aws-cdk/aws-rds:databaseProxyUniqueResourceName

*Use unique resource name for Database Proxy* (fix)

If this flag is not set, the default behavior for `DatabaseProxy` is
to use `id` of the constructor for `dbProxyName` when it's not specified in the argument.
In this case, users can't deploy `DatabaseProxy`s that have the same `id` in the same region.

If this flag is set, the default behavior is to use unique resource names for each `DatabaseProxy`.

This is a feature flag as the old behavior was technically incorrect, but users may have come to depend on it.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.65.0 | `false` | `true` |


### @aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId

*Include authorizer configuration in the calculation of the API deployment logical ID.* (fix)

The logical ID of the AWS::ApiGateway::Deployment resource is calculated by hashing
the API configuration, including methods, and resources, etc. Enable this feature flag
to also include the configuration of any authorizer attached to the API in the
calculation, so any changes made to an authorizer will create a new deployment.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.66.0 | `false` | `true` |


### @aws-cdk/aws-ec2:launchTemplateDefaultUserData

*Define user data for a launch template by default when a machine image is provided.* (fix)

The ec2.LaunchTemplate construct did not define user data when a machine image is
provided despite the document. If this is set, a user data is automatically defined
according to the OS of the machine image.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.67.0 | `false` | `true` |


### @aws-cdk/aws-secretsmanager:useAttachedSecretResourcePolicyForSecretTargetAttachments

*SecretTargetAttachments uses the ResourcePolicy of the attached Secret.* (fix)

Enable this feature flag to make SecretTargetAttachments use the ResourcePolicy of the attached Secret.
SecretTargetAttachments are created to connect a Secret to a target resource.
In CDK code, they behave like regular Secret and can be used as a stand-in in most situations.
Previously, adding to the ResourcePolicy of a SecretTargetAttachment did attempt to create a separate ResourcePolicy for the same Secret.
However Secrets can only have a single ResourcePolicy, causing the CloudFormation deployment to fail.

When enabling this feature flag for an existing Stack, ResourcePolicies created via a SecretTargetAttachment will need replacement.
This won't be possible without intervention due to limitation outlined above.
First remove all permissions granted to the Secret and deploy without the ResourcePolicies.
Then you can re-add the permissions and deploy again.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.67.0 | `false` | `true` |


### @aws-cdk/aws-redshift:columnId

*Whether to use an ID to track Redshift column changes* (fix)

Redshift columns are identified by their `name`. If a column is renamed, the old column
will be dropped and a new column will be created. This can cause data loss.

This flag enables the use of an `id` attribute for Redshift columns. If this flag is enabled, the
internal CDK architecture will track changes of Redshift columns through their `id`, rather
than their `name`. This will prevent data loss when columns are renamed.

**NOTE** - Enabling this flag comes at a **risk**. When enabled, update the `id`s of all columns,
**however** do not change the `names`s of the columns. If the `name`s of the columns are changed during
initial deployment, the columns will be dropped and recreated, causing data loss. After the initial deployment
of the `id`s, the `name`s of the columns can be changed without data loss.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.68.0 | `false` | `true` |


### @aws-cdk/aws-stepfunctions-tasks:enableEmrServicePolicyV2

*Enable AmazonEMRServicePolicy_v2 managed policies* (fix)

If this flag is not set, the default behavior for `EmrCreateCluster` is
to use `AmazonElasticMapReduceRole` managed policies.

If this flag is set, the default behavior is to use the new `AmazonEMRServicePolicy_v2`
managed policies.

This is a feature flag as the old behavior will be deprecated, but some resources may require manual
intervention since they might not have the appropriate tags propagated automatically.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.72.0 | `false` | `true` |


### @aws-cdk/aws-apigateway:requestValidatorUniqueId

*Generate a unique id for each RequestValidator added to a method* (fix)

This flag allows multiple RequestValidators to be added to a RestApi when
providing the `RequestValidatorOptions` in the `addMethod()` method.

If the flag is not set then only a single RequestValidator can be added in this way.
Any additional RequestValidators have to be created directly with `new RequestValidator`.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.78.0 | `false` | `true` |


### @aws-cdk/aws-ec2:restrictDefaultSecurityGroup

*Restrict access to the VPC default security group* (default)

Enable this feature flag to remove the default ingress/egress rules from the
VPC default security group.

When a VPC is created, a default security group is created as well and this cannot
be deleted. The default security group is created with ingress/egress rules that allow
_all_ traffic. [AWS Security best practices recommend](https://docs.aws.amazon.com/securityhub/latest/userguide/ec2-controls.html#ec2-2)
removing these ingress/egress rules in order to restrict access to the default security group.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.78.0 | `false` | `true` |

**Compatibility with old behavior:** 
      To allow all ingress/egress traffic to the VPC default security group you
      can set the `restrictDefaultSecurityGroup: false`.
    


### @aws-cdk/aws-kms:aliasNameRef

*KMS Alias name and keyArn will have implicit reference to KMS Key* (fix)

This flag allows an implicit dependency to be created between KMS Alias and KMS Key
when referencing key.aliasName or key.keyArn.

If the flag is not set then a raw string is passed as the Alias name and no
implicit dependencies will be set.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.83.0 | `false` | `true` |


### @aws-cdk/core:includePrefixInUniqueNameGeneration

*Include the stack prefix in the stack name generation process* (fix)

This flag prevents the prefix of a stack from making the stack's name longer than the 128 character limit.

If the flag is set, the prefix is included in the stack name generation process.
If the flag is not set, then the prefix of the stack is prepended to the generated stack name.

**NOTE** - Enabling this flag comes at a **risk**. If you have already deployed stacks, changing the status of this
feature flag can lead to a change in stacks' name. Changing a stack name mean recreating the whole stack, which
is not viable in some productive setups.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.84.0 | `false` | `true` |


### @aws-cdk/aws-autoscaling:generateLaunchTemplateInsteadOfLaunchConfig

*Generate a launch template when creating an AutoScalingGroup* (fix)

Enable this flag to allow AutoScalingGroups to generate a launch template when being created.
Launch configurations have been deprecated and cannot be created in AWS Accounts created after
December 31, 2023. Existing 'AutoScalingGroup' properties used for creating a launch configuration
will now create an equivalent 'launchTemplate'. Alternatively, users can provide an explicit
'launchTemplate' or 'mixedInstancesPolicy'. When this flag is enabled a 'launchTemplate' will
attempt to set user data according to the OS of the machine image if explicit user data is not
provided.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.88.0 | `false` | `true` |

**Compatibility with old behavior:** 
      If backwards compatibility needs to be maintained due to an existing autoscaling group
      using a launch config, set this flag to false.
    


### @aws-cdk/aws-opensearchservice:enableOpensearchMultiAzWithStandby

*Enables support for Multi-AZ with Standby deployment for opensearch domains* (default)

If this is set, an opensearch domain will automatically be created with
multi-az with standby enabled.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.88.0 | `false` | `true` |

**Compatibility with old behavior:** Pass `capacity.multiAzWithStandbyEnabled: false` to `Domain` construct to restore the old behavior.


### @aws-cdk/aws-efs:denyAnonymousAccess

*EFS denies anonymous clients accesses* (default)

This flag adds the file system policy that denies anonymous clients
access to `efs.FileSystem`.

If this flag is not set, `efs.FileSystem` will allow all anonymous clients
that can access over the network.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.93.0 | `false` | `true` |

**Compatibility with old behavior:** You can pass `allowAnonymousAccess: true` so allow anonymous clients access.


### @aws-cdk/aws-efs:mountTargetOrderInsensitiveLogicalId

*When enabled, mount targets will have a stable logicalId that is linked to the associated subnet.* (fix)

When this feature flag is enabled, each mount target will have a stable
logicalId that is linked to the associated subnet. If the flag is set to
false then the logicalIds of the mount targets can change if the number of
subnets changes.

Set this flag to false for existing mount targets.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.93.0 | `false` | `true` |


### @aws-cdk/aws-lambda-nodejs:useLatestRuntimeVersion

*Enables aws-lambda-nodejs.Function to use the latest available NodeJs runtime as the default* (default)

If this is set, and a `runtime` prop is not passed to, Lambda NodeJs
functions will us the latest version of the runtime provided by the Lambda
service. Do not use this if you your lambda function is reliant on dependencies
shipped as part of the runtime environment.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.93.0 | `false` | `true` |

**Compatibility with old behavior:** Pass `runtime: lambda.Runtime.NODEJS_16_X` to `Function` construct to restore the previous behavior.


### @aws-cdk/aws-appsync:useArnForSourceApiAssociationIdentifier

*When enabled, will always use the arn for identifiers for CfnSourceApiAssociation in the GraphqlApi construct rather than id.* (fix)

When this feature flag is enabled, we use the IGraphqlApi ARN rather than ID when creating or updating CfnSourceApiAssociation in 
the GraphqlApi construct. Using the ARN allows the association to support an association with a source api or merged api in another account.
Note that for existing source api associations created with this flag disabled, enabling the flag will lead to a resource replacement.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.97.0 | `false` | `true` |


### @aws-cdk/aws-rds:auroraClusterChangeScopeOfInstanceParameterGroupWithEachParameters

*When enabled, a scope of InstanceParameterGroup for AuroraClusterInstance with each parameters will change.* (fix)

When this feature flag is enabled, a scope of `InstanceParameterGroup` for
`AuroraClusterInstance` with each parameters will change to AuroraClusterInstance
from AuroraCluster.

If the flag is set to false then it can only make one `AuroraClusterInstance`
with each `InstanceParameterGroup` in the AuroraCluster.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.97.0 | `false` | `true` |


### @aws-cdk/aws-rds:preventRenderingDeprecatedCredentials

*When enabled, creating an RDS database cluster from a snapshot will only render credentials for snapshot credentials.* (fix)

The `credentials` property on the `DatabaseClusterFromSnapshotProps`
interface was deprecated with the new `snapshotCredentials` property being
recommended. Before deprecating `credentials`, a secret would be generated
while rendering credentials if the `credentials` property was undefined or
if a secret wasn't provided via the `credentials` property. This behavior
is replicated with the new `snapshotCredentials` property, but the original
`credentials` secret can still be created resulting in an extra database
secret.

Set this flag to prevent rendering deprecated `credentials` and creating an
extra database secret when only using `snapshotCredentials` to create an RDS
database cluster from a snapshot.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.98.0 | `false` | `true` |


### @aws-cdk/aws-codepipeline-actions:useNewDefaultBranchForCodeCommitSource

*When enabled, the CodeCommit source action is using the default branch name 'main'.* (fix)

When setting up a CodeCommit source action for the source stage of a pipeline, please note that the 
default branch is 'master'.
However, with the activation of this feature flag, the default branch is updated to 'main'.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.103.1 | `false` | `true` |


### @aws-cdk/aws-cloudwatch-actions:changeLambdaPermissionLogicalIdForLambdaAction

*When enabled, the logical ID of a Lambda permission for a Lambda action includes an alarm ID.* (fix)

When this feature flag is enabled, a logical ID of `LambdaPermission` for a
`LambdaAction` will include an alarm ID. Therefore multiple alarms for the same Lambda
can be created with `LambdaAction`.

If the flag is set to false then it can only make one alarm for the Lambda with
`LambdaAction`.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.124.0 | `false` | `true` |


### @aws-cdk/aws-codepipeline:crossAccountKeysDefaultValueToFalse

*Enables Pipeline to set the default value for crossAccountKeys to false.* (default)

When this feature flag is enabled, and the `crossAccountKeys` property is not provided in a `Pipeline`
construct, the construct automatically defaults the value of this property to false.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.127.0 | `false` | `true` |

**Compatibility with old behavior:** Pass `crossAccountKeys: true` to `Pipeline` construct to restore the previous behavior.


### @aws-cdk/aws-codepipeline:defaultPipelineTypeToV2

*Enables Pipeline to set the default pipeline type to V2.* (default)

When this feature flag is enabled, and the `pipelineType` property is not provided in a `Pipeline`
construct, the construct automatically defaults the value of this property to `PipelineType.V2`.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.133.0 | `false` | `true` |

**Compatibility with old behavior:** Pass `pipelineType: PipelineType.V1` to `Pipeline` construct to restore the previous behavior.


### @aws-cdk/aws-kms:reduceCrossAccountRegionPolicyScope

*When enabled, IAM Policy created from KMS key grant will reduce the resource scope to this key only.* (fix)

When this feature flag is enabled and calling KMS key grant method, the created IAM policy will reduce the resource scope from
'*' to this specific granting KMS key.


| Since | Default | Recommended |
| ----- | ----- | ----- |
| (not in v1) |  |  |
| 2.134.0 | `false` | `true` |


<!-- END details -->
