"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topologicalSort = void 0;
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter(e => e.dependencies.every(d => !remaining.has(d)));
        selectable.sort((a, b) => a.key < b.key ? -1 : b.key < a.key ? 1 : 0);
        for (const selected of selectable) {
            ret.push(selected.element);
            remaining.delete(selected.key);
        }
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(', ')}`);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,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